const Notification = require('../models/NotificationModel');
const { checkUpcomingEvents } = require('../services/notificationScheduler');

// Create a new notification
const create = async (req, res) => {
    try {
        const notification = await Notification.create(req.body);
        res.status(201).json({
            status: 201,
            message: 'Notification created successfully',
            notification
        });
    } catch (error) {
        res.status(500).json({
            status: 500,
            message: 'Error creating notification',
            error
        });
    }
};

// Get all notifications for logged in user
const getAll = async (req, res) => {
    try {
        // Check if user is authenticated
        if (!req.user || !req.user.id) {
            return res.status(401).json({
                status: 401,
                success: false,
                message: 'User not authenticated'
            });
        }

        const userId = req.user.id;
        console.log("User ID:", userId);
        const notifications = await Notification.findAll(userId);
        
        res.status(200).json({
            status: 200,
            success: true,
            message: 'Notifications fetched successfully',
            data: {
                total: notifications.length,
                notifications
            }
        });
    } catch (error) {
        console.error('Error in getAll notifications:', error);
        res.status(500).json({
            status: 500,
            success: false,
            message: 'Error fetching notifications',
            error: error.message
        });
    }
};  

// Get a single notification by ID
const getById = async (req, res) => {
    try {
        const notification = await Notification.findById(req.params.id);
        if (!notification) {
            return res.status(404).json({
                status: 404,
                message: 'Notification not found'
            });
        }
        res.status(200).json({
            status: 200,
            message: 'Notification fetched successfully',
            notification
        });
    } catch (error) {
        res.status(500).json({
            status: 500,
            message: 'Error fetching notification',
            error
        });
    }
};

// Update a notification by ID
const update = async (req, res) => {
    try {
        const notification = await Notification.findByIdAndUpdate(req.params.id, req.body, { new: true });
        if (!notification) {
            return res.status(404).json({
                status: 404,
                message: 'Notification not found'
            });
        }
        res.status(200).json({
            status: 200,
            message: 'Notification updated successfully',
            notification
        });
    } catch (error) {
        res.status(500).json({
            status: 500,
            message: 'Error updating notification',
            error
        });
    }
};

// Delete a notification by ID
const remove = async (req, res) => {
    try {
        const notification = await Notification.findByIdAndDelete(req.params.id);
        if (!notification) {            return res.status(404).json({
                status: 404,
                message: 'Notification not found'
            });
        }
        res.status(200).json({  
            status: 200,
            message: 'Notification deleted successfully',
            notification
        });
    } catch (error) {
        res.status(500).json({  
            status: 500,
            message: 'Error deleting notification',
            error
        });
    }
};

// Check for upcoming events and send notifications
const sendUpcomingNotifications = async (req, res) => {
    try {
        const notificationCount = await checkUpcomingEvents();
        
        res.status(200).json({
            status: 200,
            success: true,
            message: 'Upcoming notifications processed successfully',
            data: {
                notifications_sent: notificationCount
            }
        });
    } catch (error) {
        console.error('Error processing upcoming notifications:', error);
        res.status(500).json({
            status: 500,
            success: false,
            message: 'Error processing upcoming notifications',
            error: error.message
        });
    }
};

module.exports = {
    create,
    getAll,
    getById,
    update,
    remove,
    sendUpcomingNotifications
};



