const Contact = require('../models/contact');
const User = require('../models/user');
const multer = require('multer');
const csv = require('csv-parser');
const fs = require('fs');
const path = require('path');
const { v4: uuidv4 } = require('uuid');

// Configure multer for CSV file upload
const storage = multer.diskStorage({
    destination: function (req, file, cb) {
        const uploadDir = path.join(__dirname, '../uploads/temp');
        if (!fs.existsSync(uploadDir)) {
            fs.mkdirSync(uploadDir, { recursive: true });
        }
        cb(null, uploadDir);
    },
    filename: function (req, file, cb) {
        const uniqueFilename = `${uuidv4()}_${file.originalname}`;
        cb(null, uniqueFilename);
    }
});

const upload = multer({
    storage: storage,
    limits: {
        fileSize: 10 * 1024 * 1024 // 10MB limit
    },
    fileFilter: (req, file, cb) => {
        if (file.mimetype === 'text/csv' || file.originalname.endsWith('.csv')) {
            cb(null, true);
        } else {
            cb(new Error('Only CSV files are allowed'));
        }
    }
});


// Create new contact
const create = async (req, res) => {
    try {
        // console.log('Creating contact:', req.body);
        req.body.user_id = req.user.id;
        const contact_id = await Contact.create(req.body);
        if(contact_id?.status == 400){
            return res.status(400).json(contact_id);
        }
        const contact = await Contact.findById(contact_id);
        
        res.status(201).json({
            status: 201,
            message: 'Contact created successfully',
            contact
        });
    } catch (error) {
        // console.error('Error creating contact:', error);
        if (error.message.includes('already exists')) {
            return res.status(200).json({
                status: 409,
                message: 'Contact already exists!',
                error:  error.message
            });
        }
        res.status(500).json({
            status: 500,
            message: 'Error creating contact',
            error: error.message
        });
    }
};

// Get all contacts with search and filter
const getAll = async (req, res) => {
    try {
        const filters = {
            search: req.query.search,
            company_name: req.query.sales_accounts,
            email: req.query.email,
            phone: req.query.phone
        };

        // Get user's role and additional info
        const userId = req.user.id;
        const user = await User.findUserById(userId);
        const userRole = user.roles_id;

        // Add role-based filters
        if (userRole === 1) {
            // Admin (role_id = 1) can see all contacts
            // No additional filters needed
        } else if (userRole === 2) {
            // Territory Manager (role_id = 2) can see contacts based on territory_id
            filters.territory_id = user.territory_id;
        } else if (userRole === 3) {
            // Sales Person (role_id = 3) can see contacts based on owner_id
            filters.owner_id = userId;
        } else {
            // For any other role, restrict to owner_id
            filters.owner_id = userId;
        }

        const contacts = await Contact.findAll(filters);
        
        res.json({ 
            status: 200, 
            message: "Contacts fetched successfully", 
            contact_count: contacts.length,
            contacts,
            filters: Object.fromEntries(Object.entries(filters).filter(([_, v]) => v != null)),
            user_role: userRole
        });
    } catch (error) {
        console.error('Error fetching contacts:', error);
        res.status(500).json({ 
            status: 500, 
            message: 'Error fetching contacts', 
            error: error.message 
        });
    }
};


// Get contact by ID
const getById = async (req, res) => {
    try {
        const contact = await Contact.findById(req.params.id);
        if (!contact) {
            return res.status(404).json({ status : 404, message: 'Contact not found' });
        }
        res.json({ status: 200, message : "Contact fecthed Successfully", contact});
    } catch (error) {
        // console.error('Error fetching contact:', error);
        res.status(500).json({ status: 500, message: 'Error fetching contact' + error });
    }
};

// Update contact
const update = async (req, res) => {
    try {
        const success = await Contact.update(req.params.id, req.body);
        if (!success) {
            return res.status(404).json({ 
                status: 404, 
                message: 'Contact not found',
                id: req.params.id
             });
        }
        if(success.status == 409){
            return res.status(200).json(success);
        }
        res.json({ status: 200, message: 'Contact updated successfully',success });
    } catch (error) {
        // console.error('Error updating contact:', error);
        res.status(500).json({ status: 500, message: 'Error updating contact' + error });
    }
};

// Delete contact
const remove = async (req, res) => {
    try {
        const success = await Contact.deleteRecord(req.params.id);
        if (!success) {
            return res.status(404).json({ status: 404, message: 'Contact not found' });
        }
        res.json({ status: 200, message: 'Contact deleted successfully' });
    } catch (error) {
        // console.error('Error deleting contact:', error);
        res.status(500).json({ status: 500, message: 'Error deleting contact ' + error });
    }
};
async function getContactTasks(req, res) {
    try {
        const contactId = req.params.id;
        const tasks = await Contact.findByContactId(contactId);
        
        res.json(tasks);
    } catch (error) {
        console.error('Error in getContactTasks:', error);
        res.status(500).json({
            status: 500,
            message: error.message || 'Error fetching contact tasks'
        });
    }
}
async function getContactMeetings(req, res) {
    try {
        const contactId = req.params.id;
        const meetings = await Contact.findContactMeetings(contactId);
        
        res.json({
            status: true,
            message: "Contact meetings retrieved successfully",
            response: meetings
        });

    } catch (error) {
        console.error('Error in getContactMeetings:', error);
        res.status(500).json({
            status: false,
            message: error.message || 'Error retrieving contact meetings'
        });
    }
}

// Get contact relations
async function getContactRelations(req, res) {
    try {
        const contactId = req.params.id;
        const relations = await Contact.getContactRelations(contactId);

        if (!relations) {
            return res.status(404).json({
                status: 404,
                success: false,
                message: 'Contact not found '+contactId
            });
        }

        res.json({
            status: 200,
            success: true,
            message: 'Contact relations retrieved successfully '+contactId,
            data: relations
        });

    } catch (error) {
        console.error('Error in getContactRelations:', error);
        res.status(500).json({
            status: 500,
            success: false,
            message: error.message || 'Error retrieving contact relations'
        });
    }
}

// Lifecycle Controllers
async function getLifecycles(req, res) {
    try {
        const lifecycles = await Contact.getLifecycles();
        return res.status(200).json({
            status: true,
            message: 'Contact lifecycles fetched successfully',
            data: lifecycles
        });
    } catch (error) {
        console.error('Error fetching contact lifecycles:', error);
        return res.status(500).json({
            status: false,
            message: 'Error fetching contact lifecycles',
            error: error.message
        });
    }
}

async function createLifecycle(req, res) {
    try {
        const { name, description, active } = req.body;
        
        if (!name) {
            return res.status(400).json({
                status: false,
                message: 'Name is required'
            });
        }

        const id = await Contact.createLifecycle(req.body);
        return res.status(201).json({
            status: true,
            message: 'Contact lifecycle created successfully',
            data: { id }
        });
    } catch (error) {
        console.error('Error creating contact lifecycle:', error);
        return res.status(500).json({
            status: false,
            message: 'Error creating contact lifecycle',
            error: error.message
        });
    }
}

async function updateLifecycle(req, res) {
    try {
        const { id } = req.params;
        const { name, description, active } = req.body;

        if (!name) {
            return res.status(400).json({
                status: false,
                message: 'Name is required'
            });
        }

        const success = await Contact.updateLifecycle(id, req.body);
        if (!success) {
            return res.status(404).json({
                status: false,
                message: 'Contact lifecycle not found'
            });
        }

        return res.status(200).json({
            status: true,
            message: 'Contact lifecycle updated successfully'
        });
    } catch (error) {
        console.error('Error updating contact lifecycle:', error);
        return res.status(500).json({
            status: false,
            message: 'Error updating contact lifecycle',
            error: error.message
        });
    }
}

async function deleteLifecycle(req, res) {
    try {
        const { id } = req.params;
        const success = await Contact.deleteLifecycle(id);
        
        if (!success) {
            return res.status(404).json({
                status: false,
                message: 'Contact lifecycle not found'
            });
        }

        return res.status(200).json({
            status: true,
            message: 'Contact lifecycle deleted successfully'
        });
    } catch (error) {
        console.error('Error deleting contact lifecycle:', error);
        return res.status(500).json({
            status: false,
            message: 'Error deleting contact lifecycle',
            error: error.message
        });
    }
}

// Lifecycle Stages Controllers
async function getLifecycleStages(req, res) {
    try {
        const stages = await Contact.getLifecycleStages();
        return res.status(200).json({
            status: true,
            message: 'Contact lifecycle stages fetched successfully',
            data: stages
        });
    } catch (error) {
        console.error('Error fetching contact lifecycle stages:', error);
        return res.status(500).json({
            status: false,
            message: 'Error fetching contact lifecycle stages',
            error: error.message
        });
    }
}

async function createLifecycleStage(req, res) {
    try {
        const { status, lifecyclestage_id, probability, order } = req.body;
        
        // Validate required fields
        const requiredFields = ['status', 'lifecyclestage_id', 'probability', 'order'];
        const missingFields = requiredFields.filter(field => !req.body[field]);
        
        if (missingFields.length > 0) {
            return res.status(400).json({
                status: false,
                message: `Missing required fields: ${missingFields.join(', ')}`
            });
        }

        // Validate probability range (0-100)
        if (probability < 0 || probability > 100) {
            return res.status(400).json({
                status: false,
                message: 'Probability must be between 0 and 100'
            });
        }

        const id = await Contact.createLifecycleStage(req.body);
        return res.status(201).json({
            status: true,
            message: 'Contact lifecycle stage created successfully',
            data: { id }
        });
    } catch (error) {
        console.error('Error creating contact lifecycle stage:', error);
        return res.status(500).json({
            status: false,
            message: 'Error creating contact lifecycle stage',
            error: error.message
        });
    }
}

async function updateLifecycleStages(req, res) {
    try {
        const inputData = req.body;
        console.log('Input data:', inputData);

        // Validate that request body is an array
        if (!Array.isArray(inputData)) {
            return res.status(400).json({
                status: false,
                message: 'Request body must be an array of lifecycle configurations'
            });
        }

        // Validate each lifecycle configuration
        for (const config of inputData) {
            // Validate lifecycle object
            if (!config.lifecycle || !config.lifecycle.stage) {
                return res.status(400).json({
                    status: false,
                    message: 'Each configuration must have a lifecycle object with stage name'
                });
            }

            // Validate stages array
            if (!Array.isArray(config.stages) || config.stages.length === 0) {
                return res.status(400).json({
                    status: false,
                    message: 'Each lifecycle must have a non-empty stages array'
                });
            }

            // Validate each stage
            for (const stage of config.stages) {
                // Validate required fields
                const requiredFields = ['status', 'order'];
                const missingFields = requiredFields.filter(field => !stage[field]);
                
                if (missingFields.length > 0) {
                    return res.status(400).json({
                        status: false,
                        message: `Missing required fields in stage: ${missingFields.join(', ')}`
                    });
                }

                // Validate order is positive
                if (stage.order < 1) {
                    return res.status(400).json({
                        status: false,
                        message: `Invalid order value in stage: ${stage.status}. Must be a positive number`
                    });
                }
            }
        }

        const results = await Contact.updateLifecycleStages(inputData);

        return res.status(200).json({
            status: true,
            message: 'Lifecycle stages updated successfully',
            data: results
        });

    } catch (error) {
        console.error('Error updating lifecycle stages:', error);
        
        // Handle specific errors
        if (error.message.includes('already in use') || 
            error.message.includes('Duplicate order') ||
            error.message.includes('Invalid stage order') ||
            error.message.includes('Invalid stage sequence')) {
            return res.status(409).json({
                status: false,
                message: error.message
            });
        }

        return res.status(500).json({
            status: false,
            message: 'Error updating lifecycle stages',
            error: error.message
        });
    }
}

// get all lifecycles by id
async function getLifecyclesById(req, res) {
    try {
        const lifecycles = await Contact.getLifecyclesById(req.params.id);
        return res.status(200).json({   
            status: 200,
            success: true,
            message: 'Lifecycles retrieved successfully',
            data: lifecycles
        });
    } catch (error) {
        console.error('Error fetching lifecycles:', error); 
        return res.status(500).json({
            status: 500,
            success: false,
            message: 'Error fetching lifecycles',
            error: error.message
        });
    }
}

async function deleteLifecycleStage(req, res) {
    try {
        const { id } = req.params;
        const success = await Contact.deleteLifecycleStage(id);
        
        if (!success) {
            return res.status(404).json({
                status: 404,
                message: 'Contact lifecycle stage not found'
            });
        }

        return res.status(200).json({
            status: true,
            message: 'Contact lifecycle stage deleted successfully'
        });
    } catch (error) {
        console.error('Error deleting contact lifecycle stage:', error);
        return res.status(500).json({
            status: 500,
            message: 'Error deleting contact lifecycle stage',
            error: error.message
        });
    }
}

async function createFilterView(req, res) {
    console.log("Create Contact Filter View Controller");
    try {
        // Validate required fields
        if (!req.body.title || !req.body.share_with) {
            return res.status(400).json({
                status: false,
                message: 'Title and share with field are required'
            });
        }

        const view = await Contact.createFilterView(req.body);

        res.json({
            status: true,
            message: 'Contact View saved successfully',
            response: view
        });

    } catch (error) {
        console.error('Error in Contact:', error);
        res.status(500).json({
            status: false,
            message: error.message || 'Contact View not saved',
            response: null
        });
    }
}

// Get ALl Filters List
const getAllFiltersList = async (req, res) => {
    console.log("=== Starting getAllFiltersList Controller ===");
    try {
        console.log("Calling Contact.getAllFiltersList()");
        const filters = await Contact.getAllFiltersList();
        console.log("Filters received from model:", filters);
        
        if (!filters || filters.length === 0) {
            console.log("No filters found in the database");
            return res.status(404).json({ 
                status: 404, 
                message: 'No filters found' 
            });
        }
        
        console.log("Successfully retrieved filters, sending response");
        res.json({
            status: 200, 
            message: 'Filters retrieved successfully', 
             filters
        });
    } catch (error) {
        console.error("=== Error in getAllFiltersList Controller ===");
        console.error("Error details:", error);
        res.status(500).json({ 
            status: 500, 
            message: 'Error fetching filters: ' + error.message 
        });
    }
};


// Delete Contact Filter View
async function deleteFilterView(req, res) {
    try {
        const viewId = req.params.id;
        const success = await Contact.deleteFilterView(viewId);
        if (!success) {
            return res.status(404).json({
                status: 404,
                success: false,
                message: 'Contact Filter View not found'
            }); 
        }
        res.json({
            status: 200,
            success: true,
            message: 'Contact Filter View deleted successfully'
        }); 
    } catch (error) {
        console.error('Error in deleteFilterView:', error);
        res.status(500).json({
            status: 500,
            success: false,
            message: error.message || 'Error deleting contact filter view'
        });
    }
}

async function getContactListWithFilters(req, res) {
    try {
        const { filter } = req.body;
        const userId = req.user.id;
        const role = await User.findUserById(userId);
        // console.log("Role:", role);
        const role_id = role.roles_id;
        // console.log("Role ID:", role_id);

        // Get filtered products
        const contacts = await Contact.getContactListWithFilters(filter, userId, role_id);

        res.json({
            status: 200,
            message: 'Contacts filtered successfully',
            contacts: contacts
        });

    } catch (error) {
        console.error('Error in getContactListWithFilters:', error);
        res.status(500).json({
            status: false,
            message: error.message || 'Error filtering contacts'
        });
    }
}

async function updateFilterView(req, res) {
    console.log("Update Contact Filter View Controller");
    try {
        // Validate required fields
        if (!req.body.title || !req.body.share_with) {
            return res.status(400).json({
                status: false,
                message: 'Title and share with field are required'
            });
        }

        const view = await Contact.updateFilterView(req.body);

        res.json({
            status: true,
            message: 'Contact View updated successfully',
            response: view
        });

    } catch (error) {
        console.error('Error in Contact:', error);
        res.status(500).json({
            status: false,
            message: error.message || 'Contact View not saved',
            response: null
        });
    }
}

async function removeRelatedModule(req, res) {
    try {
        console.log("Remove Contact Filter View Controller");
        console.log("Request body:", req.body);
        const { id, module, related, related_id } = req.body;

        // Validate required fields
        if (!id) {
            return res.status(400).json({ status: 400, success: false, message: 'ID is required' });
        }

        if (!module) {
            return res.status(400).json({ status: 400, success: false, message: 'Module name is required' });
        }

        if (!related) {
            return res.status(400).json({ status: 400, success: false, message: 'Related field is required' });
        }

        if (!related_id) {
            return res.status(400).json({ status: 400, success: false, message: 'Related ID is required' });
        }

        // Validate module type
        const validModules = ['deals', 'contacts', 'accounts'];
        const moduleKey = module.toLowerCase();

        if (!validModules.includes(moduleKey)) {
            return res.status(400).json({
                status: 400,
                success: false,
                message: 'Invalid module type. Supported modules: deals, contacts, accounts'
            });
        }

        // Validate related field based on module
        const validRelations = {
            deals: ['contact', 'account'],
            contacts: ['account'],
            accounts: ['contact', 'deal']
        };

        if (!validRelations[moduleKey] || !validRelations[moduleKey].includes(related.toLowerCase())) {
            return res.status(400).json({
                status: 400,
                success: false,
                message: `Invalid related field for ${module}. Supported: ${validRelations[moduleKey].join(', ')}`
            });
        }

        // Remove the relation
        const result = await Contact.removerelated(id, module, related, related_id);
        res.json({
            status: 200,
            success: true,
            message: result.message,
            data: {
                module_type: moduleKey,
                module_id: id,
                related_field: related.toLowerCase(),
                removed: result.success,
                affected_rows: result.affectedRows
            }
        });

    } catch (error) {
        console.error('Error removing related module:', error);
        res.status(500).json({
            status: 500,
            success: false,
            message: 'Error removing related module',
            error: error.message
        });
    }
}

// Bulk import contacts
async function bulkImport(req, res) {
    try {
        // Validate required fields
        if (!req.file) {
            return res.status(400).json({
                status: 400,
                success: false,
                message: 'CSV file is required'
            });
        }

        if (!req.body.columnMappings) {
            return res.status(400).json({
                status: 400,
                success: false,
                message: 'Column mappings are required'
            });
        }

        

        let columnMappings;
        try {
            columnMappings = JSON.parse(req.body.columnMappings);
        } catch (error) {
            return res.status(400).json({
                status: 400,
                success: false,
                message: 'Invalid column mappings format'
            });
        }

        if (!Array.isArray(columnMappings)) {
            return res.status(400).json({
                status: 400,
                success: false,
                message: 'Column mappings must be an array'
            });
        }

        // Validate column mappings
        for (const mapping of columnMappings) {
            if (!mapping.csvColumn || !mapping.apiField) {
                return res.status(400).json({
                    status: 400,
                    success: false,
                    message: 'Each mapping must have csvColumn and apiField'
                });
            }
        }

        // Parse CSV file
        const csvData = [];
        const filePath = req.file.path;

        return new Promise((resolve, reject) => {
            fs.createReadStream(filePath)
                .pipe(csv())
                .on('data', (row) => {
                    csvData.push(row);
                })
                .on('end', async () => {
                    try {
                        // Clean up the uploaded file
                        fs.unlinkSync(filePath);

                        if (csvData.length === 0) {
                            return res.status(400).json({
                                status: 400,
                                success: false,
                                message: 'CSV file is empty or has no valid data'
                            });
                        }

                        // Process the import
                        const userId = req.user.id;
                        const results = await Contact.bulkImport(csvData, columnMappings, userId);

                        // if (results.errors.lenngth > 0) {
                        //     return res.status(400).json({
                        //         status: 400,
                        //         success: false,
                                
                        //         message: 'Column mappings are required'
                        //     });
                        // }

                        res.json({
                            status: 200,
                            success: true,
                            message: 'Bulk import completed',
                            data: {
                                total: results.total,
                                successful: results.successful,
                                failed: results.failed,
                                errors: results.errors.slice(0, 10) 
                            },
                            errors: results.failed,
                            file: results.errorCsvFile
                        });

                        resolve();
                    } catch (error) {
                        // Clean up the uploaded file in case of error
                        if (fs.existsSync(filePath)) {
                            fs.unlinkSync(filePath);
                        }
                        reject(error);
                    }
                })
                .on('error', (error) => {
                    // Clean up the uploaded file in case of error
                    if (fs.existsSync(filePath)) {
                        fs.unlinkSync(filePath);
                    }
                    reject(error);
                });
        });

    } catch (error) {
        console.error('Error in bulk import:', error);
        res.status(500).json({
            status: 500,
            success: false,
            message: 'Error processing bulk import',
            error: error.message
        });
    }
}

module.exports = {
    create,
    getAll,
    getById,
    update,
    remove,
    getContactTasks,
    getContactMeetings,
    getContactRelations,
    getLifecycles,
    createLifecycle,
    updateLifecycle,
    deleteLifecycle,
    getLifecycleStages,
    createLifecycleStage,
    updateLifecycleStages,
    deleteLifecycleStage,
    getAllFiltersList,
    createFilterView,
    deleteFilterView,
    getLifecyclesById,
    getContactListWithFilters,
    updateFilterView,
    removeRelatedModule,
    bulkImport,
    upload
};