const conveyanceModel = require('../models/conveyance');

/**
 * Get conveyance data based on filter
 * @param {Object} req - Express request object
 * @param {Object} res - Express response object
 * @returns {Object} JSON response
 */
async function getConveyance(req, res) {
    try {
        // Validate required fields
        if (!req.body.filter) {
            return res.status(400).json({
                status: 400,
                success: false,
                message: 'Filter field is required',
                data: []
            });
        }

        // Validate Monthly filter data
        if (req.body.filter === "Monthly" && (!req.body.month || !req.body.year)) {
            return res.status(400).json({
                status: 400,
                success: false,
                message: 'Month and Year fields are required for Monthly filter',
                data: []
            });
        }

        // Get filter data
        const filterData = {
            filter: req.body.filter,
            month: req.body.month,
            year: req.body.year
        };
        console.log(filterData);
        console.log(req.body);
        console.log(req.user);
        // Get user ID from authenticated user
        const userId = req.user.id;
        console.log(`Processing ${filterData.filter} conveyance data for user ID: ${userId}`);
        console.log(filterData);

        // Get conveyance data
        const result = await conveyanceModel.getConveyance(filterData, userId);

        // For Monthly filter, provide structured response even if no records
        if (filterData.filter === "Monthly") {
            // Get data or set defaults
            const responseData = result.status ? result.data : { 
                conveyance: [], 
                claimapply: false 
            };
            
            // Prepare data
            const monthlyTotal = responseData.conveyance && responseData.conveyance.length > 0 
                ? responseData.conveyance.find(item => item.is_total) 
                : null;
                
            const dailyData = responseData.conveyance && responseData.conveyance.length > 0 
                ? responseData.conveyance.filter(item => !item.is_total)
                : [];
            
            // Format month string (ensure integers)
            const numMonth = parseInt(filterData.month, 10);
            const numYear = parseInt(filterData.year, 10);
            const monthStr = `${numYear}-${numMonth.toString().padStart(2, '0')}`;
            
            // Always return a proper monthly structure
            return res.status(200).json({
                status: 200,
                success: true,
                message: "Conveyance info",
                response: {
                    month: monthStr,
                    total: {
                        distance: monthlyTotal ? monthlyTotal.distance : 0,
                        amount: monthlyTotal ? monthlyTotal.amount : 0,
                        timespent: monthlyTotal ? monthlyTotal.timespent : 0
                    },
                    conveyance: dailyData.map(day => ({
                        conveyance_date: day.conveyance_date,
                        distance: day.distance,
                        amount: day.amount,
                        timespent: day.timespent
                    })),
                    claimapply: responseData.claimapply || false
                }
            });
        }

        // Continue with standard response handling for other filters
        if (result.status) {
            // Format the response based on the filter type
            const responseData = result.data;
            
            // Standard response for other filters
            return res.status(200).json({
                status: 200,
                success: true,
                message: result.message,
                response: responseData
            });
        } else {
            return res.status(200).json({
                status: 204,
                success: false,
                message: result.message,
                response: []
            });
        }
    } catch (error) {
        console.error('Error in getConveyance controller:', error);
        return res.status(200).json({
            status: 500,
            success: false,
            message: `Error getting conveyance data: ${error.message}`,
            response: []
        });
    }
}

/**
 * View detailed conveyance data for a specific date
 * @param {Object} req - Express request object
 * @param {Object} res - Express response object
 * @returns {Object} JSON response
 */
async function viewConveyance(req, res) {
    try {
        // Validate required fields
        if (!req.body.date) {
            return res.status(400).json({
                status: 400,
                success: false,
                message: 'Date field is required',
                response: null
            });
        }

        // Get date and user ID
        const date = req.body.date;
        const userId = req.user.id;

        // Get conveyance data with transport modes
        const result = await conveyanceModel.getConveyanceByDate(date, userId);

        if (result.status) {
            return res.status(200).json({
                status: 200,
                success: true,
                message: result.message,
                response: {
                    date: result.data.date,
                    activities: result.data.activities,
                    transport_modes: result.data.transport_modes,
                    summary: result.data.summary
                }
            });
        } else {
            return res.status(200).json({
                status: 204,
                success: false,
                message: result.message,
                response: {
                    date: date,
                    activities: [],
                    transport_modes: [],
                    summary: {
                        total_distance: "0.00",
                        total_amount: "0.00"
                    }
                }
            });
        }

    } catch (error) {
        console.error('Error in viewConveyance:', error);
        return res.status(500).json({
            status: 500,
            success: false,
            message: 'Internal server error',
            response: {
                date: req.body.date,
                activities: [],
                transport_modes: [],
                summary: {
                    total_distance: "0.00",
                    total_amount: "0.00"
                }
            }
        });
    }
}async function claimapply(req, res) {
    try {
        const startDate = req.body.startDate;
        const endDate = req.body.endDate;
        const userId = req.user.id;
        const result = await conveyanceModel.claimapply(startDate, endDate, userId);
        return res.status(200).json({
            status: 200,
            success: true,
            message: "Claim applied",
            response: result
        });
    } catch (error) {
        console.error('Error in claimapply controller:', error);
        return res.status(500).json({
            status: 500,
            success: false,
            message: `Claim apply - ${error.message}`,
            response: []
        });
    }
}

async function claimhistory(req, res) {     

    try {

        const userId = req.user.id;
        const startDate = req.query.startDate;
        const endDate = req.query.endDate;
        // console.log(startDate);
        // console.log(endDate);
        if (startDate && endDate) { 
            const result = await conveyanceModel.claimhistory(userId, startDate, endDate);
            return res.status(200).json({
                status: 200,
                success: true,
                message: "Claim history by date",
                total_claims: result.length,
                date: {
                    startDate: startDate,
                    endDate: endDate
                },
                response: result
            });
        } else {
            const result = await conveyanceModel.claimhistory(userId);
            return res.status(200).json({
                status: 200,
                success: true,
                message: "Claim history",
                response: result
            });
        }
    
    } catch (error) {   
        console.error('Error in claimhistory controller:', error);
        return res.status(500).json({
            status: 500,
            success: false,
            message: `Claim history - ${error.message}`,
            response: []
        });
    }
}
async function claimhistoryByStatus(req, res) {     
    try {
        const userId = req.user.id;
        const startDate = req.query.startDate;
        const endDate = req.query.endDate;
        console.log(startDate, "hi", endDate)
        let result;
        if (startDate && endDate) { 
            result = await conveyanceModel.claimhistoryByStatus(userId, startDate, endDate);
            
            // Group claims by status
            const pending = result.filter(claim => 
                // console.log("pending", claim.status)
                ['Applied', 'Manager Approved', 'Reapplied'].includes(claim.status)
            );
            // console.log("pending", result.status)
            
            const rejected = result.filter(claim => 
                ['Rejected', 'Finance Rejected', 'Manager Rejected'].includes(claim.status)
            );
            
            const completed = result.filter(claim => 
                ['Approved', 'Completed' , 'Finance Approved'].includes(claim.status)
            );

            return res.status(200).json({
                status: 200,
                success: true,
                message: "Claim history by date",
                date: {
                    startDate: startDate,
                    endDate: endDate
                },
                response: {
                    pending,
                    rejected,
                    completed,
                    summary: {
                        total_claims: result.length,
                        total_pending: pending.length,
                        total_rejected: rejected.length,
                        total_completed: completed.length,
                        total_amount: {
                            pending: pending
                                .reduce((sum, claim) => sum + parseFloat(claim.claim_amount || 0), 0),
                            rejected: rejected
                                .reduce((sum, claim) => sum + parseFloat(claim.claim_amount || 0), 0),
                            completed: completed
                                .reduce((sum, claim) => sum + parseFloat(claim.claim_amount || 0), 0)
                        }
                    }
                }
            });
        } else {
            result = await conveyanceModel.claimhistoryByStatus(userId);
            
            // Group claims by status
            const pending = result.filter(claim => 
                ['Applied', 'Reapplied'].includes(claim.status)
            );
            
            const rejected = result.filter(claim => 
                ['Rejected', 'Finance Rejected'].includes(claim.status)
            );
            
            const completed = result.filter(claim => 
                ['Approved', 'Finance Approved'].includes(claim.status)
            );

            return res.status(200).json({
                status: 200,
                success: true,
                message: "Claim history",
                response: {
                    pending,
                    rejected,
                    completed,
                    summary: {
                        total_claims: result.length,
                        total_pending: pending.length,
                        total_rejected: rejected.length,
                        total_completed: completed.length,
                        total_amount: {
                            pending: pending
                                .reduce((sum, claim) => sum + parseFloat(claim.claim_amount || 0), 0),
                            rejected: rejected
                                .reduce((sum, claim) => sum + parseFloat(claim.claim_amount || 0), 0),
                            completed: completed
                                .reduce((sum, claim) => sum + parseFloat(claim.claim_amount || 0), 0)
                        }
                    }
                }
            });
        }
    } catch (error) {   
        console.error('Error in claimhistoryByStatus controller:', error);
        return res.status(500).json({
            status: 500,
            success: false,
            message: `Claim history - ${error.message}`,
            response: {
                pending: [],
                rejected: [],
                completed: [],
                summary: {
                    total_claims: 0,
                    total_pending: 0,
                    total_rejected: 0,
                    total_completed: 0,
                    total_amount: {
                        pending: 0,
                        rejected: 0,
                        completed: 0
                    }
                }
            }
        });
    }
}
const editClaim = async (req, res) => {
    try {
        const { id } = req.params;
        const { transport_mode, claim_image } = req.body;
        const amount = req.body.claim_amount;
        const files = req.files; // Multiple files for edit
        console.log("editClaim", files, claim_image);

        // Validate inputs
        if (!transport_mode) {
            return res.status(400).json({
                status: 400,
                success: false,
                message: 'Transport mode is required'
            });
        }

        // Prepare data - handle both uploaded files and comma-separated string
        let claimImages = [];
        
        // If files are uploaded, use them
        if (files && files.length > 0) {
            claimImages = files.map(file => `/uploads/claims/${file.filename}`);
        }
        // If claim_image is provided as comma-separated string, use it
        else if (claim_image) {
            claimImages = claim_image.split(',').map(img => img.trim()).filter(img => img);
        }

        const data = {
            transport_mode,
            amount,
            claim_images: claimImages
        };

        const result = await conveyanceModel.editClaimById(id, data, req.user.id);
        res.json(result);

    } catch (error) {
        console.error('Error in editClaim:', error);
        res.status(500).json({
            status: 500,
            success: false,
            message: error.message || 'Failed to update claim'
        });
    }
};

const reapplyClaim = async (req, res) => {
    try {
        const { id } = req.params;
        const { transport_mode, amount, remarks, claim_image } = req.body;
        const files = req.files; // Multiple files for reapply

        // Validate inputs
        if (!transport_mode) {
            return res.status(400).json({
                status: 400,
                success: false,
                message: 'Transport mode is required'
            });
        }

        // Prepare data - handle both uploaded files and comma-separated string
        let claimImages = [];
        
        // If files are uploaded, use them
        if (files && files.length > 0) {
            claimImages = files.map(file => `/uploads/claims/${file.filename}`);
        }
        // If claim_image is provided as comma-separated string, use it
        else if (claim_image) {
            claimImages = claim_image.split(',').map(img => img.trim()).filter(img => img);
        }

        const data = {
            transport_mode,
            amount,
            remarks,
            claim_images: claimImages
        };

        const result = await conveyanceModel.reapplyClaim(id, data, req.user.id);
        res.json(
            {
                status: 200,
                success: true,
                message: "Claim reapply",
                response: result.data
            }
            );

    } catch (error) {
        console.error('Error in reapplyClaim:', error);
        res.status(500).json({
            status: 500,
            success: false,
            message: error.message || 'Failed to update claim'
        });
    }
};

module.exports = {
    getConveyance,
    viewConveyance,
    claimapply,
    claimhistory,
    editClaim,
    claimhistoryByStatus,
    reapplyClaim
};