const conveyanceModel = require('../models/conveyance');

/**
 * Get conveyance data based on filter
 * @param {Object} req - Express request object
 * @param {Object} res - Express response object
 * @returns {Object} JSON response
 */
async function getConveyance(req, res) {
    try {
        // Validate required fields
        if (!req.body.filter) {
            return res.status(400).json({
                status: 400,
                success: false,
                message: 'Filter field is required',
                data: []
            });
        }

        // Validate Monthly filter data
        if (req.body.filter === "Monthly" && (!req.body.month || !req.body.year)) {
            return res.status(400).json({
                status: 400,
                success: false,
                message: 'Month and Year fields are required for Monthly filter',
                data: []
            });
        }

        // Get filter data
        const filterData = {
            filter: req.body.filter,
            month: req.body.month,
            year: req.body.year
        };
        console.log(filterData);
        console.log(req.body);
        console.log(req.user);
        // Get user ID from authenticated user
        const userId = req.user.id;
        console.log(`Processing ${filterData.filter} conveyance data for user ID: ${userId}`);
        console.log(filterData);

        // Get conveyance data
        const result = await conveyanceModel.getConveyance(filterData, userId);

        // For Monthly filter, provide structured response even if no records
        if (filterData.filter === "Monthly") {
            // Get data or set defaults
            const responseData = result.status ? result.data : { 
                conveyance: [], 
                claimapply: false 
            };
            
            // Prepare data
            const monthlyTotal = responseData.conveyance && responseData.conveyance.length > 0 
                ? responseData.conveyance.find(item => item.is_total) 
                : null;
                
            const dailyData = responseData.conveyance && responseData.conveyance.length > 0 
                ? responseData.conveyance.filter(item => !item.is_total)
                : [];
            
            // Format month string (ensure integers)
            const numMonth = parseInt(filterData.month, 10);
            const numYear = parseInt(filterData.year, 10);
            const monthStr = `${numYear}-${numMonth.toString().padStart(2, '0')}`;
            
            // Always return a proper monthly structure
            return res.status(200).json({
                status: 200,
                success: true,
                message: "Conveyance info",
                response: {
                    month: monthStr,
                    total: {
                        distance: monthlyTotal ? monthlyTotal.distance : 0,
                        amount: monthlyTotal ? monthlyTotal.amount : 0,
                        timespent: monthlyTotal ? monthlyTotal.timespent : 0
                    },
                    conveyance: dailyData.map(day => ({
                        conveyance_date: day.conveyance_date,
                        distance: day.distance,
                        amount: day.amount,
                        timespent: day.timespent
                    })),
                    claimapply: responseData.claimapply || false
                }
            });
        }

        // Continue with standard response handling for other filters
        if (result.status) {
            // Format the response based on the filter type
            const responseData = result.data;
            
            // Standard response for other filters
            return res.status(200).json({
                status: 200,
                success: true,
                message: result.message,
                response: responseData
            });
        } else {
            return res.status(200).json({
                status: 204,
                success: false,
                message: result.message,
                response: []
            });
        }
    } catch (error) {
        console.error('Error in getConveyance controller:', error);
        return res.status(200).json({
            status: 500,
            success: false,
            message: `Error getting conveyance data: ${error.message}`,
            response: []
        });
    }
}

/**
 * View detailed conveyance data for a specific date
 * @param {Object} req - Express request object
 * @param {Object} res - Express response object
 * @returns {Object} JSON response
 */
async function viewConveyance(req, res) {
    try {
        // Validate required fields
        if (!req.body.date) {
            return res.status(400).json({
                status: 400,
                success: false,
                message: 'Conveyance date field is required',
                response: []
            });
        }

        // Get date from request
        const date = req.body.date;
        console.log(date);
        // Get user ID from authenticated user
        const userId = req.user.id;
        console.log(`Viewing conveyance for user: ${userId} on date: ${date}`);

        // Get conveyance details for the date
        const result = await conveyanceModel.getConveyanceByDate(date, userId);



        if (result) {
            return res.status(200).json({
                status: 200,
                success: true,
                message: "Conveyance info",
                response: result
            });
        } else {
            return res.status(200).json({
                status: 204,
                success: false,
                message: "Conveyance info not found",
                response: []
            });
        }
    } catch (error) {
        console.error('Error in viewConveyance controller:', error);
        return res.status(500).json({
            status: 500,
            success: false,
            message: `View Conveyance - ${error.message}`,
            response: []
        });
    }
}

async function claimapply(req, res) {
    try {
        const startDate = req.body.startDate;
        const endDate = req.body.endDate;
        const userId = req.user.id;
        const result = await conveyanceModel.claimapply(startDate, endDate, userId);
        return res.status(200).json({
            status: 200,
            success: true,
            message: "Claim applied",
            response: result
        });
    } catch (error) {
        console.error('Error in claimapply controller:', error);
        return res.status(500).json({
            status: 500,
            success: false,
            message: `Claim apply - ${error.message}`,
            response: []
        });
    }
}

async function claimhistory(req, res) {     

    try {

        const userId = req.user.id;
        const startDate = req.query.startDate;
        const endDate = req.query.endDate;
        // console.log(startDate);
        // console.log(endDate);
        if (startDate && endDate) { 
            const result = await conveyanceModel.claimhistory(userId, startDate, endDate);
            return res.status(200).json({
                status: 200,
                success: true,
                message: "Claim history by date",
                date: {
                    startDate: startDate,
                    endDate: endDate
                },
                response: result
            });
        } else {
            const result = await conveyanceModel.claimhistory(userId);
            return res.status(200).json({
                status: 200,
                success: true,
                message: "Claim history",
                response: result
            });
        }
    
    } catch (error) {   
        console.error('Error in claimhistory controller:', error);
        return res.status(500).json({
            status: 500,
            success: false,
            message: `Claim history - ${error.message}`,
            response: []
        });
    }
}

module.exports = {
    getConveyance,
    viewConveyance,
    claimapply,
    claimhistory
}; 