const dashboardModel = require('../models/dashboard');

// Function to get total counts of accounts, contacts, deals, meetings, and tasks
async function totalcounts(req, res) {
    try {
        // Validate date format
        const { start_date, end_date } = req.body;
        if (!start_date || !end_date) {
            return res.status(400).json({
                status: 400,
                success: false,
                message: 'Both start_date and end_date are required'
            });
        }

        // Validate date format (YYYY-MM-DD)
        const dateRegex = /^\d{4}-\d{2}-\d{2}$/;
        if (!dateRegex.test(start_date) || !dateRegex.test(end_date)) {
            return res.status(400).json({
                status: 400,
                success: false,
                message: 'Invalid date format. Use YYYY-MM-DD format'
            });
        }

        // Validate date range
        if (new Date(start_date) > new Date(end_date)) {
            return res.status(400).json({
                status: 400,
                success: false,
                message: 'start_date cannot be greater than end_date'
            });
        }

        const counts = await dashboardModel.totalcounts(req);
        return res.status(200).json({
            status: 200,
            success: true,
            message: 'Dashboard counts fetched successfully',
            data: counts
        });
    } catch (error) {
        return res.status(500).json({
            status: 500,
            success: false,
            message: 'Error fetching dashboard counts',
            error: error.message
        });
    }
}

// This function fetches daily attendance data
// It validates the input dates and checks if they are in the correct format
async function getDailyAttendance(req, res) {
    try {
        // Validate date format
        const { start_date, end_date } = req.body;
        if (!start_date || !end_date) {
            return res.status(400).json({
                status: 400,
                success: false,
                message: 'Both start_date and end_date are required'
            });
        }

        // Validate date format (YYYY-MM-DD)
        const dateRegex = /^\d{4}-\d{2}-\d{2}$/;
        if (!dateRegex.test(start_date) || !dateRegex.test(end_date)) {
            return res.status(400).json({
                status: 400,
                success: false,
                message: 'Invalid date format. Use YYYY-MM-DD format'
            });
        }

        // Validate date range
        if (new Date(start_date) > new Date(end_date)) {
            return res.status(400).json({
                status: 400,
                success: false,
                message: 'start_date cannot be greater than end_date'
            });
        }

        const attendance = await dashboardModel.getDailyAttendance(req);
        return res.status(200).json({
            status: 200,
            success: true,
            message: 'Daily attendance fetched successfully',
            data: attendance
        });
    } catch (error) {
        return res.status(500).json({
            status: 500,
            success: false,
            message: 'Error fetching daily attendance',
            error: error.message
        });
    }
}

// This function fetches weekly attendance data
// It validates the input dates and checks if they are in the correct format
async function getWeeklyAttendance(req, res) {
    try {
        // Validate date format
        const { start_date, end_date } = req.body;
        if (!start_date || !end_date) {
            return res.status(400).json({
                status: 400,
                success: false,
                message: 'Both start_date and end_date are required'
            });
        }

        // Validate date format (YYYY-MM-DD)
        const dateRegex = /^\d{4}-\d{2}-\d{2}$/;
        if (!dateRegex.test(start_date) || !dateRegex.test(end_date)) {
            return res.status(400).json({
                status: 400,
                success: false,
                message: 'Invalid date format. Use YYYY-MM-DD format'
            });
        }

        // Validate date range
        if (new Date(start_date) > new Date(end_date)) {
            return res.status(400).json({
                status: 400,
                success: false,
                message: 'start_date cannot be greater than end_date'
            });
        }

        // Validate range is not more than 31 days
        const daysDiff = Math.ceil((new Date(end_date) - new Date(start_date)) / (1000 * 60 * 60 * 24));
        if (daysDiff > 31) {
            return res.status(400).json({
                status: 400,
                success: false,
                message: 'Date range cannot exceed 31 days'
            });
        }

        const attendance = await dashboardModel.getWeeklyAttendance(req);
        return res.status(200).json({
            status: 200,
            success: true,
            message: 'Weekly attendance fetched successfully',
            data: attendance
        });
    } catch (error) {
        return res.status(500).json({
            status: 500,
            success: false,
            message: 'Error fetching weekly attendance',
            error: error.message
        });
    }
}

// last 5 meetings and tasks 
async function activitylist(req, res){
    try{
        const result = await dashboardModel.activitylist()

        return res.status(200).json({
            status: 200,
            message: 'Meeting and Tasks fetch successfully',
            meetings: result.meetings,
            tasks: result.tasks
        })

    }catch (error){ 
        return res.status(500).json({
            status: 500,
            message: "Meeting and Tasks fetch not successfully",
            error: error.message
        })
    }
}

module.exports = { 
    totalcounts,
    getDailyAttendance,
    getWeeklyAttendance,
    activitylist
 };