const Deal = require('../models/deal');
const User = require('../models/user');
const multer = require('multer');
const csv = require('csv-parser');
const fs = require('fs');
const path = require('path');
const { v4: uuidv4 } = require('uuid');
const { error } = require('console');
// const db = require('../config/database');

// Configure multer for CSV file upload
const storage = multer.diskStorage({
    destination: function (req, file, cb) {
        const uploadDir = path.join(__dirname, '../uploads/temp');
        if (!fs.existsSync(uploadDir)) {
            fs.mkdirSync(uploadDir, { recursive: true });
        }
        cb(null, uploadDir);
    },
    filename: function (req, file, cb) {
        const uniqueFilename = `${uuidv4()}_${file.originalname}`;
        cb(null, uniqueFilename);
    }
});

const upload = multer({
    storage: storage,
    limits: {
        fileSize: 10 * 1024 * 1024 // 10MB limit
    },
    fileFilter: (req, file, cb) => {
        if (file.mimetype === 'text/csv' || file.originalname.endsWith('.csv')) {
            cb(null, true);
        } else {
            cb(new Error('Only CSV files are allowed'));
        }
    }
});

async function createDeal(req, res) {
    try {
        const user_id = req.user.id;
        const dealId = await Deal.create({...req.body, user_id});

        res.status(201).json({
            success: true,
            data: { id: dealId },
            message: 'Deal created successfully'
        });
    } catch (error) {
        console.error('Error creating deal:', error);
         if (error.message.includes('already exists')) {
            return res.status(200).json({
                status: 409,
                message: error.message,
                error: 'Duplicate entry'
            });
        }
        res.status(500).json({
            status: 500,
            message: 'Error creating deal',
            error: error.message
        });
    }
}
async function getAllDeals(req, res) {
    try {
        const filters = req.query;
        const deals = await Deal.findAll(filters);
        res.json({
            success: true,
            data: deals,
            message: 'Deals retrieved successfully'
        });
    } catch (error) {
        console.error('Error getting deals:', error);
        res.status(500).json({
            success: false,
            message: 'Error retrieving deals',
            error: error.message
        });
    }
}

async function getDealById(req, res) {
    try {
        const deal = await Deal.findById(req.params.id);
        if (!deal) {
            return res.status(404).json({
                status: 404,
                success: false,
                message: 'Deal not found'
            });
        }
        res.json({
            status: 200,
            success: true,
            data: deal,
            message: 'Deal retrieved successfully'
        });
    } catch (error) {
        console.error('Error getting deal:', error);
        res.status(500).json({
            status: 500,
            success: false,
            message: 'Error retrieving deal',
            error: error.message
        });
    }
}

async function updateDeal(req, res) {
    try {
        const user_id = req.user.id;
        const deal_id = req.params.id;
        console.log({data: req.body});
        const success =  await Deal.update(deal_id, {...req.body, user_id});
        console.log(success);
        if(!success){
            return res.json({
                status: 404,
                success: false,
                message: 'Deal not found '
            })
        }

        if(success.status === 409){
            return res.status(200).json({
                status: 409,
                success: false,
                message: success.message
            })
        }
        // const result = await Deal.findById(deal_id);
        // console.log(result);
        res.json({
            status: 200,
            success: true,
            message: 'Deal updated successfully',
            // deal: result
        });
    } catch (error) {
        console.error('Error updating deal:', error);
        res.status(500).json({
            status: 500,
            success: false,
            message: 'Error updating deal',
            error: error.message
        });
    }
}

async function deleteDeal(req, res) {
    try {
        const success = await Deal.deleteRecord(req.params.id);
        if (!success) {
            return res.status(404).json({
                status: 404,
                success: false,
                message: 'Deal not found'
            });
        }
        res.json({
            status: 200,
            success: true,
            message: 'Deal deleted successfully'
        });
    } catch (error) {
        console.error('Error deleting deal:', error);
        res.status(500).json({
            status: 500,
            success: false,
            message: 'Error deleting deal',
            error: error.message
        });
    }
}

// deals stages get


async function getDealMeetings(req, res) {
    try {
        const dealId = req.params.id;
        const meetings = await Deal.findDealMeetings(dealId);

        res.json({
            status: true,
            message: "Deal meetings retrieved successfully",
            response: meetings
        });

    } catch (error) {
        console.error('Error in getDealMeetings:', error);
        res.status(500).json({
            status: false,
            message: error.message || 'Error retrieving deal meetings'
        });
    }
}
async function getDealTasks(req, res) {
    try {
        const dealId = req.params.id;
        const meetings = await Deal.findByDeals(dealId);

        res.json(meetings);

    } catch (error) {
        console.error('Error in getDealMeetings:', error);
        res.status(500).json({
            status: false,
            message: error.message || 'Error retrieving deal meetings'
        });
    }
}
async function addDealProducts(req, res) {
    try {
        const dealId = req.params.id;
        const { products, deal_value } = req.body;
        const userId = req.user.id;

        // Validate required fields
        if (!dealId || !products || !Array.isArray(products)) {
            return res.status(400).json({
                status: false,
                message: 'Deal ID and products array are required'
            });
        }

        // Add products to deal
        const result = await Deal.addProducts(dealId, products, deal_value, userId);

        if (!result.success) {
            return res.status(404).json({
                status: false,
                message: result.message || 'Deal not found'
            });
        }

        res.json({
            status: true,
            message: 'Deal products added successfully'
        });

    } catch (error) {
        console.error('Error in addDealProducts:', error);
        res.status(500).json({
            status: false,
            message: error.message || 'Error adding deal products'
        });
    }
}


// get deal stages
async function getdealstages(req, res) {
    try {
        const dealPipelinesWithStages = await Deal.getdealstages();
        return res.status(200).json({
            status: 200,
            success: true,
            message: 'Deal stages retrieved successfully',
            data: dealPipelinesWithStages
        });
    } catch (error) {
        console.error('Error fetching deal stages:', error);
        return res.status(500).json({   
            status: 500,
            success: false,
            message: 'Error fetching deal stages',
            error: error.message
        });
    }
}   

// create deal stages
// async function createdealstages(req, res) {
//     try {
//         const result = await Deal.createdealstages(req.body);
//         return res.status(201).json({
//             status: true,   
//             message: 'Deal stage created successfully',
//             data: result
//         });
//     } catch (error) {
//         console.error('Error creating deal stage:', error);
//         return res.status(500).json({
//             status: false,
//             message: 'Error creating deal stage',
//             error: error.message
            
//         });
//     }
// }

// update deal stages
async function updatedealstages(req, res) {
    try {
        const inputData = req.body;
        console.log('Input data:', inputData);

        // Validate that request body is an array
        if (!Array.isArray(inputData)) {
            return res.status(400).json({
                status: 400,
                success: false,
                message: 'Request body must be an array of pipeline configurations'
            });
        }

        // Validate each pipeline configuration
        for (const config of inputData) {
            // Validate pipeline object
            if (!config.pipeline || !config.pipeline.pipeline_name) {
                return res.status(400).json({
                    status: 400,
                    success: false,
                    message: 'Each configuration must have a pipeline object with pipeline_name'
                });
            }

            // Validate stages array
            if (!Array.isArray(config.stages) || config.stages.length === 0) {
                return res.status(400).json({
                    status: 400,
                    success: false,
                    message: 'Each pipeline must have a non-empty stages array'
                });
            }

            // Validate each stage
            for (const stage of config.stages) {
                // Validate required fields
                const requiredFields = ['deal_stage', 'probability', 'index'];
                const missingFields = requiredFields.filter(field => !stage[field]);
                
                if (missingFields.length > 0) {
                    return res.status(400).json({
                        status: 400,
                        success: false,
                        message: `Missing required fields in stage: ${missingFields.join(', ')}`
                    });
                }

                // Validate probability format and range
                const probability = parseInt(stage.probability.toString().replace('%', ''));
                if (isNaN(probability) || probability < 0 || probability > 100) {
                    return res.status(400).json({
                        status: 400,
                        success: false,
                        message: `Invalid probability value in stage: ${stage.deal_stage}. Must be between 0% and 100%`
                    });
                }

                // Validate index is positive
                if (stage.index < 1) {
                    return res.status(400).json({
                        status: 400,
                        success: false,
                        message: `Invalid index value in stage: ${stage.deal_stage}. Must be a positive number`
                    });
                }
            }
        }

        const results = await Deal.updatedealstages(inputData);

        return res.status(200).json({
            status: 200,
            success: true,
            message: 'Deal stages updated successfully',
            data: results
        });

    } catch (error) {
        console.error('Error updating deal stages:', error);
        
        // Handle specific errors
        if (error.message.includes('already in use') || 
            error.message.includes('Duplicate index') ||
            error.message.includes('Invalid stage order') ||
            error.message.includes('Invalid stage sequence')) {
            return res.status(409).json({
                status: 409,
                success: false,
                message: error.message
            });
        }

        return res.status(500).json({
            status: 500,
            success: false,
            message: 'Error updating deal stages',
            error: error.message
        });
    }
}

async function getDealStageslist(req, res) {
    try {
        const dealStages = await Deal.getDealStageslist(req.params.id);
        return res.status(200).json({
            status: 200,
            success: true,
            message: 'Deal stages retrieved successfully',
            data: dealStages
        });
    } catch (error) {
        console.error('Error fetching deal stages:', error);
        return res.status(500).json({
            status: 500,
            success: false,
            message: 'Error fetching deal stages',
            error: error.message
        });
    }
}

// // delete deal stages
// async function deletedealstages(req, res) {
//     try {
//         const id = req.params.id;
//         const result = await Deal.deletedealstages(id);
//         return res.status(200).json({
//             status: true,
//             message: 'Deal stage deleted successfully',
//             data: result
//         });
//     } catch (error) {   
//         console.error('Error deleting deal stage:', error);
//         return res.status(500).json({
//             status: false,
//             message: 'Error deleting deal stage',
//             error: error.message
//         });
//     }
// }

// get deal pipeline 
async function getdealpipeline(req, res) {
    try {
        const pipelines = await Deal.getdealPipelines();
        return res.status(200).json({
            status: true,
            message: 'Deal pipelines retrieved successfully',
            data: pipelines
        });
    } catch (error) {
        console.error('Error fetching deal pipeline:', error);
        return res.status(500).json({
            status: false,
            message: 'Error fetching deal pipelines',
            error: error.message
        });
    }
}

// get deal relations
// get deal relations
async function getDealRelations(req, res) {
    try {
        const dealId = req.params.id;
        const relations = await Deal.getDealRelations(dealId);

        if (!relations) {
            return res.status(404).json({
                status: 404,
                success: false,
                message: 'Deal not found'
            });
        }

        res.json({
            status: 200,
            success: true,
            message: 'Deal relations retrieved successfully',
            data: relations
        });

    } catch (error) {
        console.error('Error in getDealRelations:', error);
        res.status(500).json({
            status: 500,
            success: false,
            message: error.message || 'Error retrieving deal relations'
        });
    }
}
// update deal stage
async function updateStage(req, res) {
    try {
        const { deal_id, stage_id } = req.body;
        console.log({"deal_id":deal_id, "stage_id":stage_id});
        // Validate required fields
        if (!deal_id || !stage_id) {
            return res.status(400).json({
                status: false,
                message: 'Deal ID and Stage ID are required'
            });
        }

        await Deal.updateDealStage(deal_id, stage_id, req.user.id);

        res.json({
            status: true,
            message: 'Deal stage updated successfully'
        });

    } catch (error) {
        console.error('Error in updateStage:', error);
        res.status(500).json({
            status: false,
            message: error.message || 'Failed to update deal stage'
        });
    }
}


async function createFilterView(req, res) {
    console.log("Create Deal Filter View Controller");
    try {
        // Validate required fields
        if (!req.body.title || !req.body.share_with) {
            return res.status(400).json({
                status: false,
                message: 'Title and share with field are required'
            });
        }

        const view = await Deal.createFilterView(req.body);

        res.json({
            status: true,
            message: 'Deal View saved successfully',
            response: view
        });

    } catch (error) {
        console.error('Error in Deal:', error);
        res.status(500).json({
            status: false,
            message: error.message || 'Deal View not saved',
            response: null
        });
    }
}

// Get ALl Filters List
const getAllFiltersList = async (req, res) => {
    console.log("=== Starting getAllFiltersList Controller ===");
    try {
        console.log("Calling Deal.getAllFiltersList()");
        const filters = await Deal.getAllFiltersList();
        console.log("Filters received from model:", filters);
        
        if (!filters || filters.length === 0) {
            console.log("No filters found in the database");
            return res.status(404).json({ 
                status: 404, 
                message: 'No filters found' 
            });
        }
        
        console.log("Successfully retrieved filters, sending response");
        res.json({
            status: 200, 
            message: 'Filters retrieved successfully', 
             filters
        });
    } catch (error) {
        console.error("=== Error in getAllFiltersList Controller ===");
        console.error("Error details:", error);
        res.status(500).json({ 
            status: 500, 
            message: 'Error fetching filters: ' + error.message 
        });
    }
};


// Delete Account Filter View
async function deleteFilterView(req, res) {
    try {
        const viewId = req.params.id;
        const success = await Contact.deleteFilterView(viewId);
        if (!success) {
            return res.status(404).json({
                status: 404,
                success: false,
                message: 'Contact Filter View not found'
            }); 
        }
        res.json({
            status: 200,
            success: true,
            message: 'Contact Filter View deleted successfully'
        }); 
    } catch (error) {
        console.error('Error in deleteFilterView:', error);
        res.status(500).json({
            status: 500,
            success: false,
            message: error.message || 'Error deleting contact filter view'
        });
    }
}

async function getFieldsForFilter(req, res) {
    try {
        const userId = req.user.id;
        const fields = await Deal.getFieldsForFilter(userId);

        if (!fields || fields.length === 0) {
            return res.status(404).json({
                status: false,
                message: "No deal fields found"
            });
        }

        res.json({
            status: true,
            message: "Deal fields info",
            response: fields
        });

    } catch (error) {
        console.error('Error in getFieldsForFilter:', error);
        res.status(500).json({
            status: false,
            message: error.message || 'Error fetching getFieldsForFilter fields'
        });
    }
}



async function getDealListWithFilters(req, res) {
    

    try {
        const { filter } = req.body;
        const userId = req.user.id;
        console.log("userId", userId);
        console.log("ConstactWithFilters Controller");
        const role = await User.findUserById(userId);
        console.log("Role:", role);
        const role_id = role.roles_id;
        console.log("Role_id:", role_id);

        // Get filtered products
        const deals = await Deal.getDealListWithFilters(filter, userId, role_id);

        res.json({
            status: 200,
            message: 'Deals filtered successfully',
            data: deals
        });

    } catch (error) {
        console.error('Error in getContactListWithFilters:', error);
        res.status(500).json({
            status: false,
            message: error.message || 'Error filtering contacts'
        });
    }
}



async function updateFilterView(req, res) {
    console.log("Update Deal Filter View Controller");
    try {
        // Validate required fields
        if (!req.body.title || !req.body.share_with) {
            return res.status(400).json({
                status: false,
                message: 'Title and share with field are required'
            });
        }

        const view = await Deal.updateFilterView(req.body);

        res.json({
            status: 200,
            message: 'Deal View updated successfully',
            response: view
        });

    } catch (error) {
        console.error('Error in Deal:', error);
        res.status(500).json({
            status: false,
            message: error.message || 'Deal View not saved',
            response: null
        });
    }
}

// Bulk import deals
async function bulkImport(req, res) {
    try {
        // Validate required fields
        if (!req.file) {
            return res.status(400).json({
                status: 400,
                success: false,
                message: 'CSV file is required'
            });
        }

        if (!req.body.columnMappings) {
            return res.status(400).json({
                status: 400,
                success: false,
                message: 'Column mappings are required'
            });
        }

        let columnMappings;
        try {
            columnMappings = JSON.parse(req.body.columnMappings);
        } catch (error) {
            return res.status(400).json({
                status: 400,
                success: false,
                message: 'Invalid column mappings format'
            });
        }

        if (!Array.isArray(columnMappings)) {
            return res.status(400).json({
                status: 400,
                success: false,
                message: 'Column mappings must be an array'
            });
        }

        // Validate column mappings
        for (const mapping of columnMappings) {
            if (!mapping.csvColumn || !mapping.apiField) {
                return res.status(400).json({
                    status: 400,
                    success: false,
                    message: 'Each mapping must have csvColumn and apiField'
                });
            }
        }

        // Parse CSV file
        const csvData = [];
        const filePath = req.file.path;

        return new Promise((resolve, reject) => {
            fs.createReadStream(filePath)
                .pipe(csv())
                .on('data', (row) => {
                    csvData.push(row);
                })
                .on('end', async () => {
                    try {
                        // Clean up the uploaded file
                        fs.unlinkSync(filePath);

                        if (csvData.length === 0) {
                            return res.status(400).json({
                                status: 400,
                                success: false,
                                message: 'CSV file is empty or has no valid data'
                            });
                        }

                        // Process the import
                        const userId = req.user.id;
                        const results = await Deal.bulkImport(csvData, columnMappings, userId);

                        res.json({
                            status: 200,
                            success: true,
                            message: 'Bulk import completed',
                            data: {
                                total: results.total,
                                successful: results.successful,
                                failed: results.failed,
                                errors: results.errors.slice(0, 10) // Limit error details to first 10
                            },
                            errors: results.errors,
                            file: results.errorCsvFile
                        });

                        resolve();
                    } catch (error) {
                        // Clean up the uploaded file in case of error
                        if (fs.existsSync(filePath)) {
                            fs.unlinkSync(filePath);
                        }
                        reject(error);
                    }
                })
                .on('error', (error) => {
                    // Clean up the uploaded file in case of error
                    if (fs.existsSync(filePath)) {
                        fs.unlinkSync(filePath);
                    }
                    reject(error);
                });
        });

    } catch (error) {
        console.error('Error in bulk import:', error);
        res.status(500).json({
            status: 500,
            success: false,
            message: 'Error processing bulk import',
            error: error.message
        });
    }
}

module.exports = {
    createDeal,
    getAllDeals,
    getDealById,
    updateDeal,
    deleteDeal,
    getDealMeetings,
    getDealTasks,
    addDealProducts,
    getdealstages,
    updatedealstages,
    getdealpipeline,
    getDealRelations,
    updateStage,
    // deletedealstages
    getAllFiltersList,
    createFilterView,
    deleteFilterView,
    getFieldsForFilter,
    getDealStageslist,
    getDealListWithFilters,
    updateFilterView,
    bulkImport,
    upload
    
};