const File = require('../models/file');

class FileController {
    static async uploadFile(req, res) {
        try {
            const { belongsto, belongsto_id } = req.body;
            const file = req.file;

            if (!file || !belongsto || !belongsto_id) {
                return res.status(400).json({
                    status: false,
                    message: 'File, belongsto and belongsto_id are required'
                });
            }

            // Validate belongsto type
            const validTypes = ['contacts', 'accounts', 'deals'];
            if (!validTypes.includes(belongsto)) {
                return res.status(400).json({
                    status: false,
                    message: 'Invalid belongsto type'
                });
            }

            // Validate belongsto_id is a number
            const belongstoId = parseInt(belongsto_id);
            if (isNaN(belongstoId)) {
                return res.status(400).json({
                    status: false,
                    message: 'belongsto_id must be a valid number'
                });
            }

            const fileData = {
                originalname: file.originalname || 'unknown',
                filename: file.filename,
                mimetype: file.mimetype || 'application/octet-stream',
                size: file.size || 0,
                belongsto: belongsto,
                belongsto_id: belongstoId,
                userId: req.user.id || null
            };

            const result = await File.upload(fileData);

            res.json({
                status: true,
                message: 'File uploaded successfully',
                response: result
            });

        } catch (error) {
            console.error('Error in uploadFile:', error);
            res.status(500).json({
                status: false,
                message: error.message || 'Failed to upload file'
            });
        }
    }

    static async getFiles(req, res) {
        try {
            const { belongsto, belongsto_id } = req.params;

            const files = await File.getFiles(belongsto, belongsto_id);

            res.json({
                status: true,
                message: 'Files retrieved successfully',
                response: files
            });

        } catch (error) {
            console.error('Error in getFiles:', error);
            res.status(500).json({
                status: false,
                message: error.message || 'Failed to get files'
            });
        }
    }

    static async deleteFile(req, res) {
        try {
            const { id } = req.params;

            await File.delete(id);

            res.json({
                status: true,
                message: 'File deleted successfully'
            });

        } catch (error) {
            console.error('Error in deleteFile:', error);
            res.status(500).json({
                status: false,
                message: error.message || 'Failed to delete file'
            });
        }
    }
}

module.exports = FileController;