const multer = require('multer');
const path = require('path');
const { v4: uuidv4 } = require('uuid');
const FileUpload = require('../models/fileUpload');
const { stat } = require('fs');

// Configure multer for file upload
const storage = multer.diskStorage({
    destination: function (req, file, cb) {
        cb(null, FileUpload.uploadDir);
    },
    filename: function (req, file, cb) {
        const uniqueFilename = `${uuidv4()}${path.extname(file.originalname)}`;
        cb(null, uniqueFilename);
    }
});

const upload = multer({
    storage: storage,
    limits: {
        fileSize: 5 * 1024 * 1024 // 5MB limit
    },
    fileFilter: (req, file, cb) => {
        // Add file type validation if needed
        cb(null, true);
    }
});

async function uploadFile(req, res) {
    try {
        if (!req.file) {
            return res.status(400).json({
                status: 400,
                success: false,
                message: 'No file uploaded'
            });
        }

        const fileData = {
            originalname: req.file.originalname,
            filename: req.file.filename,
            mimetype: req.file.mimetype,
            size: req.file.size,
            path: req.file.path,
            owner_id: req.user.id, // Assuming user info is added by auth middleware
            share_with_team: req.body.share_with_team,
            related_to: req.body.related_to,
            related_id: req.body.related_id,
        };
        // console.log({fileData});
        const fileId = await FileUpload.saveFileInfo(fileData);
        const file = await FileUpload.getFileById(fileId);

        res.status(201).json({
            status: 201,
            success: true,
            message: 'File uploaded successfully',
            data: file
        });
    } catch (error) {
        console.error('Error in file upload:', error);
        res.status(500).json({
            status: 500,
            success: false,
            message: 'Error uploading file',
            error: error.message || error.response || 'Internal Server Error'
        });
    }
}

async function getFiles(req, res) {
    try {
        if (req.params.id) {
            const file = await FileUpload.getFileById(req.params.id);
            if (!file || file.owner_id !== req.user.id) {
                return res.status(404).json({
                    status: 404,
                    success: false,
                    message: 'File not found'
                });
            }
            const dataUrl = `/uploads/${file.url}`;
            file.dataUrl = dataUrl;
            return res.json({
                status: 200,
                success: true,
                data: file
            });
        }
        const files = await FileUpload.getFilesByUser(req.user.id);
        res.json({
            status: 200,
            success: true,
            data: files
        });
    } catch (error) {
        console.error('Error fetching files:', error);
        res.status(500).json({
            status: 500,
            success: false,
            message: 'Error fetching files',
            error: error.message || error.response || 'Internal Server Error'
        });
    }
}

async function deleteFile(req, res) {
    try {
        const fileId = req.params.id;
        const success = await FileUpload.deleteFile(fileId);

        if (!success) {
            return res.status(404).json({
                status: 404,
                success: false,
                message: 'File not found'
            });
        }

        res.json({
            status: 200,
            success: true,
            message: 'File deleted successfully'
        });
    } catch (error) {
        console.error('Error deleting file:', error);
        res.status(500).json({
            status: 500,
            success: false,
            message: 'Error deleting file',
            error: error.message
        });
    }
}

module.exports = {
    upload,
    uploadFile,
    getFiles,
    deleteFile
};