const Product = require('../models/products');
const multer = require('multer');
const path = require('path');
const fs = require('fs');
// Create new product
async function create(req, res) {
    try {
        // Validate required fields
        if (!req.body.product || !Array.isArray(req.body.product)) {
            return res.status(400).json({
                status: 400,
                success: false,
                message: 'Product fields are required and must be an array'
            });
        }

        const productId = await Product.create(req.body);
        
        // Fetch the created product
        const product = await Product.findById(productId);

        res.status(200).json({
            status: 200,
            success: true,
            message: 'Product created successfully',
            product: product
        });

    } catch (error) {
        console.error('Error in createProduct:', error);
        res.status(500).json({
            status: 500,
            success: false,
            message: error.message || 'Error creating product'
        });
    }
}

// Get all products
async function getAll(req, res) {
    try {
        const products = await Product.findAll();
        
        res.json({
            status: 200,
            success: true,
            message: "Products fetched successfully",
            products: products
        });

    } catch (error) {
        console.error('Error in getAllProducts:', error);
        res.status(500).json({
            status: 500,
            success: false,
            message: error.message || 'Error fetching products'
        });
    }
}

// Get product by ID
async function getById(req, res) {
    try {
        const product = await Product.findById(req.params.id);
        
        if (!product) {
            return res.status(404).json({
                status: 404,
                success: false,
                message: 'Product not found'
            });
        }

        res.json({
            status: 200,
            success: true,
            message: "Product fetched successfully",
            product: product
        });

    } catch (error) {
        console.error('Error in getProductById:', error);
        res.status(500).json({
            status: 500,
            success: false,
            message: error.message || 'Error fetching product'
        });
    }
}

// Update product
async function update(req, res) {
    try {
        // Validate required fields
        if (!req.body.product || !Array.isArray(req.body.product)) {
            return res.status(400).json({
                status: 400,
                success: false,
                message: 'Product fields are required and must be an array'
            });
        }

        const success = await Product.update(req.params.id, req.body);
        
        if (!success) {
            return res.status(404).json({
                status: 404,
                success: false,
                message: 'Product not found'
            });
        }

        // Fetch updated product
        const product = await Product.findById(req.params.id);

        res.json({
            status: 200,
            success: true,
            message: 'Product updated successfully',
            product: product
        });

    } catch (error) {
        console.error('Error in updateProduct:', error);
        res.status(500).json({
            status: 500,
            success: false,
            message: error.message || 'Error updating product'
        });
    }
}

// Delete product
async function remove(req, res) {
    try {
        const success = await Product.deleteRecord(req.params.id);
        
        if (!success) {
            return res.status(404).json({
                status: 404,
                success: false,
                message: 'Product not found'
            });
        }

        res.json({
            status: 200,
            success: true,
            message: 'Product deleted successfully'
        });

    } catch (error) {
        console.error('Error in deleteProduct:', error);
        res.status(500).json({
            status: 500,
            success: false,
            message: error.message || 'Error deleting product'
        });
    }
}
async function filterProducts(req, res) {
    try {
        const { filter } = req.body;
        const userId = req.user.id;

        // Get filtered products
        const products = await Product.filterProducts(filter, userId);

        res.json({
            status: 200,
            message: 'Products filtered successfully',
            products: products
        });

    } catch (error) {
        console.error('Error in filterProducts:', error);
        res.status(500).json({
            status: false,
            message: error.message || 'Error filtering products'
        });
    }
}
async function updateProductPrice(req, res) {
    try {
        const { productid } = req.body;

        // Validate required fields
        if (!productid) {
            return res.status(400).json({
                status: 400,
                message: 'Product ID is required'
            });
        }

        // Pass the entire req.body as pricingData
        const result = await Product.updatePrice(productid, req.body);

        res.json({
            status: 200,
            message: 'Product price updated successfully',
            response: result
        });

    } catch (error) {
        console.error('Erro r in updateProductPrice:', error);
        res.status(500).json({
            status: false,
            message: error.message || 'Error updating product price'
        });
    }
}
async function getProductFields(req, res) {
    try {
        const userId = req.user.id;
        const fields = await Product.getProductFields(userId);

        if (!fields || fields.length === 0) {
            return res.status(404).json({
                status: 404,
                message: "No product fields found"
            });
        }

        res.json({
            status: 200,
            message: "Product fields info",
            response: fields
        });

    } catch (error) {
        console.error('Error in getProductFields:', error);
        res.status(500).json({
            status: false,
            message: error.message || 'Error fetching product fields'
        });
    }
}
async function addProductView(req, res) {
    try {
        // Validate required fields
        if (!req.body.title || !req.body.share_with) {
            return res.status(400).json({
                status: false,
                message: 'Title and share with field are required'
            });
        }

        const view = await Product.createProductView(req.body);

        res.json({
            status: true,
            message: 'Product View saved successfully',
            response: view
        });

    } catch (error) {
        console.error('Error in addProductView:', error);
        res.status(500).json({
            status: false,
            message: error.message || 'Product View not saved',
            response: null
        });
    }
}
async function deletePrice(req, res) {
    try {
        const { productid } = req.body;

        // Validate required field
        if (!productid) {
            return res.status(400).json({
                status: false,
                message: 'Product ID field is required'
            });
        }

        const success = await Product.deletePrice(productid);

        if (success) {
            res.json({
                status: true,
                message: 'Price remove success'
            });
        } else {
            res.status(404).json({
                status: false,
                message: 'Price remove failed'
            });
        }

    } catch (error) {
        console.error('Error in deletePrice:', error);
        res.status(500).json({
            status: false,
            message: error.message || 'Price remove failed'
        });
    }
}
async function editProductView(req, res) {
    try {
        const { view_id } = req.body;

        if (!view_id) {
            return res.status(400).json({
                status: false,
                message: 'View ID field is required'
            });
        }

        const updatedView = await Product.updateProductView(req.body);

        res.json({
            status: true,
            message: 'Product View updated successfully',
            response: updatedView
        });

    } catch (error) {
        console.error('Error in editProductView:', error);
        res.status(500).json({
            status: false,
            message: error.message || 'Product View not updated',
            response: null
        });
    }
}

async function deleteProductView(req, res) {
    try {
        const { view_id } = req.body;

        if (!view_id) {
            return res.status(400).json({
                status: false,
                message: 'View ID field is required'
            });
        }

        const success = await Product.deleteProductView(view_id);

        if (success) {
            res.json({
                status: true,
                message: 'Product View deleted successfully',
                response: { id: view_id }
            });
        } else {
            res.status(404).json({
                status: false,
                message: 'Product View not found',
                response: null
            });
        }

    } catch (error) {
        console.error('Error in deleteProductView:', error);
        res.status(500).json({
            status: false,
            message: error.message || 'Product View not deleted',
            response: null
        });
    }
}
async function getCategories(req, res) {
    try {
        const categories = await Product.getCategories();
        
        res.json({
            status: true,
            message: 'Categories retrieved successfully',
            response: categories
        });

    } catch (error) {
        console.error('Error in getCategories:', error);
        res.status(500).json({
            status: false,
            message: error.message || 'Error fetching categories'
        });
    }
}
async function addDealProducts(req, res) {
    try {
        const { dealid, products, deal_value } = req.body;
        const userId = req.user.id;
        // Validate required fields
        if (!dealid) {
            return res.status(400).json({
                status: false,
                message: 'Deal ID is required'
            });
        }

        if (!Array.isArray(products) || products.length === 0) {
            return res.status(400).json({
                status: false,
                message: 'Products array is required'
            });
        }

        // Add user_id to each product
        const productsWithUser = products.map(product => ({
            ...product,
            user_id: req.user.id
        }));

        const success = await Product.addDealProducts(dealid, productsWithUser, deal_value,userId);

        res.json({
            status: true,
            message: 'Deal products added successfully'
        });

    } catch (error) {
        console.error('Error in addDealProducts:', error);
        res.status(500).json({
            status: false,
            message: error.message || 'Failed to add deal products'
        });
    }
}
async function getDealProducts(req, res) {
    try {
        console.log(req.body);
        const { dealid } = req.body;

        // Validate required fields
        if (!dealid) {
            return res.status(400).json({
                status: false,
                message: 'Deal ID is required'
            });
        }

        const result = await Product.getDealProducts(dealid);

        res.json({
            status: true,
            message: 'Deal products info',
            response: result
        });

    } catch (error) {
        console.error('Error in getDealProducts:', error);
        res.status(500).json({
            status: false,
            message: error.message || 'Failed to get deal products'
        });
    }
}
async function syncQuote(req, res) {
    try {
        const { dealid, quoteid } = req.body;

        // Validate required fields
        if (!dealid) {
            return res.status(400).json({
                status: false,
                message: 'Deal ID is required'
            });
        }

        if (!quoteid) {
            return res.status(400).json({
                status: false,
                message: 'Quote ID is required'
            });
        }

        const result = await Product.syncQuote(dealid, quoteid, req.user.id);

        res.status(200).json({
            status: true,
            message: result.message,
            hasProducts: result.hasProducts
        });

    } catch (error) {
        console.error('Error in syncQuote:', error);
        res.status(500).json({
            status: false,
            message: error.message || 'Quotation sync failed'
        });
    }
}
async function removeSync(req, res) {
    try {
        const { dealid } = req.body;

        // Validate required field
        if (!dealid) {
            return res.status(400).json({
                status: false,
                message: 'Deal ID is required'
            });
        }

        await Product.removeSync(dealid);

        res.json({
            status: true,
            message: 'Quotation unsync success'
        });

    } catch (error) {
        console.error('Error in removeSync:', error);
        res.status(500).json({
            status: false,
            message: error.message || 'Quotation unsync failed'
        });
    }
}
async function removeDealProduct(req, res) {
    try {
        const { dealid, productid } = req.body;

        // Validate required fields
        if (!dealid) {
            return res.status(400).json({
                status: false,
                message: 'Deal ID is required'
            });
        }

        if (!productid) {
            return res.status(400).json({
                status: false,
                message: 'Product ID is required'
            });
        }

        await Product.removeDealProduct(dealid, productid);

        res.json({
            status: true,
            message: 'Product removed from deal successfully'
        });

    } catch (error) {
        console.error('Error in removeDealProduct:', error);
        res.status(500).json({
            status: false,
            message: error.message || 'Failed to remove product from deal'
        });
    }
}
async function uploadFile(req, res) {
    try {
        // Check if file exists
        if (!req.file) {
            return res.status(400).json({
                status: false,
                message: 'File field is required '+JSON.stringify(req.file)
            });
        }

        try {
            const result = await Product.uploadFile(req.file, req.body, req.user.id);
            
            // Clean up temp file
            fs.unlinkSync(req.file.path);

            res.json({
                status: true,
                message: 'File add success',
                response: result
            });

        } catch (error) {
            // Clean up temp file on error
            if (req.file) {
                fs.unlinkSync(req.file.path);
            }
            throw error;
        }

    } catch (error) {
        console.error('Error in uploadFile:', error);
        res.status(500).json({
            status: false,
            message: error.message || 'File add failed'
        });
    }
}
async function deleteFile(req, res) {
    try {
        const { fileid } = req.body;

        // Validate required field
        if (!fileid) {
            return res.status(400).json({
                status: false,
                message: 'File ID is required'
            });
        }

        await Product.deleteFile(fileid);

        res.json({
            status: true,
            message: 'File delete success'
        });

    } catch (error) {
        console.error('Error in deleteFile:', error);
        res.status(500).json({
            status: false,
            message: error.message || 'File delete failed'
        });
    }
}
async function getLookupData(req, res) {
    try {
        const data = await Product.getLookupData();
        
        res.json({
            status: true,
            message: 'Lookup data retrieved successfully',
            response: data
        });

    } catch (error) {
        console.error('Error in getLookupData:', error);
        res.status(500).json({
            status: false,
            message: error.message || 'Failed to retrieve lookup data'
        });
    }
}



// Get ALl Filters List
const getAllFiltersList = async (req, res) => {
    console.log("=== Starting getAllFiltersList Controller ===");
    try {
        console.log("Calling Product.getAllFiltersList()");
        const filters = await Product.getAllFiltersList();
        console.log("Filters received from model:", filters);
        
        if (!filters || filters.length === 0) {
            console.log("No filters found in the database");
            return res.status(404).json({ 
                status: 404, 
                message: 'No filters found' 
            });
        }
        
        console.log("Successfully retrieved filters, sending response");
        res.json({
            status: 200, 
            message: 'Filters retrieved successfully', 
             filters
        });
    } catch (error) {
        console.error("=== Error in getAllFiltersList Controller ===");
        console.error("Error details:", error);
        res.status(500).json({ 
            status: 500, 
            message: 'Error fetching filters: ' + error.message 
        });
    }
};

async function updateFilterView(req, res) {
    console.log("Update Product Filter View Controller");
    try {
        // Validate required fields
        if (!req.body.title || !req.body.share_with) {
            return res.status(400).json({
                status: false,
                message: 'Title and share with field are required'
            });
        }

        const view = await Product.updateFilterView(req.body);

        res.json({
            status: true,
            message: 'Product View updated successfully',
            response: view
        });

    } catch (error) {
        console.error('Error in Product:', error);
        res.status(500).json({
            status: false,
            message: error.message || 'Product View not saved',
            response: null
        });
    }
}

// Delete Product Filter View
async function deleteFilterView(req, res) {
    try {
        const viewId = req.params.id;
        const success = await Product.deleteFilterView(viewId);
        if (!success) {
            return res.status(404).json({
                status: 404,
                success: false,
                message: 'Product Filter View not found'
            }); 
        }
        res.json({
            status: 200,
            success: true,
            message: 'Product Filter View deleted successfully'
        }); 
    } catch (error) {
        console.error('Error in deleteFilterView:', error);
        res.status(500).json({
            status: 500,
            success: false,
            message: error.message || 'Error deleting product filter view'
        });
    }
}
module.exports = {
    create,
    getAll,
    getById,
    update,
    remove,
    filterProducts,
    updateProductPrice,
    getProductFields,
    addProductView,
    deletePrice,
    editProductView,
    deleteProductView,
    getCategories,
    addDealProducts,
    getDealProducts,
    syncQuote,
    removeSync,
    removeDealProduct,
    uploadFile,
    deleteFile,
    getLookupData,
    getAllFiltersList,
    updateFilterView,
    deleteFilterView
};