const Quotation = require('../models/quotation');

class QuotationController {
    static async create(req, res) {
        try {
            if (!Array.isArray(req.body) || req.body.length === 0) {
                return res.status(400).json({
                    status: 400,
                    success: false,
                    message: 'Invalid input format'
                });
            }

            const requiredFields = req.body.filter(field => field.required === 'on');
            const missingFields = requiredFields.filter(field => !field.field_value);

            if (missingFields.length > 0) {
                return res.status(400).json({
                    status: 400,
                    success: false,
                    message: `Required fields missing: ${missingFields.map(f => f.field_label).join(', ')}`
                });
            }

            const result = await Quotation.create(req.body, req.user.id);

            res.json({
                status: 200,
                success: true,
                message: 'Quotation created successfully',
                response: {
                    id: result.id,
                    quotation_no: result.quotation_no
                }
            });

        } catch (error) {
            console.error('Error in create:', error);
            res.status(500).json({
                status: 500,
                success: false,
                message: error.message || 'Quotation creation failed'
            });
        }
    }

    static async getAll(req, res) {
        try {
            const quotations = await Quotation.getAll(req.query);

            res.json({
                status: 200,
                success: true,
                message: 'Quotations retrieved successfully',
                response: quotations
            });

        } catch (error) {
            console.error('Error in getAll:', error);
            res.status(500).json({
                status: 500,
                success: false,
                message: error.message || 'Failed to retrieve quotations'
            });
        }
    }

    static async getById(req, res) {
        try {
            const quotation = await Quotation.getById(req.params.id);

            res.json({
                status: 200,
                success: true,
                message: 'Quotation retrieved successfully',
                response: quotation
            });

        } catch (error) {
            console.error('Error in getById:', error);
            res.status(500).json({
                status: 500,
                success: false,
                message: error.message || 'Failed to retrieve quotation'
            });
        }
    }

    static async update(req, res) {
        try {
            if (!Array.isArray(req.body) || req.body.length === 0) {
                return res.status(400).json({
                    status: 400,
                    success: false,
                    message: 'Invalid input format'
                });
            }

            const requiredFields = req.body.filter(field => field.required === 'on');
            const missingFields = requiredFields.filter(field => !field.field_value);

            if (missingFields.length > 0) {
                return res.status(400).json({
                    status: 400,
                    success: false,
                    message: `Required fields missing: ${missingFields.map(f => f.field_label).join(', ')}`
                });
            }

            await Quotation.update(req.params.id, req.body, req.user.id);

            res.json({
                status: 200,
                success: true,
                message: 'Quotation updated successfully'
            });

        } catch (error) {
            console.error('Error in update:', error);
            res.status(500).json({
                status: 500,
                success: false,
                message: error.message || 'Failed to update quotation'
            });
        }
    }

static async delete(req, res) {
        try {
            const { id } = req.params;

            await Quotation.delete(id, req.user.id);

            res.json({
                status: 200,
                success: true,
                message: 'Quotation deleted successfully'
            });

        } catch (error) {
            console.error('Error in delete:', error);
            res.status(500).json({
                status: 500,
                success: false,
                message: error.message || 'Failed to delete quotation'
            });
        }
    }
    static async manageQuoteProducts(req, res) {
        try {
            const { quoteid, products } = req.body;

            // Validate required fields
            if (!quoteid) {
                return res.status(400).json({
                    status: 400,
                    success: false,
                    message: 'Quote ID is required'
                });
            }

            if (!Array.isArray(products) || products.length === 0) {
                return res.status(400).json({
                    status: 400,
                    success: false, 
                    message: 'Products array is required'
                });
            }

            // Validate each product
            for (const product of products) {
                if (!product.product_id || !product.unitprice || !product.quantity) {
                    return res.status(400).json({
                        status: 400,
                        success: false, 
                        message: 'Each product must have product_id, unitprice, and quantity'
                    });
                }
            }

            const result = await Quotation.manageQuoteProducts(quoteid, products, req.user.id);

            res.json({
                status: 200,
                success: true,
                message: 'Quote products updated successfully',
                response: result
            });

        } catch (error) {
            console.error('Error in manageQuoteProducts:', error);
            res.status(500).json({
                status: 500,
                success: false,
                message: error.message || 'Failed to update quote products'
            });
        }
    }
    static async deleteQuoteProduct(req, res) {
        try {
            const { quoteproductid, quoteid } = req.body;

            // Validate required fields
            if (!quoteproductid) {
                return res.status(400).json({
                    status: 400,
                    success: false, 
                    message: 'Quote product ID is required'
                });
            }

            if (!quoteid) {
                return res.status(400).json({
                    status: 400,
                    success: false, 
                    message: 'Quote ID is required'
                });
            }

            await Quotation.deleteQuoteProduct(quoteproductid, quoteid, req.user.id);

            res.json({
                status: 200,
                success: true,
                message: 'Quote product deleted successfully'
            });

        } catch (error) {
            console.error('Error in deleteQuoteProduct:', error);
            res.status(500).json({
                status: 500,
                success: false,
                message: error.message || 'Failed to delete quote product'
            });
        }
    }
    static async updateStage(req, res) {
        try {
            const { quoteid, quotestage } = req.body;

            // Validate required fields
            if (!quoteid) {
                return res.status(400).json({
                    status: 400,
                    success: false, 
                    message: 'Quote ID is required'
                });
            }

            if (!quotestage) {
                return res.status(400).json({
                    status: 400,
                    success: false, 
                    message: 'Quote stage is required'
                });
            }

            await Quotation.updateStage(quoteid, { quotestage });

            res.json({
                status: 200,
                success: true,
                message: 'Quote stage updated successfully'
            });

        } catch (error) {
            console.error('Error in updateStage:', error);
            res.status(500).json({
                status: 500,
                success: false,
                message: error.message || 'Quote stage update failed'
            });
        }
    }
    static async generateAndSharePDF(req, res) {
        try {
            const { id } = req.params;

            const result = await Quotation.generateAndSharePDF(id);

            res.json({
                status: 200,
                success: true,
                message: 'PDF generated successfully',
                response: {
                    pdf_url: result.pdf_url
                }
            });

        } catch (error) {
            console.error('Error in generateAndSharePDF:', error);
            res.status(500).json({
                status: 500,
                success: false,
                message: error.message || 'Failed to generate PDF'
            });
        }
    }
    static async getQuoteProducts(req, res) {
        try {
            const { id } = req.params;

            const result = await Quotation.getQuoteProducts(id);

            res.json({
                status: 200,
                success: true,
                message: 'Quotation products retrieved successfully',
                response: result
            });

        } catch (error) {
            console.error('Error in getQuoteProducts:', error);
            res.status(500).json({
                status: 500,
                success: false,
                message: error.message || 'Failed to retrieve quotation products'
            });
        }
    }
}
module.exports = QuotationController;