const db = require('../config/db');
const Role = require('../models/role');

const validateAccessScope = (accessScope) => {
    const requiredModules = ['contact', 'sales_account', 'deal', 'appointment', 'task', 'notes', 'calllog'];
    const requiredPermissions = ['create', 'view', 'edit', 'delete'];
    
    for (const module of requiredModules) {
        if (!accessScope[module]) {
            throw new Error(`Missing permissions for module: ${module}`);
        }
        
        for (const permission of requiredPermissions) {
            if (accessScope[module][permission] === undefined) {
                throw new Error(`Missing ${permission} permission for ${module}`);
            }
            
            // Validate view/edit/delete can be either boolean or "restricted"/"global"
            if (permission === 'view' || permission === 'edit' || permission === 'delete') {
                const value = accessScope[module][permission];
                if (typeof value !== 'boolean' && value !== 'restricted' && value !== 'global' && value !== 'owned' && value !== 'territory') {
                    throw new Error(`Invalid ${permission} permission value for ${module}`);
                }
            }
        }
    }
};

async function createRole(req, res) {
    const connection = await db.getConnection();
    try {
        const {
            name,
            scope,
            reporting_to,
            location,
            eligible_transport,
            default_transport,
            permissions,
            description,
            active = 1
        } = req.body;
        
        const access_scope = permissions;

        // Validate required fields
        if (!name || !access_scope) {
            return res.status(400).json({
                status: false,
                message: 'Name and access scope are required'
            });
        }

        // Validate access_scope structure
        validateAccessScope(access_scope);

        await connection.beginTransaction();

        const [result] = await connection.execute(
            `INSERT INTO roles (
                name, 
                scope,
                reporting_to,
                location,
                eligible_transport,
                default_transport,
                access_scope,
                active,
                created_at,
                updated_at
            ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, NOW(), NOW())`,
            [
                name,
                scope || null,
                reporting_to || null,
                location || null,
                eligible_transport || null,
                default_transport || null,
                JSON.stringify(access_scope),
                active
            ]
        );

        await connection.commit();

        res.status(201).json({
            status: true,
            message: 'Role created successfully',
            data: {
                id: result.insertId,
                name,
                scope,
                reporting_to,
                location,
                eligible_transport,
                default_transport,
                description,
                access_scope,
                active
            }
        });

    } catch (error) {
        await connection.rollback();
        res.status(500).json({
            status: false,
            message: 'Error creating role',
            error: error.message
        });
    } finally {
        connection.release();
    }
}

async function getAllRoles(req, res) {
    try {
        const roles = await Role.findAll();
        res.json({ data: {
                    status: 200,
                    success: true,
                    message: 'Roles fetched successfully',
                    roles: roles.rows,
                    reporting_to: roles.rows1
                }
            });
    } catch (error) {
        res.status(500).json({ data: {
                                status: 500,
                                success: false,
                                message: error.message
                            }
                        });
    }
}

async function getRoleById(req, res) {
    try {
        const role = await Role.findById(req.params.id);
        if (!role) {
            return res.status(404).json({ data: {
                                        status: 404,
                                        success: false,
                                        message: 'Role not found' 
                                    }
                                });
        }
        res.json({ data: {
                    status: 200,
                    success: true, 
                    message: 'Role fetched successfully',
                    data: role 
                }
            });
    } catch (error) {
        res.status(500).json({ data: {
                                status: 500,
                                success: false,
                                message: error.message 
                            }
                        });
    }
}

async function updateRole(req, res) {
    const connection = await db.getConnection();
    try {
        const { id } = req.params;
        const {
            name,
            scope,
            reporting_to,
            location,
            eligible_transport,
            default_transport,
            description,
            access_scope,
            active
        } = req.body;

        // Validate access_scope if provided
        if (access_scope) {
            validateAccessScope(access_scope);
        }

        await connection.beginTransaction();

        // Build update query dynamically
        const updateFields = [];
        const updateValues = [];

        if (name) {
            updateFields.push('name = ?');
            updateValues.push(name);
        }
        if (scope !== undefined) {
            updateFields.push('scope = ?');
            updateValues.push(scope);
        }
        if (reporting_to !== undefined) {
            updateFields.push('reporting_to = ?');
            updateValues.push(reporting_to);
        }
        if (location !== undefined) {
            updateFields.push('location = ?');
            updateValues.push(location);
        }
        if (eligible_transport !== undefined) {
            updateFields.push('eligible_transport = ?');
            updateValues.push(eligible_transport);
        }
        if (default_transport !== undefined) {
            updateFields.push('default_transport = ?');
            updateValues.push(default_transport);
        }
        if (description !== undefined) {
            updateFields.push('description = ?');
            updateValues.push(description);
        }
        if (access_scope) {
            updateFields.push('access_scope = ?');
            updateValues.push(JSON.stringify(access_scope));
        }
        if (active !== undefined) {
            updateFields.push('active = ?');
            updateValues.push(active);
        }

        updateFields.push('updated_at = NOW()');
        updateValues.push(id);

        const [result] = await connection.execute(
            `UPDATE roles SET ${updateFields.join(', ')} WHERE id = ?`,
            updateValues
        );

        if (result.affectedRows === 0) {
            return res.status(404).json({
                status: false,
                message: 'Role not found'
            });
        }

        await connection.commit();

        res.json({
            status: true,
            message: 'Role updated successfully',
            data: {
                id,
                name,
                scope,
                reporting_to,
                location,
                eligible_transport,
                default_transport,
                description,
                access_scope,
                active
            }
        });

    } catch (error) {
        await connection.rollback();
        res.status(500).json({
            status: false,
            message: 'Error updating role',
            error: error.message
        });
    } finally {
        connection.release();
    }
}

async function deleteRole(req, res) {
    try {
        const success = await Role.deleteRole(req.params.id);
        if (!success) {
            return res.status(404).json({  data: {
                                        status: 404,
                                        success: false,
                                        message: 'Role not found' 
                                    }
                                });
        }
        res.json({ data: {
                    status: 200,
                    success: true 
                }
            });
    } catch (error) {
        res.status(500).json({ data: {  
                                status: 500,
                                success: false,
                                message: error.message 
                            }
                        });
    }
}

async function roleactivate(req, res) {
    try {
        if (req.body.active === 1) {
            const success = await Role.update(req.params.id, { active: req.body.active });
            res.json({ data: { status: 200, success: true, message: 'Role activated successfully' } });
        } else {
            const success = await Role.update(req.params.id, { active: req.body.active });
            res.json({ data: { status: 200, success: true, message: 'Role deactivated successfully' } });
        }
    } catch (error) {
        res.status(500).json({ data: { status: 500, success: false, message: error.message } });
    }
}
module.exports = {
    createRole,
    getAllRoles,
    getRoleById,
    updateRole,
    deleteRole,
    roleactivate
};