const SalesAccount = require('../models/salesAccounts');
const User = require('../models/user');
const multer = require('multer');
const csv = require('csv-parser');
const fs = require('fs');
const path = require('path');
const { v4: uuidv4 } = require('uuid');
const { error } = require('console');

// Configure multer for CSV file upload
const storage = multer.diskStorage({
    destination: function (req, file, cb) {
        const uploadDir = path.join(__dirname, '../uploads/temp');
        if (!fs.existsSync(uploadDir)) {
            fs.mkdirSync(uploadDir, { recursive: true });
        }
        cb(null, uploadDir);
    },
    filename: function (req, file, cb) {
        const uniqueFilename = `${uuidv4()}_${file.originalname}`;
        cb(null, uniqueFilename);
    }
});

const upload = multer({
    storage: storage,
    limits: {
        fileSize: 10 * 1024 * 1024 // 10MB limit
    },
    fileFilter: (req, file, cb) => {
        if (file.mimetype === 'text/csv' || file.originalname.endsWith('.csv')) {
            cb(null, true);
        } else {
            cb(new Error('Only CSV files are allowed'));
        }
    }
});

// Create new SalesAccount
const create = async (req, res) => {
    try {
        const SalesAccountId = await SalesAccount.create(req.body);
        res.status(201).json({
            status: 201,
            success: true,
            message: 'SalesAccount created successfully',
            SalesAccountId
        });
    } catch (error) {
        console.error('Error creating SalesAccount:', error);
         if (error.message.includes('already exists')) {
            return res.status(200).json({
                status: 409,
                success: false,
                message: error.message,
                error: 'Duplicate entry'
            });
        }
        res.status(500).json({
            status: 500,
            success: false,
            message: 'Error creating account',
            error: error.message
        });
    }
};
// Get all SalesAccounts
const getAll = async (req, res) => {
    try {
         const userId = req.user.id; // Get from auth middleware
         console.log(userId, "userId");
        const sales_accounts = await SalesAccount.findAll(userId);
        
        //const sales_accounts = await SalesAccount.findAll();
        
        res.json({ 
            status: 200,
            success: true,
            message:"Sales Accounts fecthed successfully", 
            sales_accounts 
        });
    } catch (error) {
        console.error('Error fetching SalesAccounts:', error);
        res.status(500).json({ 
            status: 500,
            success: false,
            message: 'Error fetching SalesAccounts' 
        });
    }
};
// Get SalesAccount by ID
const getById = async (req, res) => {
    try {
        const sales_account = await SalesAccount.findById(req.params.id);
        if (!sales_account) {
            return res.status(404).json({ 
                status: 404,
                success: false,
                message: 'SalesAccount not found' 
            });
        }
        res.json({
            status: 200,
            success: true,
            message:"Sales Account fecthed successfully", 
            account: sales_account.account
        });
    } catch (error) {
        // console.error('Error fetching SalesAccount:', error);
        res.status(500).json({ 
            status: 500,
            success: false,
            message: 'Error fetching SalesAccount ' + error 
        });
    }
};

// Update SalesAccount
const update = async (req, res) => {
    try {
        const success = await SalesAccount.update(req.params.id, req.body);
// console.log("Update SalesAccount", success);
        if (!success) {
            return res.status(404).json({
                status: 404,
                success: false,
                message: 'SalesAccount not found' 
            });
        }
        if(success.status == 409){
            return res.status(200).json(success);
        }

        res.json({ 
            status: 200,
            success: true,
            message: 'SalesAccount updated successfully' 
        });
    } catch (error) {
        console.error('Error updating SalesAccount:', error);
        res.status(500).json({ 
            status: 500, 
            success: false,
            message: 'Error updating SalesAccount' + error 
        });
    }
};

// Delete SalesAccount
const remove = async (req, res) => {
    try {
        const success = await SalesAccount.deleteRecord(req.params.id);
        if (!success) {
            return res.status(404).json({ 
                status: 404,
                success: false,
                message: 'SalesAccount not found' 
            });
        }
        res.json({ 
            status: 200,
            success: true,
            message: 'SalesAccount deleted successfully' 
        });
    } catch (error) {
        // console.error('Error deleting SalesAccount:', error);
        res.status(500).json({ 
            status: 500,
            success: false,
            message: 'Error deleting SalesAccount ' + error 
        });
    }
};
async function getAccountMeetings(req, res) {
    try {
        const accountId = req.params.id;
        const meetings = await SalesAccount.findAccountMeetings(accountId);
        
        res.json({
            status: 200,
            success: true,
            message: "Account meetings retrieved successfully",
            response: meetings
        });

    } catch (error) {
        console.error('Error in getAccountMeetings:', error);
        res.status(500).json({
            status: false,
            success: false,
            message: error.message || 'Error retrieving account meetings'   
        });
    }
}

async function getAccountTasks(req, res) {
    try {
        const accountId = req.params.id;
        const meetings = await SalesAccount.findByTasks(accountId);
        
        res.json({
            status: 200,
            success: true,
            message: "Account tasks retrieved successfully",
            tasks   : meetings
        });

    } catch (error) {
        console.error('Error in getAccountTasks:', error);
        res.status(500).json({
            status: false,
            success: false,
            message: error.message || 'Error retrieving account tasks'
        });
    }
}

// get account relations
async function getAccountRelations(req, res) {
    try {
        const accountId = req.params.id;
        const relations = await SalesAccount.getAccountRelations(accountId);

        if (!relations) {
            return res.status(404).json({
                status: 404,
                success: false,
                message: 'Sales Account not found'
            });
        }

        res.json({
            status: 200,
            success: true,
            message: 'Account relations retrieved successfully',
            data: relations
        });

    } catch (error) {
        console.error('Error in getAccountRelations:', error);
        res.status(500).json({
            status: 500,
            success: false,
            message: error.message || 'Error retrieving account relations'
        });
    }
}


async function createAccountFilterView(req, res) {
    try {
        // Validate required fields
        if (!req.body.title || !req.body.share_with) {
            return res.status(400).json({
                status: 400,
                success: false,
                message: 'Title and share with field are required'
            });
        }

        const view = await SalesAccount.createAccountFilterView(req.body);

        res.json({
            status: 200,
            success: true,
            message: 'Account View saved successfully',
            response: view
        });

    } catch (error) {
        console.error('Error in Account:', error);
        res.status(500).json(
            {
            status: 500,
            success: false,
            message: error.message || 'Account View not saved',
            response: null
        });
    }
}

// Get ALl Filters List
const getAllFiltersList = async (req, res) => {
    console.log("Read Accounts Filters Controller");
    try {
        const filters = await SalesAccount.getAllFiltersList();
        if (!filters) {
            return res.status(404).json({ 
                status: 404,
                success: false,
                message: 'Filters not found' 
            });
        }
        res.json({ 
            status: 200,
            success: true,
            message: 'Filters List', 
            response: filters
        });
    } catch (error) {
        console.log("Error in Read Accounts Filters Controller "+error);
        res.status(500).json({ 
            status: 500,
            success: false,
            message: error 
        });
    }
};

// update sales activity
async function updateSalesActivity(req, res) {
    try {
        const { id } = req.params;
        const data = req.body;
        const result = await salesActivityModel.updateMeetingStatus(id, data);
        if (!result) {
            return res.status(404).json({ 
                status: 404, 
                success: false,
                message: 'Sales activity not found'
             });        }
        res.json({ 
            status: 200,
            success: true,
            message: 'Sales activity updated successfully' 
            });
        }catch (error) {
            res.status(500).json({ 
                status: 500,
                success: false,
                message: 'Error updating sales activity',
                error: error.message    
            });
        }
}

// Delete Account Filter View
async function deleteAccountFilterView(req, res) {
    try {
        const viewId = req.params.id;
        const success = await SalesAccount.deleteAccountFilterView(viewId);
        if (!success) {
            return res.status(404).json({
                status: 404,
                success: false,
                message: 'Account Filter View not found'
            }); 
        }
        res.json({
            status: 200,
            success: true,
            message: 'Account Filter View deleted successfully'
        }); 
    } catch (error) {
        console.error('Error in deleteAccountFilterView:', error);
        res.status(500).json({
            status: 500,
            success: false,
            message: error.message || 'Error deleting account filter view'
        });
    }
}
async function getFieldsForFilter(req, res) {
    try {
        const userId = req.user.id;
        const fields = await SalesAccount.getFieldsForFilter(userId);

        if (!fields || fields.length === 0) {
            return res.status(404).json({
                status: 404,
                success: false, 
                message: "No account fields found"
            });
        }

        res.json({
            status: 200,
            success: true,
            message: "Account fields info",
            response: fields
        });

    } catch (error) {
        console.error('Error in getFieldsForFilter:', error);
        res.status(500).json({
            status: false,
            message: error.message || 'Error fetching getFieldsForFilter fields'
        });
    }
}


async function getAccountListWithFilters(req, res) {
    console.log("ConstactWithFilters Controller");
    try {
        const { filter } = req.body;
        const userId = req.user.id;
        console.log("userId", userId);
        const role = await User.findUserById(userId);
        console.log("role", role);
        const role_id = role.roles_id;
        console.log("role_id", role_id);  

        // Get filtered products
        const accounts = await SalesAccount.getAccountListWithFilters(filter, userId, role_id);

        res.json({
            status: 200,
            success: true,
            message: 'Accounts filtered successfully',
            sales_accounts: accounts
        });

    } catch (error) {
        console.error('Error in getContactListWithFilters:', error);
        res.status(500).json({
            status: 500,
            success: false,
            message: error.message || 'Error filtering contacts'
        });
    }
}


async function updateFilterView(req, res) {
    console.log("Update Account Filter View Controller");
    try {
        // Validate required fields
        if (!req.body.title || !req.body.share_with) {
            return res.status(400).json({
                status: 400,
                success: false,
                message: 'Title and share with field are required'
            });
        }

        const view = await SalesAccount.updateFilterView(req.body);

        res.json({
            status: 200,
            success: true,
            message: 'Account View updated successfully',
            response: view
        });

    } catch (error) {
        console.error('Error in Account:', error);
        res.status(500).json({
            status: 500,
            success: false,
            message: error.message || 'Account View not saved',
            response: null
        });
    }
}

// Bulk import sales accounts
async function bulkImport(req, res) {
    try {
        // Validate required fields
        if (!req.file) {
            return res.status(400).json({
                status: 400,
                success: false,
                message: 'CSV file is required'
            });
        }

        if (!req.body.columnMappings) {
            return res.status(400).json({
                status: 400,
                success: false,
                message: 'Column mappings are required'
            });
        }

        let columnMappings;
        try {
            columnMappings = JSON.parse(req.body.columnMappings);
        } catch (error) {
            return res.status(400).json({
                status: 400,
                success: false,
                message: 'Invalid column mappings format'
            });
        }

        if (!Array.isArray(columnMappings)) {
            return res.status(400).json({
                status: 400,
                success: false,
                message: 'Column mappings must be an array'
            });
        }

        // Validate column mappings
        for (const mapping of columnMappings) {
            if (!mapping.csvColumn || !mapping.apiField) {
                return res.status(400).json({
                    status: 400,
                    success: false,
                    message: 'Each mapping must have csvColumn and apiField'
                });
            }
        }

        // Parse CSV file
        const csvData = [];
        const filePath = req.file.path;

        return new Promise((resolve, reject) => {
            fs.createReadStream(filePath)
                .pipe(csv())
                .on('data', (row) => {
                    csvData.push(row);
                })
                .on('end', async () => {
                    try {
                        // Clean up the uploaded file
                        fs.unlinkSync(filePath);

                        if (csvData.length === 0) {
                            return res.status(400).json({
                                status: 400,
                                success: false,
                                message: 'CSV file is empty or has no valid data'
                            });
                        }

                        // Process the import
                        const userId = req.user.id;
                        const results = await SalesAccount.bulkImport(csvData, columnMappings, userId);

                        res.json({
                            status: 200,
                            success: true,
                            message: 'Bulk import completed',
                            data: {
                                total: results.total,
                                successful: results.successful,
                                failed: results.failed,
                                errors: results.errors.slice(0, 10) 
                            },
                            errors: results.failed,
                            file: results.errorCsvFile
                        });

                        resolve();
                    } catch (error) {
                        // Clean up the uploaded file in case of error
                        if (fs.existsSync(filePath)) {
                            fs.unlinkSync(filePath);
                        }
                        reject(error);
                    }
                })
                .on('error', (error) => {
                    // Clean up the uploaded file in case of error
                    if (fs.existsSync(filePath)) {
                        fs.unlinkSync(filePath);
                    }
                    reject(error);
                });
        });

    } catch (error) {
        console.error('Error in bulk import:', error);
        res.status(500).json({
            status: 500,
            success: false,
            message: 'Error processing bulk import',
            error: error.message
        });
    }
}


module.exports = {
    create,
    getAll,
    getById,
    update,
    remove,
    getAccountMeetings,
    getAccountTasks,
    getAccountRelations,
    getAllFiltersList,
    createAccountFilterView,
    deleteAccountFilterView,
    getFieldsForFilter,
    getAccountListWithFilters,
    updateFilterView,
    updateSalesActivity,
    bulkImport,
    upload
}; 