const salesActivityModel = require('../models/salesActivity');
const salesActivityDataModel = require('../models/appointments');
const userModel = require('../models/user');
const SalesActivityData = require('../models/appointments');
const multer = require('multer');
const path = require('path');
const { v4: uuidv4 } = require('uuid');
const FileUpload = require('../models/fileUpload');

// Configure multer for file upload
const storage = multer.diskStorage({
    destination: function (req, file, cb) {
        cb(null, FileUpload.uploadDir);
    },
    filename: function (req, file, cb) {
        const uniqueFilename = `${uuidv4()}${path.extname(file.originalname)}`;
        cb(null, uniqueFilename);
    }
});

const upload = multer({
    storage: storage,
    limits: {
        fileSize: 5 * 1024 * 1024 // 5MB limit
    },
    fileFilter: (req, file, cb) => {
        if (file.fieldname === 'attachment' || file.fieldname === 'bill_attachment') {
            cb(null, true);
        } else {
            cb(new Error('Invalid fieldname. Only attachment and bill_attachment are allowed.'));
        }

    }

});


async function addSalesActivity(req, res) {
    try {
        const id = await salesActivityModel.createSalesActivity(req.body);
        res.status(201).json({ id });
    } catch (error) {
        res.status(500).json({ error: error.message });
    }
}

async function listSalesActivities(req, res) {
    try {
        const salesActivities = await salesActivityModel.getSalesActivities();
        res.status(200).json({
            status: 200,
            success: true,
            salesActivities
        });
    } catch (error) {
        res.status(500).json({ 
            status: 500,
            success: false,
            error: error.message 
        });
    }
}

async function editSalesActivity(req, res) {
    try {
        await salesActivityModel.updateSalesActivity(req.params.id, req.body);
        res.status(200).json({ 
            status: 200,
            success: true,
            message: 'Sales activity updated successfully' 
        });
    } catch (error) {
        res.status(500).json({ 
            status: 500,
            success: false,
            message: 'Error updating sales activity',
            error: error.message 
        });
    }
}

async function deleteSalesActivity(req, res) {
    try {
        await salesActivityModel.softDeleteSalesActivity(req.params.id);

        res.status(200).json({ 
            status: 200,
            success: true,
            message: 'Sales activity deleted successfully' 
        });
    } catch (error) {
        res.status(500).json({ 
            status: 500,
            success: false,
            message: 'Error deleting sales activity',
            error: error.message 
        });
    }
}

// create attendance for all users
async function createAttendanceForAllUsers(req, res) {
    try {
        const users = await userModel.getActiveUsers();
        console.log("date", new Date().toISOString().split('T')[0]);

        const start_date = new Date().toISOString().split('T')[0];
        const end_date = start_date;
        const start_time = "09:00";
        const end_time = "18:00";
  
        for (const user of users) {
            await salesActivityDataModel.createSalesActivityData({
                salesactivities_id:10,
                activity_type: 'Attendance',
                activity_date: start_date,
                activity_title: `Attendance for ${start_date} for ${user.name}`,
                start_date: start_date,
                end_date: end_date,
                start_time: start_time,
                end_time: end_time,
                owner_id: user.id,
                creater_id: 141,
                status: 'Pending'
            });
        }

        res.status(201).json({ 
            status: 201,
            success: true,
            message: 'Attendance sales activities created for all active users' });
    } catch (error) {
        res.status(500).json({ 
            status: 500,
            success: false,
            message: 'Error creating attendance sales activities',
            error: error.message 
        });
    }
}

// get all sales activities
async function getSalesActivities(req, res) 
 {
    try {
        const { date } = req.body;
        const userId = req.user.id; // Assuming you have authentication middleware
        console.log("datecontroller", date);

        // Validate date
        if (!date) {
            return res.status(400).json({
                status: false,
                message: 'Date field is required'
            });
        }

        // Validate date format
        if (!/^\d{4}-\d{2}-\d{2}$/.test(date)) {
            return res.status(400).json({
                status: false,
                message: 'Invalid date format. Please use YYYY-MM-DD'
            });
        }

        const data = await SalesActivityData.getSalesActivitiesByDate(date, userId);

        if (!data.records.length && !data.salesactivities.length && !data.customactivities.length) {
            return res.status(200).json({
                status: false,
                message: 'Activities info not exists',
                response: null
            });
        }

        return res.status(200).json({
            status: 200,
            success: true,
            message: 'Activities info',
            response: data
        });

    } catch (error) {
        console.error('Error in getSalesActivities:', error);
        return res.status(500).json({
            status: 500,
            success: false,
            message: 'Internal server error',
            error: error.message
        });
    }
};

async function createSalesActivityData(req, res) {
    try {
        // const data = req.body;
        const id = await salesActivityModel.createSalesActivityData(req.body);
        if(id?.status == 400){
            return res.status(400).json(id);
        }
        res.status(201).json({ 
            status: 201, 
            success: true,
            message: 'Sales activity data Created successfully',
            data: { id }
        });

    } catch (error) {
        console.error('Error creating sales activity:', error);
        res.status(500).json({ 
            status: 500,
            success: false,
            message: 'Error creating sales activity',
            error: error.message 
        });
    }
}
async function getAllSalesActivityData(req, res) {
    try {
        const { id } = req.user;
        const response = await salesActivityModel.getAllSalesActivityData(id);
        // if(response?.status == 400){
        //     return res.status(400).json({
        //         status: 400,
        //         success: false,
        //         message: 'Error fetching sales activities',
        //         error: response.message 
        //     });
        // }
        res.json({ 
            status: 200, 
            success: true,
            message: 'Sales activities fetched successfully', 
            response
        });
    } catch (error) {
        res.status(500).json({ 
            status: 500,
            success: false,
            message: 'Error fetching sales activities',
            error: error.message 
        });
    }
}
async function getSalesActivityData(req, res) {
    try {
        const { id } = req.params;
        const userId = req.user.id;
        console.log(id,userId);
        const response = await salesActivityModel.getSalesActivityData(id, userId);
        if (!response) {
            return res.status(404).json({ 
                status: 404, 
                success: false,
                message: 'Sales activity data not found' });
        }
        res.json({ 
            status: 200, 
            success: true,
            message: 'Sales activity fetched successfully', 
            response
        });
    } catch (error) {
        res.status(500).json({ 
            status: 500,
            success: false,
            message: 'Error fetching sales activity',
            error: error.message 
        });         
    }
}


// get all meetings
async function getAllMeetings(req, res) {
    try {
        const { start_date, end_date, status, user } = req.query;
        console.log("start_date", start_date, "end_date", end_date, "status", status, "user", user);
        console.log("user", req.query);
        const user_id = req.user.id;

        // Get user's role
        const user_data = await userModel.findUserById(user_id);
        const role_id = user_data.roles_id;

        // Validate date format if dates are provided
        if ((start_date || end_date) && !(start_date && end_date)) {
            return res.status(400).json({
                status: 400,
                success: false,
                message: "Both start_date and end_date are required for date filtering"
            });
        }

        if (start_date && end_date) {
            // Validate date format (YYYY-MM-DD)
            const dateRegex = /^\d{4}-\d{2}-\d{2}$/;
            if (!dateRegex.test(start_date) || !dateRegex.test(end_date)) {
                return res.status(400).json({
                    status: 400,
                    success: false,
                    message: "Invalid date format. Use YYYY-MM-DD"
                });
            }

            // Validate that end_date is not before start_date
            if (new Date(end_date) < new Date(start_date)) {
                return res.status(400).json({
                    status: 400,
                    success: false,
                    message: "End date cannot be before start date"
                });
            }
        }

        // Validate status if provided
        const validStatuses = ['Pending', 'Inprogress', 'Closed', 'Cancelled'];
        if (status && !validStatuses.includes(status)) {
            return res.status(400).json({
                status: 400,
                success: false,
                message: "Invalid status. Valid statuses are: " + validStatuses.join(', ')
            });
        }

        const response = await salesActivityModel.getAllMeetings(start_date, end_date, status, role_id, user_id, user);

        res.json({ 
            status: 200, 
            success: true, 
            message: 'Meetings fetched successfully', 
            total: response.length,
            data: response,
            filters: {
                start_date: start_date || null,
                end_date: end_date || null,
                status: status || null,
                role_id: role_id,
                territory_filtered: role_id === 2
            },
            count: response.length
        });
    } catch (error) {
        res.status(500).json({ 
            status: 500, 
            success: false, 
            message: 'Error fetching meetings', 
            error: error.message 
        });
    }
}

async function updateSalesActivityData(req, res) {
    try {
        const { id } = req.params;
        const data = req.body;
        console.log({data})
        const result = await salesActivityModel.updateSalesActivityData(data, id);
        if (!result) {
            return res.status(404).json({ 
                status: 404, 
                success: false,
                message: 'Sales activity not found' 
            });
            
        }
        if(result.status == 400){
            return res.status(400).json({
                status: 400,
                success: false,
                message: 'Error updating sales activity',
                error: result.message 
            });
        }
        res.status(200).json({
            status: 200,
            success: true,
            message: 'Sales activity updated successfully',
            data: { id: id }
        });
    } catch (error) {
        res.status(500).json({ 
            status: 500,
            success: false,
            message: 'Error updating sales activity',
            error: error.message 
        }); 
    }
}
async function updateActivityStatus(req, res) {
    try {
        const { id } = req.params;
        console.log(req.body)
        const { latitude, longitude, location ,remarks, transport_mode, amount} = req.body;
        const userId = req.user.id; // Get logged in user's ID from auth middleware

        // Validate required fields
        if (!latitude || !longitude || !location) {
            return res.status(400).json({
                status: 400,
                success: false,
                message: 'Latitude, longitude and location are required'
            });
        }

        const result = await salesActivityModel.updateActivityStatus(id, {
            latitude,
            longitude,
            location,
            attachment: req.files?.attachment?.[0]?.filename || null,
            bill_attachment: req.files?.bill_attachment?.[0]?.filename || null,
            transport_mode,
            amount,
            remarks

        }, userId);
	 return res.status(200).json({
                status: 200,
                success: true,
                message: `Activity ${result.newStatus === 'Inprogress' ? 'checked-in' : 'checked-out'} successfully`,
            });
    } catch (error) {
        console.error('Error updating activity status:', error);
        res.status(error.message === 'Activity not found' ? 404 : 500).json({
            status: error.message === 'Activity not found' ? 404 : 500,
            success: false,
            message: error.message
        });
    }
}
async function getAttendanceHistory(req, res) {
    try {
        const userId = req.user.id;
        const { filter = 'today', month, year } = req.query;

        // Validate monthly filter parameters
        if (filter === 'monthly' && (!month || !year)) {
            return res.status(400).json({
                status: 400,
                success: false,
                message: 'Month and year are required for monthly filter'
            });
        }

        const attendanceHistory = await salesActivityModel.getAttendanceHistory(
            userId,
            filter,
            month,
            year
        );

        res.status(200).json({
            status: 200,
            success: true,
            message: 'Attendance history retrieved successfully',
            data: attendanceHistory
        });

    } catch (error) {
        console.error('Error fetching attendance history:', error);
        res.status(500).json({
            status: 500,
            success: false,
            message: error.message || 'Error retrieving attendance history'
        });
    }
}


// Attendance history by all users for admin 
async function getAllAttendanceHistory(req, res) {
    try {
        const { filter = 'today', month, year, start_date, end_date, user } = req.query;



        // If start_date and end_date are provided, use date range filter
        if (start_date && end_date) {
            // Validate date format
            const dateRegex = /^\d{4}-\d{2}-\d{2}$/;
            if (!dateRegex.test(start_date) || !dateRegex.test(end_date)) {
                return res.status(400).json({
                    status: 400,
                    success: false,
                    message: 'Invalid date format. Please use YYYY-MM-DD format'
                });
            }

            // Validate that end_date is not before start_date
            if (new Date(end_date) < new Date(start_date)) {
                return res.status(400).json({
                    status: 400,
                    success: false,
                    message: 'End date cannot be before start date'
                });
            }

            const attendanceHistory = await salesActivityModel.getAllAttendanceHistory(
                'date_range',
                null,
                null,
                start_date,
                end_date,
                user
            );

            return res.status(200).json({
                status: 200,
                success: 'true',
                message: 'Attendance history retrieved successfully',
                data: attendanceHistory
            });
        }

        // Existing filter logic
        if (filter === 'monthly' && (!month || !year)) {
            return res.status(400).json({
                status: 400,
                success: false,
                message: 'Month and year are required for monthly filter'
            });
        }

        const attendanceHistory = await salesActivityModel.getAllAttendanceHistory(
            filter,
            month,
            year
        );

        res.status(200).json({
            status: 200,
            success: true,
            message: 'Attendance history retrieved successfully',
            data: attendanceHistory
        });
    } catch (error) {
        console.error('Error fetching attendance history:', error);
        res.status(500).json({
            status: 500,
            success: false,
            message: error.message || 'Error retrieving attendance history'
        });
    }
}




async function getConveyanceList(req, res) {
    try {
        const userId = req.user.id;
        const { filter = 'today', month, year } = req.body;

        // Validate monthly filter parameters
        if (filter === 'monthly' && (!month || !year)) {
            return res.status(400).json({
                status: 400,
                success: false,
                message: 'Month and year are required for monthly filter'
            });
        }

        const conveyanceList = await salesActivityModel.getConveyanceList(
            userId,
            filter,
            month,
            year
        );

        res.json({
            status: 200,
            success: true,
            message: 'Conveyance list retrieved successfully',
            data: conveyanceList
        });

    } catch (error) {
        console.error('Error fetching conveyance list:', error);
        res.status(500).json({
            status: 500,
            success: false,
            message: error.message || 'Error retrieving conveyance list'
        });
    }
}

// all conveyance list
async function getconveyance(req, res) {
    try {
        const { start_date, end_date, user } = req.query;
        const role = await userModel.findUserById(req.user.id);
        const role_id = role.roles_id;
        const user_id = req.user.id;        
        // Validate date format if provided
        if ((start_date || end_date) && !(start_date && end_date)) {
            return res.status(400).json({
                status: 400,
                success: false,
                message: "Both start_date and end_date are required for date filtering"
            });
        }

        if (start_date && end_date) {
            // Validate date format (YYYY-MM-DD)
            const dateRegex = /^\d{4}-\d{2}-\d{2}$/;
            if (!dateRegex.test(start_date) || !dateRegex.test(end_date)) {
                return res.status(400).json({
                    status: 400,
                    success: false,
                    message: "Invalid date format. Use YYYY-MM-DD"
                });
            }

            // Validate that end_date is not before start_date
            if (new Date(end_date) < new Date(start_date)) {
                return res.status(400).json({
                    status: 400,
                    success: false,
                    message: "End date cannot be before start date"
                });
            }
        }

        const result = await salesActivityModel.getconveyance(start_date, end_date, user_id, role_id, user);
        
        res.status(200).json({
            status: 200,
            success: true,
            message: "Conveyance data fetched successfully",
            counts: result.records.length,
            data: result.records,
            summary: {
                total_distance: result.totals.total_distance + " km",
                total_hours: result.totals.total_hours + " hours"
            },
            filters: {
                start_date: start_date || null,
                end_date: end_date || null
            }
        });
    } catch (error) {
        console.error('Error fetching conveyance data:', error);
        res.status(500).json({
            status: 500,
            success: false,
            message: error.message || 'Error fetching conveyance data'
        });
    }
}


async function deleteSalesActivityData(req, res) {
    try {
        const success=await salesActivityModel.softDeleteSalesActivityData(req.params.id);
         if (!success) {
            return res.status(404).json({ 
                status: 404,
                success: false,
                message: 'Meeting not found' });
        }
        res.json({ 
            status: 200,
            success: true,
            message: 'Meeting deleted successfully' });
    
    } catch (error) {
        res.status(500).json({ 
            status: 500,
            success: false,
            error: error.message 
        });
    }
}

// claims
async function getallclaims(req, res) {
    try {
        const { start_date, end_date, user_id, territory } = req.query;
        const role = await userModel.findUserById(req.user.id);
        const role_id = role.roles_id; 
        console.log(role_id);
        // Validate date format if provided
        if ((start_date || end_date) && !(start_date && end_date)) {
            return res.status(400).json({
                status: 400,
                success: false,
                message: "Both start_date and end_date are required for date filtering"
            });
        }

        if (start_date && end_date) {
            // Validate date format (YYYY-MM-DD)
            const dateRegex = /^\d{4}-\d{2}-\d{2}$/;
            if (!dateRegex.test(start_date) || !dateRegex.test(end_date)) {
                return res.status(400).json({
                    status: 400,
                    success: false,
                    message: "Invalid date format. Use YYYY-MM-DD"
                });
            }

            // Validate that end_date is not before start_date
            if (new Date(end_date) < new Date(start_date)) {
                return res.status(400).json({
                    status: 400,
                    success: false,
                    message: "End date cannot be before start date"
                });
            }
        }

        const changes = await salesActivityModel.getallclaims(start_date, end_date, user_id, role_id, territory);
        
        res.status(200).json({
            status: 200,
            success: true,
            message: "Claims data fetched successfully",
            counts: changes.length,
            data: changes,
            filters: {
                start_date: start_date || null,
                end_date: end_date || null
            }
        });
    } catch (error) {
        console.error('Error fetching claims data:', error);
        res.status(500).json({
            status: 500,
            success: false,
            message: error.message || 'Error fetching claims data'
        });
    }
}


// update Claims
async function updateclaim(req, res) {
    try {
        const id = req.params.id;
        // console.log(req.body, id);
        const status = req.body.status;
        const remarks = req.body.reason;
        // console.log(status);
        // console.log(id);
        const result = await salesActivityModel.updateclaim(id, status, remarks);
        return res.status(200).json({
            status: 200,
            success: true,
            message: 'Claim updated successfully',
        });
    }catch (error) {    
        console.error('Error updating claim:', error);
        return res.status(500).json({
            status: 500,
            success: false,
            message: 'Error updating claim',
            error: error.message
        });
    }
}

async function updateBulkClaims(req, res) {
    try {
        const { claimIds, status, remarks } = req.body;

        console.log(claimIds, status, remarks);

        // Validate required fields
        if (!claimIds || !Array.isArray(claimIds) || claimIds.length === 0) {
            return res.status(400).json({
                status: 400,
                success: false,
                message: 'Claim IDs array is required and cannot be empty'
            });
        }

        if (!status) {
            return res.status(400).json({
                status: 400,
                success: false,
                message: 'Status is required'
            });
        }

        // Remarks can be optional, so we don't validate it as required
        // If remarks is undefined, it will be converted to null in the model

        // Validate that all claimIds are numbers
        if (!claimIds.every(id => Number.isInteger(Number(id)) && Number(id) > 0)) {
            return res.status(400).json({
                status: 400,
                success: false,
                message: 'All claim IDs must be valid positive integers'
            });
        }

        const result = await salesActivityModel.updateBulkClaims(claimIds, status, remarks);
        
        return res.status(200).json({
            status: 200,
            success: true,
            message: `Bulk claim update completed. ${result.updatedClaims} out of ${result.totalClaims} claims updated successfully.`,
            data: result
        });
    } catch (error) {
        console.error('Error updating bulk claims:', error);
        return res.status(500).json({
            status: 500,
            success: false,
            message: 'Error updating bulk claims',
            error: error.message
        });
    }
}

// convayance changes
async function getConvayanceChanges(req, res) {
    try {
        const changes = await adminModel.getConvayanceChanges();
        res.status(200).json({
            status: 200,
            success: true,
            message: "Convayance changes fetched successfully",
            data: changes
        });
    } catch (error) {
        console.error('Error fetching convayance changes:', error);
        res.status(500).json({
            status: 500,
            success: false,
            message: error.message || 'Error fetching convayance changes'
        });
    }
}

// Update convayance changes
async function updateConvayanceChanges(req, res) {
    try {
        const id = req.params.id;
        const conveyance_charge = req.body.conveyance_charge;  
        const result = await adminModel.updateConvayanceChanges(id, conveyance_charge);
        res.status(200).json({
            status: 200,
            success: true,
            message: "Convayance changes updated successfully",
            // data: result
        });
    } catch (error) {
        console.error('Error updating convayance changes:', error);
        res.status(500).json({
             status:500,
             success: false,
             message: error.message || 'Error updating convayance changes'
        })
    }
}

// get all sales activity with data
async function getAllSalesActivitywithData(req, res) {  
    try {
        const { start_date, end_date } = req.query;
        
        // Validate date format if dates are provided
        if (start_date && !/^\d{4}-\d{2}-\d{2}$/.test(start_date)) {
            return res.status(400).json({
                status: 400,
                success: false,
                message: 'Invalid start date format. Please use YYYY-MM-DD format'
            });
        }
        
        if (end_date && !/^\d{4}-\d{2}-\d{2}$/.test(end_date)) {
            return res.status(400).json({
                status: 400,
                success: false,
                message: 'Invalid end date format. Please use YYYY-MM-DD format'
            });
        }

        // Validate that dates are valid
        if (start_date && isNaN(new Date(start_date).getTime())) {
            return res.status(400).json({
                status: 400,
                success: false,
                message: 'Invalid start date'
            });
        }

        if (end_date && isNaN(new Date(end_date).getTime())) {
            return res.status(400).json({
                status: 400,
                success: false,
                message: 'Invalid end date'
            });
        }

        const result = await salesActivityModel.getAllSalesActivityWithData(start_date, end_date);
        res.status(200).json({ 
            status: 200, 
            success: true, 
            message: 'Sales activities with data fetched successfully', 
            response: result 
        });
    } catch (error) {
        console.error('Error fetching sales activities with data:', error);
        res.status(500).json({
            status: 500,
            success: false,
            message: error.message || 'Error fetching sales activities with data'
        });
    }
}


// get all meetings by id
async function getAllMeetingsById(req, res) {
    try {
        console.log('Controller: Fetching meeting with ID:', req.params.id);
        const result = await salesActivityModel.getAllMeetingsById(req.params.id);
        // console.log('Controller: Model result:', result);

        if (!result) {
            console.log('Controller: No meeting found');
            return res.status(404).json({
                status: 404,
                success: false,
                message: "Meeting not found"
            });
        }

        console.log('Controller: Sending successful response');
        return res.status(200).json({
            status: 200,
            success: true,
            message: "Meeting fetched successfully",
            response: result
        });
    } catch (error) {
        console.error('Controller error in getAllMeetingsById:', error);
        return res.status(500).json({
            status: 500,
            success: false,
            message: error.message || "Error fetching meeting details"
        });
    }
}
async function updateActivityWithCustomFields(req, res) {
    try {
        const { id } = req.params;
        const data = req.body;

        // Validate input
        if (!id) {
            return res.status(400).json({
                status: 400,
                success: false,
                message: 'Activity ID is required'
            });
        }

        const updatedActivity = await salesActivityModel.updateSalesActivityDataWithCustomFields(id, data);

        res.json({
            status: 200,
            success: true,
            message: 'Activity updated successfully',
            data: updatedActivity
        });

    } catch (error) {
        console.error('Update activity error:', error);
        res.status(500).json({
            status: 500,
            success: false,
            message: 'Error updating activity',
            error: error.message
        });
    }
}

async function getActivitiesByModule(req, res) {
    try {
        const id = req.body.id;
        const module = req.body.module;
        const activities = await salesActivityModel.getActivitiesByModule(id, module);
        res.json({
            status: 200,
            message: 'Activities fetched successfully',
            data: activities
        });
    } catch (error) {
        console.error('Error fetching activities by module:', error);
        res.status(500).json({
            status: 500,
            message: 'Error fetching activities by module',
            error: error.message
        });
    }
}

async function soft_checkout(req, res) {
    const data = req.body
    console.log("data",data);
    const user_id = req.user.id
    // console.log("user",user_id);
    // const id = req.params.id
    const [meeting] = await salesActivityModel.getinprogress(user_id);
    // console.log("meeting",meeting);
    if (!meeting) {
        return res.status(400).json({
            status: 400,
            success: false,
            message: "Meeting not found"
        });
    }
    const id = meeting.id;
    console.log("id",id);

    console.log(req.params);
    try {
        const result = await salesActivityModel.softCheckout(id, data, true);
        if(!result){
            res.json({
                status: 200,
                success: true,
                message: "User in lease than 2 KM."
            })
        }
        res.status(200).json({
            status: 200,
            success: true,
            message: "User cross the 2 KM.",
            // data: result
        });
    } catch (error) {
        console.error('Error updating soft checkout:', error);
        res.status(500).json({
            status: 500,
            map: "hello",
            success: false,
            message: error.message || 'Error updating soft checkout'
        })
    }
}

module.exports = {
    addSalesActivity,
    listSalesActivities,
    editSalesActivity,
    deleteSalesActivity,
    createAttendanceForAllUsers,
    getSalesActivities,
    createSalesActivityData,
    getAllSalesActivityData,
    getSalesActivityData,
    updateSalesActivityData,
    updateActivityStatus,
    getAttendanceHistory,
    getAllAttendanceHistory,
    getConveyanceList,
    deleteSalesActivityData,
    upload,
    getActivitiesByModule,
    getconveyance,
    getallclaims,
    updateclaim,
    updateBulkClaims,
    getConvayanceChanges,
    updateConvayanceChanges,
    getAllSalesActivitywithData,
    getAllMeetings,
    getAllMeetingsById,
    soft_checkout,
    updateActivityWithCustomFields
};