const salesActivityModel = require('../models/salesActivity');
const salesActivityDataModel = require('../models/appointments');
const userModel = require('../models/user');
const SalesActivityData = require('../models/appointments');
const multer = require('multer');
const path = require('path');
const { v4: uuidv4 } = require('uuid');
const FileUpload = require('../models/fileUpload');
const fs = require('fs');

// Configure multer for file upload
const storage = multer.diskStorage({
    destination: function (req, file, cb) {
        cb(null, FileUpload.uploadDir);
    },
    filename: function (req, file, cb) {
        const uniqueFilename = `${uuidv4()}${path.extname(file.originalname)}`;
        cb(null, uniqueFilename);
    }
});

const upload = multer({
    storage: storage,
    limits: {
        fileSize: 5 * 1024 * 1024 // 5MB limit
    },
    fileFilter: (req, file, cb) => {
        // Add file type validation if needed
        cb(null, true);
    }

});


async function addSalesActivity(req, res) {
    try {
        const id = await salesActivityModel.createSalesActivity(req.body);
        res.status(201).json({ id });
    } catch (error) {
        res.status(500).json({ error: error.message });
    }
}

async function listSalesActivities(req, res) {
    try {
        const salesActivities = await salesActivityModel.getSalesActivities();
        res.status(200).json(salesActivities);
    } catch (error) {
        res.status(500).json({ error: error.message });
    }
}

async function editSalesActivity(req, res) {
    try {
        await salesActivityModel.updateSalesActivity(req.params.id, req.body);
        res.status(200).json({ message: 'Sales activity updated successfully' });
    } catch (error) {
        res.status(500).json({ error: error.message });
    }
}

async function deleteSalesActivity(req, res) {
    try {
        await salesActivityModel.softDeleteSalesActivity(req.params.id);
        res.status(200).json({ message: 'Sales activity deleted successfully' });
    } catch (error) {
        res.status(500).json({ error: error.message });
    }
}

async function createAttendanceForAllUsers(req, res) {
    try {
        const users = await userModel.getActiveUsers();

        const start_date = new Date().toISOString().split('T')[0];
        const end_date = start_date;
        const start_time = "09:00";
        const end_time = "18:00";

        for (const user of users) {
            await salesActivityDataModel.createSalesActivityData({
                salesactivities_id:10,
                activity_type: 'Attendance',
                activity_date: start_date,
                activity_title: `Attendance for ${start_date} for ${user.name}`,
                start_date: start_date,
                end_date: end_date,
                start_time: start_time,
                end_time: end_time,
                owner_id: user.id,
                creater_id: 141,
                status: 'Pending'
            });
        }

        res.status(201).json({ message: 'Attendance sales activities created for all active users' });
    } catch (error) {
        res.status(500).json({ error: error.message });
    }
}
async function getSalesActivities(req, res) 
 {
    try {
        const { date } = req.body;
        const userId = req.user.id; // Assuming you have authentication middleware

        // Validate date
        if (!date) {
            return res.status(400).json({
                status: false,
                message: 'Date field is required'
            });
        }

        // Validate date format
        if (!/^\d{4}-\d{2}-\d{2}$/.test(date)) {
            return res.status(400).json({
                status: false,
                message: 'Invalid date format. Please use YYYY-MM-DD'
            });
        }

        const data = await SalesActivityData.getSalesActivitiesByDate(date, userId);

        if (!data.records.length && !data.salesactivities.length && !data.customactivities.length) {
            return res.status(200).json({
                status: false,
                message: 'Activities info not exists',
                response: null
            });
        }

        return res.status(200).json({
            status: 200,
            success: true,
            message: 'Activities info',
            response: data
        });

    } catch (error) {
        console.error('Error in getSalesActivities:', error);
        return res.status(500).json({
            status: 500,
            success: false,
            message: 'Internal server error',
            error: error.message
        });
    }
};

async function createSalesActivityData(req, res) {
    try {
        const data = req.body;
        const id = await salesActivityModel.createSalesActivityData(data);
        res.status(201).json({ status: 201, message: 'Sales activity data Created successfully' , data : {id} });
    } catch (error) {
        res.status(500).json({ error: error.message });
    }
}
async function getAllSalesActivityData(req, res) {
    try {
        const { id } = req.user;
        const response = await salesActivityModel.getAllSalesActivityData(id);
        res.json({ status: 200, message: 'Sales activities fetched successfully', response});
    } catch (error) {
        res.status(500).json({ error: error.message });
    }
}
async function getSalesActivityData(req, res) {
    try {
        const { id } = req.params;
        const response = await salesActivityModel.getSalesActivityData(id);
        if (!response) {
            return res.status(404).json({ status: 404, message: 'Sales activity data not found' });
        }
        res.json({ status: 200, message: 'Sales activity fetched successfully', response});
    } catch (error) {
        res.status(500).json({ error: error.message });
    }
}


async function updateSalesActivityData(req, res) {
    try {
        const { id } = req.params;
        const data = req.body;
        const result = await salesActivityModel.updateSalesActivityData(data, id);
        if (!result) {
            return res.status(404).json({ status: 404, message: 'Sales activity not found' });
        }
	res.status(200).json({
            status: 200,
	    message: 'Sales activity updated successfully',
            data: { id: id }
        });
	
        
    } catch (error) {
        res.status(500).json({ error: error.message });
    }
}
async function updateActivityStatus(req, res) {
    try {
        const { id } = req.params;
        const { latitude, longitude, location, remarks, transport_mode, amount } = req.body;
        const userId = req.user.id; // Get logged in user's ID from auth middleware

        // Validate required fields
        if (!latitude || !longitude || !location) {
            return res.status(400).json({
                status: false,
                message: 'Latitude, longitude and location are required'
            });
        }

        // Get file names from uploaded files
        const attachments = req.files.attachment ? req.files.attachment.map(file => file.filename) : [];
        const billAttachments = req.files.bill_attachment ? req.files.bill_attachment.map(file => file.filename) : [];

        console.log("Attachments:", attachments);
        console.log("Bill Attachments:", billAttachments);

        const result = await salesActivityModel.updateActivityStatus(id, {
            latitude,
            longitude,
            location,
            attachment: attachments.length > 0 ? attachments.join(',') : null,
            bill_attachment: billAttachments.length > 0 ? billAttachments.join(',') : null,
            remarks,
            transport_mode,
            amount
        }, userId);

        return res.status(200).json({
            status: 200,
            message: `Activity ${result.newStatus === 'Inprogress' ? 'checked-in' : 'checked-out'} successfully`,
        });

    } catch (error) {
        console.error('Error updating activity status:', error);
        res.status(error.message === 'Activity not found' ? 404 : 500).json({
            success: false,
            message: error.message
        });
    }
}

const updateStatus = async (req, res) => {
    try {
        const { id } = req.params;
        const billFile = req.files && req.files.bill_attachment;

        // Prepare update data
        const updateData = {
            ...req.body,
            latitude: req.body.latitude,
            longitude: req.body.longitude,
            location: req.body.location,
            transport_mode: req.body.transport_mode,
            amount: req.body.amount,
            remarks: req.body.remarks
        };

        // Handle bill attachment for Public Transport
        if (updateData.transport_mode === 'Public Transport') {
            if (!billFile) {
                return res.status(400).json({
                    status: false,
                    message: 'Bill attachment is required for Public Transport'
                });
            }

            // Save bill attachment
            const uploadDir = path.join(process.cwd(), 'public', 'uploads', 'bills');
            if (!fs.existsSync(uploadDir)) {
                fs.mkdirSync(uploadDir, { recursive: true });
            }

            const billFileName = `bill-${Date.now()}-${billFile.name}`;
            await billFile.mv(path.join(uploadDir, billFileName));
            updateData.claim_image = `/uploads/bills/${billFileName}`;
        }

        const result = await salesActivityModel.updateActivityStatus(
            id,
            updateData,
            req.user.id
        );

        res.json({
            status: true,
            message: 'Status updated successfully',
            data: result
        });

    } catch (error) {
        console.error('Error in updateStatus:', error);
        res.status(500).json({
            status: false,
            message: error.message
        });
    }
};

async function getAttendanceHistory(req, res) {
    try {
        const userId = req.user.id;
        const { filter = 'today', month, year } = req.query;

        // Validate monthly filter parameters
        if (filter === 'monthly' && (!month || !year)) {
            return res.status(400).json({
                status: false,
                message: 'Month and year are required for monthly filter'
            });
        }

        const attendanceHistory = await salesActivityModel.getAttendanceHistory(
            userId,
            filter,
            month,
            year
        );

        res.status(200).json({
            status: 200,
            success: true,
            message: 'Attendance history retrieved successfully',
            data: attendanceHistory
        });

    } catch (error) {
        console.error('Error fetching attendance history:', error);
        res.status(500).json({
            status: 500,
            success: false,
            message: error.message || 'Error retrieving attendance history'
        });
    }
}


// Attendance history by all users for admin 
async function getAllAttendanceHistory(req, res) {
    try {
        const { filter = 'today', month, year, start_date, end_date } = req.query;

        // If start_date and end_date are provided, use date range filter
        if (start_date && end_date) {
            // Validate date format
            const dateRegex = /^\d{4}-\d{2}-\d{2}$/;
            if (!dateRegex.test(start_date) || !dateRegex.test(end_date)) {
                return res.status(400).json({
                    status: false,
                    message: 'Invalid date format. Please use YYYY-MM-DD format'
                });
            }

            // Validate that end_date is not before start_date
            if (new Date(end_date) < new Date(start_date)) {
                return res.status(400).json({
                    status: false,
                    message: 'End date cannot be before start date'
                });
            }

            const attendanceHistory = await salesActivityModel.getAllAttendanceHistory(
                'date_range',
                null,
                null,
                start_date,
                end_date
            );

            return res.status(200).json({
                status: 200,
                success: 'true',
                message: 'Attendance history retrieved successfully',
                data: attendanceHistory
            });
        }

        // Existing filter logic
        if (filter === 'monthly' && (!month || !year)) {
            return res.status(400).json({
                status: false,
                message: 'Month and year are required for monthly filter'
            });
        }

        const attendanceHistory = await salesActivityModel.getAllAttendanceHistory(
            filter,
            month,
            year
        );

        res.status(200).json({
            status: 200,
            success: true,
            message: 'Attendance history retrieved successfully',
            data: attendanceHistory
        });
    } catch (error) {
        console.error('Error fetching attendance history:', error);
        res.status(500).json({
            status: 500,
            success: false,
            message: error.message || 'Error retrieving attendance history'
        });
    }
}




async function getConveyanceList(req, res) {
    try {
        const userId = req.user.id;
        const { filter = 'today', month, year } = req.body;

        // Validate monthly filter parameters
        if (filter === 'monthly' && (!month || !year)) {
            return res.status(400).json({
                status: false,
                message: 'Month and year are required for monthly filter'
            });
        }

        const conveyanceList = await salesActivityModel.getConveyanceList(
            userId,
            filter,
            month,
            year
        );

        res.json({
            status: 200,
            success: true,
            message: 'Conveyance list retrieved successfully',
            data: conveyanceList
        });

    } catch (error) {
        console.error('Error fetching conveyance list:', error);
        res.status(500).json({
            status: 500,
            success: false,
            message: error.message || 'Error retrieving conveyance list'
        });
    }
}

// all conveyance list
async function getconveyance(req, res) {
    try {
        const { start_date, end_date } = req.query;

        // Validate date format if provided
        if ((start_date || end_date) && !(start_date && end_date)) {
            return res.status(400).json({
                status: false,
                message: "Both start_date and end_date are required for date filtering"
            });
        }

        if (start_date && end_date) {
            // Validate date format (YYYY-MM-DD)
            const dateRegex = /^\d{4}-\d{2}-\d{2}$/;
            if (!dateRegex.test(start_date) || !dateRegex.test(end_date)) {
                return res.status(400).json({
                    status: false,
                    message: "Invalid date format. Use YYYY-MM-DD"
                });
            }

            // Validate that end_date is not before start_date
            if (new Date(end_date) < new Date(start_date)) {
                return res.status(400).json({
                    status: false,
                    message: "End date cannot be before start date"
                });
            }
        }

        const result = await salesActivityModel.getconveyance(start_date, end_date);
        
        res.status(200).json({
            status: 200,
            success: true,
            message: "Conveyance data fetched successfully",
            counts: result.records.length,
            data: result.records,
            summary: {
                total_distance: result.totals.total_distance + " km",
                total_hours: result.totals.total_hours + " hours"
            },
            filters: {
                start_date: start_date || null,
                end_date: end_date || null
            }
        });
    } catch (error) {
        console.error('Error fetching conveyance data:', error);
        res.status(500).json({
            status: 500,
            success: false,
            message: error.message || 'Error fetching conveyance data'
        });
    }
}


async function deleteSalesActivityData(req, res) {
    try {
        const success=await salesActivityModel.softDeleteSalesActivityData(req.params.id);
         if (!success) {
            return res.status(404).json({ status: 404, message: 'Meeting not found' });
        }
        res.json({ status: 200, message: 'Meeting deleted successfully' });
    
    } catch (error) {
        res.status(500).json({ error: error.message });
    }
}

// claims
async function getallclaims(req, res) {
    try {
        const { start_date, end_date } = req.query;

        // Validate date format if provided
        if ((start_date || end_date) && !(start_date && end_date)) {
            return res.status(400).json({
                status: false,
                message: "Both start_date and end_date are required for date filtering"
            });
        }

        if (start_date && end_date) {
            // Validate date format (YYYY-MM-DD)
            const dateRegex = /^\d{4}-\d{2}-\d{2}$/;
            if (!dateRegex.test(start_date) || !dateRegex.test(end_date)) {
                return res.status(400).json({
                    status: false,
                    message: "Invalid date format. Use YYYY-MM-DD"
                });
            }

            // Validate that end_date is not before start_date
            if (new Date(end_date) < new Date(start_date)) {
                return res.status(400).json({
                    status: false,
                    message: "End date cannot be before start date"
                });
            }
        }

        const changes = await salesActivityModel.getallclaims(start_date, end_date);
        
        res.status(200).json({
            status: 200,
            success: true,
            message: "Claims data fetched successfully",
            counts: changes.length,
            data: changes,
            filters: {
                start_date: start_date || null,
                end_date: end_date || null
            }
        });
    } catch (error) {
        console.error('Error fetching claims data:', error);
        res.status(500).json({
            status: 500,
            success: false,
            message: error.message || 'Error fetching claims data'
        });
    }
}

// convayance changes
async function getConvayanceChanges(req, res) {
    try {
        const changes = await adminModel.getConvayanceChanges();
        res.status(200).json({
            status: 200,
            success: true,
            message: "Convayance changes fetched successfully",
            data: changes
        });
    } catch (error) {
        console.error('Error fetching convayance changes:', error);
        res.status(500).json({
            status: false,
            message: error.message || 'Error fetching convayance changes'
        });
    }
}

// Update convayance changes
async function updateConvayanceChanges(req, res) {
    try {
        const id = req.params.id;
        const conveyance_charge = req.body.conveyance_charge;  
        const result = await adminModel.updateConvayanceChanges(id, conveyance_charge);
        res.status(200).json({
            status: 200,
            success: true,
            message: "Convayance changes updated successfully",
            // data: result
        });
    } catch (error) {
        console.error('Error updating convayance changes:', error);
        res.status(500).json({
             status:500,
             success: false,
             message: error.message || 'Error updating convayance changes'
        })
    }
}

// get all sales activity with data
async function getAllSalesActivitywithData(req, res) {  
    try {
        const result = await salesActivityModel.getAllSalesActivityWithData();
        res.status(200).json({ 
            status: 200, 
            success: true, 
            message: 'Sales activities with data fetched successfully', 
            response: result 
        }
    );
    } catch (error) {
        console.error('Error fetching sales activities with data:', error);
        res.status(500).json({
            status: 500,
            success: false,
            message: error.message || 'Error fetching sales activities with data'
        });
    }
}




module.exports = {
    addSalesActivity,
    listSalesActivities,
    editSalesActivity,
    deleteSalesActivity,
    createAttendanceForAllUsers,
    getSalesActivities,
    createSalesActivityData,
    getAllSalesActivityData,
    getSalesActivityData,
    updateSalesActivityData,
    updateActivityStatus,
    updateStatus,
    getAttendanceHistory,
    getAllAttendanceHistory,
    getConveyanceList,
    deleteSalesActivityData,
    upload,
    getconveyance,
    getallclaims,
    getConvayanceChanges,
    updateConvayanceChanges,
    getAllSalesActivitywithData
};