const task = require('../models/task');
const multer = require('multer');
const csv = require('csv-parser');
const fs = require('fs');
const path = require('path');
const { v4: uuidv4 } = require('uuid');

// Configure multer for CSV file upload
const storage = multer.diskStorage({
    destination: function (req, file, cb) {
        const uploadDir = path.join(__dirname, '../uploads/temp');
        if (!fs.existsSync(uploadDir)) {
            fs.mkdirSync(uploadDir, { recursive: true });
        }
        cb(null, uploadDir);
    },
    filename: function (req, file, cb) {
        const uniqueFilename = `${uuidv4()}_${file.originalname}`;
        cb(null, uniqueFilename);
    }
});

const upload = multer({
    storage: storage,
    limits: {
        fileSize: 10 * 1024 * 1024 // 10MB limit
    },
    fileFilter: (req, file, cb) => {
        if (file.mimetype === 'text/csv' || file.originalname.endsWith('.csv')) {
            cb(null, true);
        } else {
            cb(new Error('Only CSV files are allowed'));
        }
    }
});
// Create new task
async function create(req, res) {
    try {
        const taskData = req.body;
        
        // Add any default values or timestamps
        // taskData.created_at = new Date().toISOString().slice(0, 19).replace('T', ' ');
        
        const createdTask = await task.create(taskData);

        return res.status(201).json({
            status: 201,
            message: 'Task created successfully',
            response: createdTask
        });

    } catch (error) {
        // console.error('Error in createTask:', error);
        return res.status(200).json({
            status: false,
            message: error.message || error.response || error 
        });
    }
}

// Get all tasks
const getAll = async (req, res) => {
    try {
        const filters = {
            contact_id: req.query.contact_id,
            sales_account_id: req.query.sales_account_id,
            deal_id: req.query.deal_id
        };
        const tasks = await task.findAll(req.user.id, filters);
        res.json({ status: 200, message : "task fetched successfully", tasks});
    } catch (error) {
        // console.error('Error fetching tasks:', error);
        res.status(500).json({ status: 500, message: 'Error fetching tasks ' + error });
    }
};
const getFields = async (req, res) => {
    try {
        const userId = req.user.id;
        const tasks = await task.findFields(userId);
        res.json({ status: 200, message : "task fields fetched successfully", tasks});
    } catch (error) {
        // console.error('Error fetching tasks:', error);
        res.status(500).json({ status: 500, message: 'Error fetching tasks Fields' + error });
    }
};

// Get task by ID
const getById = async (req, res) => {
    try {
        const SingleTask = await task.findById(req.params.id);
        // console.log(SingleTask);
        if (!SingleTask) {
            return res.status(404).json({ status: 404, message: 'task not found ' });
        }
        res.json({ status: 200, message : "task fetched successfully", task : SingleTask});
    } catch (error) {
        // console.error('Error fetching task:', error);
        res.status(500).json({ status: 500, message: 'Error fetching task ' + error });
    }
};

// Update task
const update = async (req, res) => {
    try {
        const success = await task.update(req.params.id, req.body);
        if (!success) {
            return res.status(404).json({ message: 'task not found' });
        }
        res.json({ status: 200, message: 'task updated successfully' });
    } catch (error) {
        // console.error('Error updating task:', error);
        res.status(500).json({ status: 500, message: 'Error updating task ' + error });
    }
};

// Delete task
const remove = async (req, res) => {
    try {
        const success = await task.deleteRecord(req.params.id);
        if (!success) {
            return res.status(404).json({ status: 404, message: 'task not found' });
        }
        res.json({ status: 200, message: 'task deleted successfully' });
    } catch (error) {
        // console.error('Error deleting task:', error);
        res.status(500).json({ status: 500, message: 'Error deleting task ' + error });
    }
};
async function getAccountTasks(req, res) {
    try {
        const accountId = req.params.id;
        const tasks = await task.findByAccountId(accountId);
        
        res.json(tasks);
    } catch (error) {
        console.error('Error in getAccountTasks:', error);
        res.status(500).json({
            status: 500,
            message: error.message || 'Error fetching account tasks'
        });
    }
}
async function getDealTasks(req, res) {
    try {
        const dealId = req.params.id;
        const tasks = await task.findByDealId(dealId);
        
        res.json(tasks);
    } catch (error) {
        console.error('Error in getDealTasks:', error);
        res.status(500).json({
            status: 500,
            message: error.message || 'Error fetching deal tasks'
        });
    }
}

// get all tasks
async function getAllTasks(req, res) {
    try {
        const { start_date, end_date, status, user } = req.query;
        const user_id = req.user.id;
        // console.log("task123",user_id);

        // Get user's role
        const userModel = require('../models/user');
        const user_data = await userModel.findUserById(user_id);
        const role_id = user_data.roles_id;

        // Validate date format if dates are provided
        if ((start_date || end_date) && !(start_date && end_date)) {
            return res.status(400).json({
                status: 400,
                success: false,
                message: "Both start_date and end_date are required for date filtering"
            });
        }

        if (start_date && end_date) {
            // Validate date format (YYYY-MM-DD)
            const dateRegex = /^\d{4}-\d{2}-\d{2}$/;
            if (!dateRegex.test(start_date) || !dateRegex.test(end_date)) {
                return res.status(400).json({
                    status: 400,
                    success: false,
                    message: "Invalid date format. Use YYYY-MM-DD"
                });
            }

            // Validate that end_date is not before start_date
            if (new Date(end_date) < new Date(start_date)) {
                return res.status(400).json({
                    status: 400,
                    success: false,
                    message: "End date cannot be before start date"
                });
            }
        }

        // Validate status if provided
        const validStatuses = ['Pending', 'Completed'];
        if (status && !validStatuses.includes(status)) {
            return res.status(400).json({
                status: 400,
                success: false,
                message: "Invalid status. Valid statuses are: " + validStatuses.join(', ')
            });
        }

        const tasks = await task.getalltasks(start_date, end_date, status, role_id, user_id, user);
        
        res.json({
            status: 200,
            success: true,
            message: 'Tasks fetched successfully',
            total: tasks.length,
            data: tasks,
            filters: {
                start_date: start_date || null,
                end_date: end_date || null,
                status: status || null,
                user: user || null,
                role_id: role_id,
                territory_filtered: role_id === 2
            },
            count: tasks.length
        });
    } catch (error) {
        console.error('Error in getAllTasks:', error);
        res.status(500).json({
            status: 500,
            success: false,
            message: error.message || 'Error fetching all tasks'
        });
    }
}

// get task by id
async function getTaskById(req, res) {
    try {
        const taskData = await task.gettasksbyid(req.params.id);
        if (!taskData) {
            return res.status(404).json({
                status: 404,
                success: false,
                message: 'Task not found'
            });
        }
        res.json({
            status: 200,
            success: true,
            message: 'Task fetched successfully',
            task: taskData
        });
    } catch (error) {
        console.error('Error in getTaskById:', error);
        res.status(500).json({
            status: 500,
            success: false,
            message: error.message || 'Error fetching task by id'
        });
    }
}

// Bulk import tasks
async function bulkImport(req, res) {
    try {
        // Validate required fields
        if (!req.file) {
            return res.status(400).json({
                status: 400,
                success: false,
                message: 'CSV file is required'
            });
        }

        if (!req.body.columnMappings) {
            return res.status(400).json({
                status: 400,
                success: false,
                message: 'Column mappings are required'
            });
        }

        let columnMappings;
        try {
            columnMappings = JSON.parse(req.body.columnMappings);
        } catch (error) {
            return res.status(400).json({
                status: 400,
                success: false,
                message: 'Invalid column mappings format'
            });
        }

        if (!Array.isArray(columnMappings)) {
            return res.status(400).json({
                status: 400,
                success: false,
                message: 'Column mappings must be an array'
            });
        }

        // Validate column mappings
        for (const mapping of columnMappings) {
            if (!mapping.csvColumn || !mapping.apiField) {
                return res.status(400).json({
                    status: 400,
                    success: false,
                    message: 'Each mapping must have csvColumn and apiField'
                });
            }
        }

        // Parse CSV file
        const csvData = [];
        const filePath = req.file.path;

        return new Promise((resolve, reject) => {
            fs.createReadStream(filePath)
                .pipe(csv())
                .on('data', (row) => {
                    csvData.push(row);
                })
                .on('end', async () => {
                    try {
                        // Clean up the uploaded file
                        fs.unlinkSync(filePath);

                        if (csvData.length === 0) {
                            return res.status(400).json({
                                status: 400,
                                success: false,
                                message: 'CSV file is empty or has no valid data'
                            });
                        }

                        // Process the import
                        const userId = req.user.id;
                        const results = await task.bulkImport(csvData, columnMappings, userId);

                        res.json({
                            status: 200,
                            success: true,
                            message: 'Bulk import completed',
                            data: {
                                total: results.total,
                                successful: results.successful,
                                failed: results.failed,
                                errors: results.errors.slice(0, 10), // Limit error details to first 10
                                errorCsvFile: results.errorCsvFile // Include error CSV file path if available
                            }
                        });

                        resolve();
                    } catch (error) {
                        // Clean up the uploaded file in case of error
                        if (fs.existsSync(filePath)) {
                            fs.unlinkSync(filePath);
                        }
                        reject(error);
                    }
                })
                .on('error', (error) => {
                    // Clean up the uploaded file in case of error
                    if (fs.existsSync(filePath)) {
                        fs.unlinkSync(filePath);
                    }
                    reject(error);
                });
        });

    } catch (error) {
        console.error('Error in bulk import:', error);
        res.status(500).json({
            status: 500,
            success: false,
            message: 'Error processing bulk import',
            error: error.message
        });
    }
}

module.exports = {
    create,
    getAll,
    getFields,
    getById,
    update,
    remove,
    getAccountTasks,
    getDealTasks,
    getAllTasks,
    getTaskById,
    bulkImport,
    upload
};