const multer = require('multer');
const path = require('path');
const fs = require('fs');

const storage = multer.diskStorage({
    destination: function (req, file, cb) {
        // Validate and get module type
        const belongsto = req.body.belongsto;
        if (!belongsto) {
            cb(new Error('belongsto is required'), null);
            return;
        }

        // Create module-specific upload directory
        const uploadDir = path.join(process.cwd(), 'public', 'uploads', belongsto);
        
        // Create directory if not exists
        if (!fs.existsSync(uploadDir)) {
            fs.mkdirSync(uploadDir, { recursive: true });
        }
        
        cb(null, uploadDir);
    },
    filename: function (req, file, cb) {
        // Get module type for filename prefix
        const belongsto = req.body.belongsto;
        if (!belongsto) {
            cb(new Error('belongsto is required'), null);
            return;
        }
        
        // Clean original filename
        const cleanFileName = file.originalname.replace(/[^a-zA-Z0-9.]/g, '_');
        
        // Generate unique filename
        const uniqueSuffix = Date.now() + '-' + Math.round(Math.random() * 1E9);
        const filename = `${belongsto}-${uniqueSuffix}-${cleanFileName}`;
        
        cb(null, filename);
    }
});

// Add file filter
const fileFilter = (req, file, cb) => {
    // Validate required fields
    if (!req.body.belongsto || !req.body.belongsto_id) {
        cb(new Error('belongsto and belongsto_id are required'), false);
        return;
    }

    // Validate belongsto_id is a number
    const belongstoId = parseInt(req.body.belongsto_id);
    if (isNaN(belongstoId)) {
        cb(new Error('belongsto_id must be a valid number'), false);
        return;
    }

    // Validate belongsto type
    const validTypes = ['contacts', 'accounts', 'deals'];
    if (!validTypes.includes(req.body.belongsto)) {
        cb(new Error('Invalid belongsto type'), false);
        return;
    }

    // Validate file type
    const allowedMimes = [
        'image/jpeg',
        'image/png',
        'image/gif',
        'application/pdf',
        'application/msword',
        'application/vnd.openxmlformats-officedocument.wordprocessingml.document'
    ];
    
    if (!allowedMimes.includes(file.mimetype)) {
        cb(new Error('Invalid file type. Allowed types: JPG, PNG, GIF, PDF, DOC, DOCX'), false);
        return;
    }

    cb(null, true);
};

// Configure multer with limits
const upload = multer({
    storage: storage,
    fileFilter: fileFilter,
    limits: {
        fileSize: 5 * 1024 * 1024, // 5MB limit
        files: 1 // Single file only
    }
});

module.exports = upload;