const db = require('../config/db');
const admin = require('../config/firebaseAdmin');

async function create(data) {
    try {
        // Insert notification into database
        const [result] = await db.execute(
            `INSERT INTO firebaseinboxes (
                user_id, 
                message, 
                trigger_date, 
                trigger_time,
                read_status,
                targetable_type,
                targetable_id,
                created_at
            ) VALUES (?, ?, ?, ?, ?, ?, ?, CURRENT_TIMESTAMP)`,
            [
                data.user_id,
                data.message,
                data.trigger_date || new Date().toISOString().split('T')[0],
                data.trigger_time || new Date().toTimeString().split(' ')[0],
                '0',
                data.targetable_type,
                data.targetable_id
            ]
        );

        // Get user's FCM token
        const [users] = await db.execute(
            'SELECT token FROM users WHERE id = ? AND active = 1',
            [data.user_id]
        );

        if (users.length > 0 && users[0].token) {
            // Prepare Firebase message
            const message = {
                notification: {
                    title: 'New Notification',
                    body: data.message
                },
                data: {
                    notification_id: result.insertId.toString(),
                    type: data.targetable_type,
                    target_id: data.targetable_id.toString(),
                    click_action: 'FLUTTER_NOTIFICATION_CLICK'
                },
                token: users[0].token
            };

            // Send Firebase notification
            try {
                await admin.messaging().send(message);
                
                // Update notification status
                await db.execute(
                    `UPDATE firebaseinboxes SET status = 'sent' WHERE id = ?`,
                    [result.insertId]
                );
            } catch (firebaseError) {
                console.error('Firebase notification error:', firebaseError);
                await db.execute(
                    `UPDATE firebaseinboxes SET status = 'failed', error = ? WHERE id = ?`,
                    [firebaseError.message, result.insertId]
                );
            }
        }

        return {
            success: true,
            notification_id: result.insertId,
            message: 'Notification created and sent successfully'
        };

    } catch (error) {
        console.error('Error in create notification:', error);
        throw new Error(`Failed to create notification: ${error.message}`);
    }
}

async function findAll(userId) {
    try {
        const [notifications] = await db.execute(
            `SELECT 
                f.*,
                DATE_FORMAT(f.created_at, '%Y-%m-%d %H:%i:%s') as formatted_created_at,
                DATE_FORMAT(f.updated_at, '%Y-%m-%d %H:%i:%s') as formatted_updated_at
            FROM firebaseinboxes f
            WHERE f.user_id = ?
            ORDER BY f.created_at DESC`,
            [userId]
        );

        return notifications.map(notification => ({
            id: notification.id,
            message: notification.message,
            trigger_date: notification.trigger_date,
            trigger_time: notification.trigger_time,
            read_status: notification.read_status,
            targetable_type: notification.targetable_type,
            targetable_id: notification.targetable_id,
            created_at: notification.formatted_created_at,
            updated_at: notification.formatted_updated_at
        }));

    } catch (error) {
        console.error('Error in findAll notifications:', error);
        throw new Error(`Failed to fetch notifications: ${error.message}`);
    }
}

async function findById(id) {
    const [rows] = await db.execute('SELECT * FROM firebaseinboxes WHERE id = ?', [id]);
    return rows[0];
}       

async function findByIdAndUpdate(id, data) {
    try {
        // Validate read_status value
        const validStatuses = [0, 1];  // 0 = unread, 1 = read
        const readStatus = parseInt(data.read_status);
        
        if (!validStatuses.includes(readStatus)) {
            throw new Error('Invalid read status. Must be either 0 (unread) or 1 (read)');
        }

        const [result] = await db.execute(
            `UPDATE firebaseinboxes 
             SET 
                read_status = ?,
                updated_at = CURRENT_TIMESTAMP
             WHERE id = ?`,
            [readStatus, id]
        );

        if (result.affectedRows === 0) {
            throw new Error('Notification not found');
        }

        return {
            success: true,
            message: `Notification marked as ${readStatus === 1 ? 'read' : 'unread'}`,
            notification_id: id
        };

    } catch (error) {
        console.error('Error in update notification:', error);
        throw new Error(`Failed to update notification: ${error.message}`);
    }
}
// Delete a notification by ID
async function findByIdAndDelete(id) {
    const [result] = await db.execute('UPDATE firebaseinboxes SET active = 1 WHERE id = ?', [id]);
    return result.affectedRows > 0;
}      

module.exports = {
    create,
    findAll,
    findById,
    findByIdAndUpdate,
    findByIdAndDelete
};
