const db = require('../config/db');

async function create(fieldData) {
    const {
        custom_field = false,
        field_label = '',
        field_name = '',
        field_type = '',
        required = "off",
        quick_add = "off",
        tool_tip = null,
        placeholder = null,
        choices = [],
        read_only = "off",
        unique_field = "off",
        has_dependent = false,
        accountsgroups_id = null,
        active = true,
        order = 0,
        lookup_type = null,
        lookup_column = null
    } = fieldData;

    // Validate required fields
    if (!field_label || !field_name || !field_type) {
        throw new Error('field_label, field_name, and field_type are required');
    }

    // Start transaction
    try {

        // Insert into accountsfields table
        const [result] = await db.execute(
            `INSERT INTO accountsfields (
                field_label, field_name, field_type, required, quick_add,
                tool_tip, placeholder, read_only, unique_field,
                has_dependent, accountsgroups_id, active, \`order\`, lookup_type, lookup_column,
                custom_field, created_at, updated_at
            ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW(), NOW())`,

            [
                field_label,
                field_name,
                field_type,
                required,
                quick_add,
                tool_tip,
                placeholder,
                read_only,
                unique_field,
                has_dependent ? 1 : 0,
                accountsgroups_id,
                active ? 1 : 0,
                order,
                lookup_type,
                lookup_column,
                custom_field ? 1 : 0
            ]
        );

        const accountFieldId = result.insertId;

        // If choices are provided and field type is Dropdown or multiselect, insert them into accountsfieldchoices table
        if ((field_type === 'Dropdown' || field_type === 'multiselect' || field_type === 'radio button' || field_type === 'checkbox') && Array.isArray(choices) && choices.length > 0) {
            for (const choice of choices) {
                if (choice) {  // Only insert if choice is not null/undefined/empty
                    await db.execute(
                        `INSERT INTO accountsfieldchoices (
                            accountsfield_id, custom_option, created_at, updated_at
                        ) VALUES (?, ?, NOW(), NOW())`,
                        [accountFieldId, choice]
                    );
                }
            }
        }

        return accountFieldId;
    } catch (error) {
        throw error;
    } 
}

async function findAll(userId) {
    const connection = await db.getConnection();
    try {
        // Get user's role and access scope
        const [userRole] = await connection.execute(
            `SELECT r.access_scope, r.id as role_id, u.territory_id 
             FROM users u 
             JOIN roles r ON u.roles_id = r.id 
             WHERE u.id = ?`,
            [userId]
        );

        if (!userRole || !userRole.length) {
            throw new Error('User role not found');
        }
        
        const accessScope = JSON.parse(userRole[0].access_scope);

        let [rows] = await db.execute('SELECT * FROM accountsfields where active = 1 ORDER BY `order` ASC');
        const [choices] = await db.execute('SELECT * FROM accountsfieldchoices');

        // Process each field
        const processedRows = await Promise.all(rows.map(async field => {
            // Handle regular choices
            field.choices = choices.filter(choice => choice.accountsfield_id === field.id);
            field.field_value = null;

            // Handle lookup type fields
            if (field.field_type.toLowerCase() === 'lookup' && field.lookup_type && field.lookup_column) {
                console.log({field});
                try {
                    let query = `SELECT id, ${field.lookup_column} FROM ${field.lookup_type} WHERE active = 1`;
                    const params = [];
                    // if (field.lookup_type === 'contacts') {
                    //     query += ' AND owner_id = ?';
                    //     params.push(userId);
                    // } else if (field.lookup_type === 'sales_accounts') {
                    //     query += ' AND owner_id = ?';
                    //     params.push(userId);
                    // } else if (field.lookup_type === 'deals') {
                    //     query += ' AND owner_id = ?';
                    //     params.push(userId);
                    // } 

                    // Get view permission for the specific module
                    const modulePermission = field.lookup_type === 'contacts' ? accessScope?.contact?.view :
                                          field.lookup_type === 'accounts' ? accessScope?.sales_account?.view :
                                          field.lookup_type === 'deals' ? accessScope?.deal?.view : 'global';

                    switch (modulePermission) {
                        case 'owned':
                            query += ' AND owner_id = ?';
                            params.push(userId);
                            break;

                        case 'territory':
                            if (userRole[0].territory_id) {
                                const territories = userRole[0].territory_id.split(',');
                                const placeholders = territories.map(() => '?').join(',');
                                query += ` AND territory_id IN (${placeholders})`;
                                params.push(...territories);
                            } else {
                                field.choices = [];
                                return field;
                            }
                            break;

                        case 'restricted':
                            field.choices = [];
                            return field;

                        case 'global':
                        default:
                            // No additional filters for global access
                            break;
                    }

                    // Get data from lookup table with permissions applied
                    const [lookupData] = await connection.execute(query, params);

                    // Format lookup data as choices
                    field.choices = lookupData.map(item => ({
                        id: item.id,
                        accountfield_id: field.id,
                        custom_option: item[field.lookup_column]
                    }));
                                   
                } catch (error) {
                    console.error(`Error fetching lookup data for field ${field.field_name}:`, error);
                    field.choices = [];
                }
            }

            return field;
        }));

        return processedRows;

    } catch (error) {
        console.error('Error in findAll:', error);
        throw error;
    } finally {
        connection.release();
    }
}

async function findById(id) {
    try {
        const [fields] = await db.execute('SELECT * FROM accountsfields WHERE id = ?', [id]);
        if (!fields.length) return null;

        const [choices] = await db.execute('SELECT * FROM accountsfieldchoices');
        
        // Process the field
        const field = fields[0];
        
        // Handle regular choices
        field.choices = choices.filter(choice => choice.accountsfield_id === field.id);
        field.field_value = null;

        // Handle lookup type fields
        if (field.field_type === 'lookup' && field.lookup_type && field.lookup_column) {
            try {
                const [lookupData] = await db.execute(
                    `SELECT id, ${field.lookup_column} FROM ${field.lookup_type} WHERE active = 1`
                );

                // Format lookup data as choices
                field.choices = lookupData.map(item => ({
                    id: item.id,
                    accountfield_id: field.id,
                    custom_option: item[field.lookup_column],
                    created_at: null,
                    updated_at: null
                }));
            } catch (error) {
                console.error(`Error fetching lookup data for field ${field.field_name}:`, error);
                field.choices = [];
            }
        }

        return field;

    } catch (error) {
        console.error('Error in findById:', error);
        throw new Error(`Failed to fetch account field: ${error.message}`);
    }
}
async function update(id, fieldData) {
    const {
        custom_field = false,
        field_label,
        field_name,
        field_type,
        required = "off",
        quick_add = "off",
        tool_tip = null,
        placeholder = null,
        choices = [],
        read_only = "off",
        unique_field = "off",
        has_dependent = false,
        accountsgroups_id = null,
        active = true,
        order = 0,
        lookup_type = null,
        lookup_column = null
    } = fieldData;

    // Validate required fields
    if (!field_label || !field_name || !field_type) {
        throw new Error('field_label, field_name, and field_type are required');
    }

    // Start transaction
    ;
    try {
        // Update accountsfields table
        const [result] = await db.execute(      
            `UPDATE accountsfields SET 
                field_label = ?,
                field_name = ?,
                field_type = ?,
                required = ?,
                quick_add = ?,
                tool_tip = ?,
                placeholder = ?,
                read_only = ?,
                unique_field = ?,
                has_dependent = ?,
                accountsgroups_id = ?,
                active = ?,
                \`order\` = ?,
                lookup_type = ?,
                lookup_column = ?,
                custom_field = ?,
                updated_at = NOW()
            WHERE id = ?`,
            [
                field_label,
                field_name,
                field_type,
                required,
                quick_add ,
                tool_tip,
                placeholder,
                read_only,
                unique_field,
                has_dependent ? 1 : 0,
                accountsgroups_id,
                active ? 1 : 0,
                order,
                lookup_type,
                lookup_column,
                custom_field ? 1 : 0,
                id
            ]
        );

        // Handle choices if field type is select or multiselect
        if (field_type === 'Dropdown' || field_type === 'multiselect') {
            // Get existing choices
            const [existingChoices] = await db.execute(
                'SELECT id, custom_option FROM accountsfieldchoices WHERE accountsfield_id = ?',
                [id]
            );
            console.log("existingChoices",existingChoices);

            // Create maps for easy comparison
            const existingChoicesMap = new Map();
            existingChoices.forEach(choice => {
                existingChoicesMap.set(choice.custom_option, choice.id);
            });

            console.log("existingChoicesMap",existingChoicesMap);

            const newChoicesSet = new Set();
            if (Array.isArray(choices) && choices.length > 0) {
                choices.forEach(choice => {
                    if (choice && choice.trim() !== '') {
                        newChoicesSet.add(choice.trim());
                    }
                });
            }
            console.log("newChoicesSet",newChoicesSet);

            // Helper function to calculate string similarity
            const calculateSimilarity = (str1, str2) => {
                if (str1 === str2) return 1;
                if (str1.length === 0) return str2.length === 0 ? 1 : 0;
                if (str2.length === 0) return 0;
                
                // Check if one is a substring of the other (for cases like "1a" -> "1a11")
                if (str1.includes(str2) || str2.includes(str1)) {
                    return 0.8; // High similarity for substring matches
                }
                
                // Calculate edit distance
                const matrix = [];
                for (let i = 0; i <= str2.length; i++) {
                    matrix[i] = [i];
                }
                for (let j = 0; j <= str1.length; j++) {
                    matrix[0][j] = j;
                }
                for (let i = 1; i <= str2.length; i++) {
                    for (let j = 1; j <= str1.length; j++) {
                        if (str2.charAt(i - 1) === str1.charAt(j - 1)) {
                            matrix[i][j] = matrix[i - 1][j - 1];
                        } else {
                            matrix[i][j] = Math.min(
                                matrix[i - 1][j - 1] + 1,
                                matrix[i][j - 1] + 1,
                                matrix[i - 1][j] + 1
                            );
                        }
                    }
                }
                
                const maxLength = Math.max(str1.length, str2.length);
                return maxLength === 0 ? 1 : (maxLength - matrix[str2.length][str1.length]) / maxLength;
            };

            // Find potential updates by matching similar strings
            const choicesToUpdate = [];
            const usedNewChoices = new Set();
            const usedExistingChoices = new Set();

            // First, try exact matches
            for (const existingChoice of existingChoices) {
                if (newChoicesSet.has(existingChoice.custom_option)) {
                    usedNewChoices.add(existingChoice.custom_option);
                    usedExistingChoices.add(existingChoice.custom_option);
                }
            }

            // Then, try to match similar strings (potential updates)
            for (const existingChoice of existingChoices) {
                if (usedExistingChoices.has(existingChoice.custom_option)) continue;
                
                let bestMatch = null;
                let bestSimilarity = 0.6; // Minimum similarity threshold
                
                for (const newChoice of newChoicesSet) {
                    if (usedNewChoices.has(newChoice)) continue;
                    
                    const similarity = calculateSimilarity(existingChoice.custom_option, newChoice);
                    if (similarity > bestSimilarity) {
                        bestSimilarity = similarity;
                        bestMatch = newChoice;
                    }
                }
                
                if (bestMatch) {
                    choicesToUpdate.push({
                        id: existingChoice.id,
                        oldValue: existingChoice.custom_option,
                        newValue: bestMatch
                    });
                    usedNewChoices.add(bestMatch);
                    usedExistingChoices.add(existingChoice.custom_option);
                }
            }

            // Find choices to delete (exist in DB but not matched)
            const choicesToDelete = existingChoices.filter(choice => 
                !usedExistingChoices.has(choice.custom_option)
            );
            console.log("choicesToDelete",choicesToDelete);

            // Find choices to insert (exist in new data but not matched)
            const choicesToInsert = Array.from(newChoicesSet).filter(choice => 
                !usedNewChoices.has(choice)
            );
            console.log("choicesToInsert",choicesToInsert);
            console.log("choicesToUpdate",choicesToUpdate);

            // Delete choices that are no longer needed
            if (choicesToDelete.length > 0) {
                const deleteIds = choicesToDelete.map(choice => choice.id);
                const placeholders = deleteIds.map(() => '?').join(',');
                await db.execute(
                    `DELETE FROM accountsfieldchoices WHERE id IN (${placeholders})`,
                    deleteIds
                );
            }

            // Update existing choices that have changed values
            if (choicesToUpdate.length > 0) {
                for (const choice of choicesToUpdate) {
                    await db.execute(
                        `UPDATE accountsfieldchoices 
                         SET custom_option = ?, updated_at = NOW() 
                         WHERE id = ?`,
                        [choice.newValue, choice.id]
                    );
                }
            }

            // Insert new choices
            if (choicesToInsert.length > 0) {
                for (const choice of choicesToInsert) {
                    await db.execute(
                        `INSERT INTO accountsfieldchoices (
                            accountsfield_id, custom_option, created_at, updated_at
                        ) VALUES (?, ?, NOW(), NOW())`,
                        [id, choice]
                    );
                }
            }
        }

        return result.affectedRows > 0;
    } catch (error) {
        throw error;
    } 
}

async function deleteRecord (id) {
    // First, get the current active status
    const [rows] = await db.execute('SELECT active FROM accountsfields WHERE id = ?', [id]);

    if (rows.length === 0) {
        return false; // Record not found
    }

    // Toggle the active status (0 to 1, 1 to 0)
    const currentActive = rows[0].active;
    const newActive = currentActive === 1 ? 0 : 1;

    // Update the record with the new active status
    const [result] = await db.execute('UPDATE accountsfields SET active = ? WHERE id = ?', [newActive, id]);
    return result.affectedRows > 0;
}

// Get all account fields with groups
async function findAllWithGroupName() {
    try {
        // Get all fields with their group information using JOIN
        const query = `
            SELECT 
                af.*,
                ag.group_name
            FROM accountsfields af
            LEFT JOIN accountsgroups ag ON af.accountsgroups_id = ag.id
            WHERE af.active = 1 AND (ag.active = 1 OR af.accountsgroups_id IS NULL)
            ORDER BY ag.id, af.id`;

        let [fields] = await db.execute(query);
        const [choices] = await db.execute('SELECT * FROM accountsfieldchoices');

        // Process each field
        const processedFields = await Promise.all(fields.map(async field => {
            // Handle regular choices
            field.choices = choices.filter(choice => choice.accountsfield_id === field.id);
            field.field_value = null;

            // Handle lookup type fields
            if (field.field_type === 'lookup' && field.lookup_type && field.lookup_column) {
                try {
                    // Get data from lookup table
                    const [lookupData] = await db.execute(
                        `SELECT id, ${field.lookup_column} FROM ${field.lookup_type}`
                    );

                    // Format lookup data as choices
                    field.choices = lookupData.map(item => ({
                        id: item.id,
                        accountfield_id: field.id,
                        custom_option: item[field.lookup_column],
                        created_at: null,
                        updated_at: null
                    }));
                } catch (error) {
                    console.error(`Error fetching lookup data for field ${field.field_name}:`, error);
                    field.choices = [];
                }
            }

            return {
                id: field.id,
                field_label: field.field_label,
                field_name: field.field_name,
                field_type: field.field_type,
                required: field.required,
                quick_add: field.quick_add,
                tool_tip: field.tool_tip,
                placeholder: field.placeholder,
                read_only: field.read_only,
                has_dependent: field.has_dependent,
                accountsgroups_id: field.accountsgroups_id,
                group_name: field.group_name,
                active: field.active,
                lookup_type: field.lookup_type,
                lookup_column: field.lookup_column,
                created_at: field.created_at,
                updated_at: field.updated_at,
                choices: field.choices,
                field_value: field.field_value
            };
        }));

        // Group fields by accountsgroups_id
        const groupedFields = processedFields.reduce((groups, field) => {
            const groupId = field.accountsgroups_id || 'ungrouped';
            
            if (!groups[groupId]) {
                groups[groupId] = {
                    group_id: field.accountsgroups_id,
                    group_name: field.group_name || "Ungrouped Fields",
                    is_active: 1,
                    fields: []
                };
            }
            
            groups[groupId].fields.push(field);
            return groups;
        }, {});

        // Convert groups object to array and sort by group id
        const sortedGroups = Object.values(groupedFields)
            .sort((a, b) => {
                if (a.group_id === null) return 1;
                if (b.group_id === null) return -1;
                return a.group_id - b.group_id;
            })
            .map(group => ({
                ...group,
                total_fields: group.fields.length,
                fields: group.fields.sort((a, b) => a.id - b.id)
            }));

        return {
            status: 200,
            success: true,
            message: "Account fields with groups fetched successfully",
            data: {
                total_groups: sortedGroups.length,
                total_fields: processedFields.length,
                groups: sortedGroups
            }
        };

    } catch (error) {
        console.error('Error in findAllWithGroupName:', error);
        throw new Error(`Failed to fetch account fields with groups: ${error.message}`);
    }
}

async function order(fields) {
    // console.log(fields, "fields");
    try {
        for (const field of fields) {
            await db.execute(
                `UPDATE accountsfields 
                    SET \`order\` = ?, 
                        updated_at = NOW() 
                    WHERE id = ?`,
                [field.order, field.id]
            );
        }
        
        return {
            status: 200,
            success: true,
            message: 'Field order updated successfully'
        };
    } catch (error) {
        console.error('Error updating field order:', error);
        throw new Error(`Failed to update field order: ${error.message}`);
    }

}



module.exports = {
    create,
    findAll,
    findById,
    update,
    deleteRecord,
    findAllWithGroupName,
    order
};