const db = require('../config/db');

async function create(fieldData) {
    const {
        custom_field = false,
        field_label = '',
        field_name = '',
        field_type = '',
        required = false,
        quick_add = false,
        tool_tip = null,
        placeholder = null,
        choices = [],
        read_only = false,
        has_dependent = false,
        accountsgroups_id = null,
        active = true,
        order = 0,
        lookup_type = null,
        lookup_column = null
    } = fieldData;

    // Validate required fields
    if (!field_label || !field_name || !field_type) {
        throw new Error('field_label, field_name, and field_type are required');
    }

    // Start transaction
    const connection = await db.getConnection();
    try {
        await connection.beginTransaction();

        // Insert into accountsfields table
        const [result] = await connection.execute(
            `INSERT INTO accountsfields (
                field_label, field_name, field_type, required, quick_add,
                tool_tip, placeholder, read_only, has_dependent,
                accountsgroups_id, active, \`order\`, lookup_type, lookup_column,
                custom_field, created_at, updated_at
            ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW(), NOW())`,
            [
                field_label,
                field_name,
                field_type,
                required ? 1 : 0,
                quick_add ? 1 : 0,
                tool_tip,
                placeholder,
                read_only ? 1 : 0,
                has_dependent ? 1 : 0,
                accountsgroups_id,
                active ? 1 : 0,
                order,
                lookup_type,
                lookup_column,
                custom_field ? 1 : 0
            ]
        );

        const accountFieldId = result.insertId;

        // If choices are provided and field type is select or multiselect, insert them into accountsfieldchoices table
        if ((field_type === 'select' || field_type === 'multiselect') && Array.isArray(choices) && choices.length > 0) {
            for (const choice of choices) {
                if (choice) {  // Only insert if choice is not null/undefined/empty
                    await connection.execute(
                        `INSERT INTO accountsfieldchoices (
                            accountsfield_id, custom_option, created_at, updated_at
                        ) VALUES (?, ?, NOW(), NOW())`,
                        [accountFieldId, choice]
                    );
                }
            }
        }

        await connection.commit();
        return accountFieldId;
    } catch (error) {
        await connection.rollback();
        throw error;
    } finally {
        connection.release();
    }
}
async function findAll() {
    try {
        // Get all fields and choices
        let [rows] = await db.execute('SELECT * FROM accountsfields WHERE active=1 ORDER BY `order` ASC');
        const [choices] = await db.execute('SELECT * FROM accountsfieldchoices');

        // Process each field
        const processedRows = await Promise.all(rows.map(async field => {
            // Add choices if any
            field.choices = choices.filter(choice => choice.accountsfield_id === field.id);
            field.field_value = null;

            // Handle lookup type fields
            if (field.field_type === 'lookup' && field.lookup_type && field.lookup_column) {
                try {
                    // Get lookup values from referenced table
                    const [lookupValues] = await db.execute(
                        `SELECT id, ${field.lookup_column} as value FROM ${field.lookup_type}`
                    );
                    
                    // Add lookup values as choices
                    field.choices = lookupValues.map(item => ({
                        id: item.id,
                        accountfield_id: field.id,
                        custom_option: item.value,
                        created_at: null,
                        updated_at: null
                    }));
                } catch (error) {
                    console.error(`Error fetching lookup values for field ${field.field_name}:`, error);
                    field.choices = [];
                }
            }

            return field;
        }));

        return processedRows;
        
    } catch (error) {
        console.error('Error in findAll:', error);
        throw new Error('Failed to fetch account fields');
    }
}
async function findById(id) {
    try {
        const [fields] = await db.execute('SELECT * FROM accountsfields WHERE id = ?', [id]);
        if (!fields.length) return null;

        const [choices] = await db.execute('SELECT * FROM accountsfieldchoices');
        
        // Process the field
        const field = fields[0];
        
        // Handle regular choices
        field.choices = choices.filter(choice => choice.accountsfield_id === field.id);
        field.field_value = null;

        // Handle lookup type fields
        if (field.field_type === 'lookup' && field.lookup_type && field.lookup_column) {
            try {
                const [lookupData] = await db.execute(
                    `SELECT id, ${field.lookup_column} FROM ${field.lookup_type} WHERE active = 1`
                );

                // Format lookup data as choices
                field.choices = lookupData.map(item => ({
                    id: item.id,
                    accountfield_id: field.id,
                    custom_option: item[field.lookup_column],
                    created_at: null,
                    updated_at: null
                }));
            } catch (error) {
                console.error(`Error fetching lookup data for field ${field.field_name}:`, error);
                field.choices = [];
            }
        }

        return field;

    } catch (error) {
        console.error('Error in findById:', error);
        throw new Error(`Failed to fetch account field: ${error.message}`);
    }
}
async function update(id, fieldData) {
    const {
        custom_field = false,
        field_label,
        field_name,
        field_type,
        required = false,
        quick_add = false,
        tool_tip = null,
        placeholder = null,
        choices = [],
        read_only = false,
        has_dependent = false,
        accountsgroups_id = null,
        active = true,
        order = 0,
        lookup_type = null,
        lookup_column = null
    } = fieldData;

    // Validate required fields
    if (!field_label || !field_name || !field_type) {
        throw new Error('field_label, field_name, and field_type are required');
    }

    // Start transaction
    const connection = await db.getConnection();
    try {
        await connection.beginTransaction();

        // Update accountsfields table
        const [result] = await connection.execute(
            `UPDATE accountsfields SET 
                field_label = ?,
                field_name = ?,
                field_type = ?,
                required = ?,
                quick_add = ?,
                tool_tip = ?,
                placeholder = ?,
                read_only = ?,
                has_dependent = ?,
                accountsgroups_id = ?,
                active = ?,
                \`order\` = ?,
                lookup_type = ?,
                lookup_column = ?,
                custom_field = ?,
                updated_at = NOW()
            WHERE id = ?`,
            [
                field_label,
                field_name,
                field_type,
                required ? 1 : 0,
                quick_add ? 1 : 0,
                tool_tip,
                placeholder,
                read_only ? 1 : 0,
                has_dependent ? 1 : 0,
                accountsgroups_id,
                active ? 1 : 0,
                order,
                lookup_type,
                lookup_column,
                custom_field ? 1 : 0,
                id
            ]
        );

        // Handle choices if field type is select or multiselect
        if (field_type === 'select' || field_type === 'multiselect') {
            // First, delete existing choices
            await connection.execute(
                'DELETE FROM accountsfieldchoices WHERE accountsfield_id = ?',
                [id]
            );

            // Then insert new choices if provided
            if (Array.isArray(choices) && choices.length > 0) {
                for (const choice of choices) {
                    if (choice) {  // Only insert if choice is not null/undefined/empty
                        await connection.execute(
                            `INSERT INTO accountsfieldchoices (
                                accountsfield_id, custom_option, created_at, updated_at
                            ) VALUES (?, ?, NOW(), NOW())`,
                            [id, choice]
                        );
                    }
                }
            }
        }

        await connection.commit();
        return result.affectedRows > 0;
    } catch (error) {
        await connection.rollback();
        throw error;
    } finally {
        connection.release();
    }
}

async function deleteRecord (id) {
    // First, get the current active status
    const [rows] = await db.execute('SELECT active FROM accountsfields WHERE id = ?', [id]);

    if (rows.length === 0) {
        return false; // Record not found
    }

    // Toggle the active status (0 to 1, 1 to 0)
    const currentActive = rows[0].active;
    const newActive = currentActive === 1 ? 0 : 1;

    // Update the record with the new active status
    const [result] = await db.execute('UPDATE accountsfields SET active = ? WHERE id = ?', [newActive, id]);
    return result.affectedRows > 0;
}

// Get all account fields with groups
async function findAllWithGroupName() {
    try {
        // Get all fields with their group information using JOIN
        const query = `
            SELECT 
                af.*,
                ag.group_name
            FROM accountsfields af
            LEFT JOIN accountsgroups ag ON af.accountsgroups_id = ag.id
            WHERE af.active = 1 AND (ag.active = 1 OR af.accountsgroups_id IS NULL)
            ORDER BY ag.id, af.id`;

        let [fields] = await db.execute(query);
        const [choices] = await db.execute('SELECT * FROM accountsfieldchoices');

        // Process each field
        const processedFields = await Promise.all(fields.map(async field => {
            // Handle regular choices
            field.choices = choices.filter(choice => choice.accountsfield_id === field.id);
            field.field_value = null;

            // Handle lookup type fields
            if (field.field_type === 'lookup' && field.lookup_type && field.lookup_column) {
                try {
                    // Get data from lookup table
                    const [lookupData] = await db.execute(
                        `SELECT id, ${field.lookup_column} FROM ${field.lookup_type}`
                    );

                    // Format lookup data as choices
                    field.choices = lookupData.map(item => ({
                        id: item.id,
                        accountfield_id: field.id,
                        custom_option: item[field.lookup_column],
                        created_at: null,
                        updated_at: null
                    }));
                } catch (error) {
                    console.error(`Error fetching lookup data for field ${field.field_name}:`, error);
                    field.choices = [];
                }
            }

            return {
                id: field.id,
                field_label: field.field_label,
                field_name: field.field_name,
                field_type: field.field_type,
                required: field.required,
                quick_add: field.quick_add,
                tool_tip: field.tool_tip,
                placeholder: field.placeholder,
                read_only: field.read_only,
                has_dependent: field.has_dependent,
                accountsgroups_id: field.accountsgroups_id,
                group_name: field.group_name,
                active: field.active,
                lookup_type: field.lookup_type,
                lookup_column: field.lookup_column,
                created_at: field.created_at,
                updated_at: field.updated_at,
                choices: field.choices,
                field_value: field.field_value
            };
        }));

        // Group fields by accountsgroups_id
        const groupedFields = processedFields.reduce((groups, field) => {
            const groupId = field.accountsgroups_id || 'ungrouped';
            
            if (!groups[groupId]) {
                groups[groupId] = {
                    group_id: field.accountsgroups_id,
                    group_name: field.group_name || "Ungrouped Fields",
                    is_active: 1,
                    fields: []
                };
            }
            
            groups[groupId].fields.push(field);
            return groups;
        }, {});

        // Convert groups object to array and sort by group id
        const sortedGroups = Object.values(groupedFields)
            .sort((a, b) => {
                if (a.group_id === null) return 1;
                if (b.group_id === null) return -1;
                return a.group_id - b.group_id;
            })
            .map(group => ({
                ...group,
                total_fields: group.fields.length,
                fields: group.fields.sort((a, b) => a.id - b.id)
            }));

        return {
            status: 200,
            success: true,
            message: "Account fields with groups fetched successfully",
            data: {
                total_groups: sortedGroups.length,
                total_fields: processedFields.length,
                groups: sortedGroups
            }
        };

    } catch (error) {
        console.error('Error in findAllWithGroupName:', error);
        throw new Error(`Failed to fetch account fields with groups: ${error.message}`);
    }
}

async function order(fields) {
    // console.log(fields, "fields");
    try {
        for (const field of fields) {
            await db.execute(
                `UPDATE accountsfields 
                    SET \`order\` = ?, 
                        updated_at = NOW() 
                    WHERE id = ?`,
                [field.order, field.id]
            );
        }
        
        return {
            status: 200,
            success: true,
            message: 'Field order updated successfully'
        };
    } catch (error) {
        console.error('Error updating field order:', error);
        throw new Error(`Failed to update field order: ${error.message}`);
    }

}



module.exports = {
    create,
    findAll,
    findById,
    update,
    deleteRecord,
    findAllWithGroupName,
    order
};