const { json } = require('express');
const db = require('../config/db');
const { object } = require('@google/maps/lib/internal/validate');
const dayjs = require('dayjs');


// async function create(Data) {
//     // console.log(Data);
//     // console.log("hi");
//     let connection = null;
//     try {
//         connection = db;

//         const { contact: allFields } = Data;
//         if (!allFields || !Array.isArray(allFields)) {
//             throw new Error('Invalid input data format');
//         }
//         // console.log("1");
//         // Get unique fields from contactfields table
//         const [uniqueFields] = await connection.execute(
//             'SELECT field_name FROM contactfields WHERE unique_field = "on"'
//         );
//         // console.log("2");
//         // Check for uniqueness constraints
//         for (const field of allFields) {
//             if (uniqueFields.some(uf => uf.field_name === field.field_name) && field.field_value) {
//                 // Check if value already exists
//                 const [existing] = await connection.execute(
//                     `SELECT id FROM contacts WHERE ${field.field_name} = ? AND active = 1`,
//                     [field.field_value]
//                 );
//                 console.log(existing);

//                 if (existing.length > 0) {
//                     throw new Error(`Contact with ${field.field_name} "${field.field_value}" already exists`);
//                 }
//             }
//         }

//         // console.log("3");
//         // Prepare data for insertion
//         const custom_fields = Array.from(allFields)
//             .filter(field => field.custom_field == 1)
//             .map(field => ({ [field.field_name]: field.field_value }));

//         const standardFields = Array.from(allFields)
//             .filter(field => field.custom_field == 0);

//         const contact = [
//             ...standardFields.map(field => 	field.field_name === 'owner_id' && field.field_value == 'null' ? Data.user_id : field.field_value),
//             JSON.stringify(custom_fields),
//             1 // active status
//         ];

//         const keys = standardFields
//             .map(field => field.field_name)
//             .concat(['custom_field', 'active'])
//             .join(', ');
        
//         const placeholders = '?' + ', ?'.repeat(contact.length - 1);
//         // console.log(`INSERT INTO contacts (${keys}) VALUES (${placeholders})`,contact);

//         const [result] = await connection.execute(
//             `INSERT INTO contacts (${keys}) VALUES (${placeholders})`,
//             contact
//         );
//         // console.log(result);
//         // await connection.commit();
//         return result.insertId;

//     } catch (error) {
//         // if (connection) {
//         //     await connection.rollback();
//         // }
//         console.error('Error creating contact:', error);
//         throw new Error(`Failed to create contact: ${error.message}`);
//     }
//     //  finally {
//     //     if (connection) {
//     //         // connection.release();
//     //     }
//     // }
// }


async function create(Data) {
    try {
        connection = db;
        const { contact: allFields } = Data;
        if (!allFields || !Array.isArray(allFields)) {
            throw new Error('Invalid input data format');
        }
        const uniqueFields = Array.from(allFields).filter(field => field.unique_field == 1 || field.unique_field == 'on');
        for (const field of allFields) {
            if (uniqueFields.some(uf => uf.field_name === field.field_name) && field.field_value) {
                // Check if value already exists
                const [existing] = await connection.execute(
                    `SELECT id FROM contacts WHERE ${field.field_name} = ? AND active = 1`,
                    [field.field_value]
                );
                if (existing.length > 0) {
                    console.log(`Contact with ${field.field_name} "${field.field_value}" already exists`);
                    return {
                        status: 400,
                        success: false,
                        message: `Contact with ${field.field_name} "${field.field_value}" already exists`,
                    };
                }
            }
        }
        const custom_fields = Array.from(allFields)
            .filter(field => field.custom_field == 1)
            .map(field => ({ [field.field_name]: field.field_value }));
        const standardFields = Array.from(allFields)
            .filter(field => field.custom_field == 0);

        const contact = [
            ...standardFields.map(field => 	field.field_name === 'owner_id' && field.field_value == 'null' ? Data.user_id : field.field_value),
            JSON.stringify(custom_fields),
            1 // active status
        ];
        const keys = standardFields
            .map(field => field.field_name)
            .concat(['custom_field', 'active'])
            .join(', ');
        const placeholders = '?' + ', ?'.repeat(contact.length - 1);
        const [result] = await connection.execute(
            `INSERT INTO contacts (${keys}) VALUES (${placeholders})`,
            contact
        );
        return result.insertId;
    } catch (error) {
        console.error('Error creating contact:', error);
        throw new Error(`Failed to create contact: ${error.message}`);
    }

}

async function findAll(filters = {}) {
    let query = `
        SELECT c.*, u.name as owner_name 
        FROM contacts c 
        LEFT JOIN users u ON c.owner_id = u.id 
        WHERE c.active = 1`;
    const params = [];
    console.log(filters)

    // Add role-based access control
    if (filters.territory_id) {
        query += ' AND c.territory_id = ?';
        params.push(filters.territory_id);
    }
    if (filters.owner_id) {
        query += ' AND c.owner_id = ?';
        params.push(filters.owner_id);
    }

    // Add search functionality
    if (filters.search) {
        query += ' AND (c.sales_accounts LIKE ? OR c.email LIKE ? OR c.phone LIKE ?)';
        const searchTerm = `%${filters.search}%`;
        params.push(searchTerm, searchTerm, searchTerm);
    }

    // Add specific field filters
    if (filters.sales_accounts) {
        query += ' AND c.sales_accounts = ?';
        params.push(filters.sales_accounts);
    }
    if (filters.email) {
        query += ' AND c.email = ?';
        params.push(filters.email);
    }
    if (filters.phone) {
        query += ' AND c.phone = ?';
        params.push(filters.phone);
    }

    // Add sorting
    query += ' ORDER BY c.created_at DESC';

    const [rows] = await db.execute(query, params);
    return rows;
}


// async function findById(id) {
//     try {
//         const query = `
//             SELECT 
//                 c.*,
//                 t.id as task_id,t.active as task_status, t.title as task_title, t.description as task_description,
//                 sa.id as sales_account_id, sa.name as sales_account_name,
//                 m.id as meeting_id, m.activity_title as meeting_title, 
//                 m.description as meeting_description, m.status as meeting_status,
//                 m.start_date, m.start_time, m.end_date, m.end_time,
//                 n.id as note_id, n.notes, n.voicenote, n.belongsto_id, n.belongsto, 
//                 n.users_id, n.note_time, n.active as note_active, n.is_deleted,
//                 n.created_at as note_created_at, n.updated_at as note_updated_at,
//                 f.id as file_id, f.filename, f.file_url,
//                 u.name as user_name
//             FROM contacts c
//             LEFT JOIN tasks t ON t.related_to = c.id AND t.active=1
//             LEFT JOIN accounts sa ON sa.id = c.sales_accounts
//             LEFT JOIN salesactivitydatas m ON m.salesactivities_id = 3 and m.active=1 
//                 AND m.targetable_id = c.id 
//                 AND (m.targetable_type = 'contacts' OR m.targetable_type = 'Contacts')
//             LEFT JOIN notes n ON n.belongsto_id = c.id 
//                 AND n.belongsto = 'contacts' AND n.is_deleted = 0
//             LEFT JOIN files f ON f.belongsto_id = c.id 
//                 AND (f.belongsto = 'contacts' OR f.belongsto = 'Contacts')
//             LEFT JOIN users u ON u.id = n.users_id
//             WHERE c.id = ? AND c.active = 1`;

//         const [rows] = await db.execute(query, [id]);
//         if (!rows.length) return null;

//         const [fields] = await db.execute('SELECT * FROM contactfields WHERE active=1 ORDER BY id');
//         const [choices] = await db.execute('SELECT * FROM customselectoptions');

//         // Process fields with lookup handling
//         const contacts = await Promise.all(fields.map(async field => {
//             // Handle regular choices
//             field.choices = choices.filter(choice => choice.contactfield_id === field.id);
            
//             // Set field value
//             field.field_value = rows[0][field.field_name];
            
//             // Handle custom fields
//             if (field.custom_field === 1 && rows[0].custom_field) {
//                 const customFields = JSON.parse(rows[0].custom_field);
//                 field.field_value = customFields.find(cf => 
//                     cf.hasOwnProperty(field.field_name))?.[field.field_name] || null;
//             }

//             // Handle lookup type fields
//             if (field.field_type.toLowerCase() === 'lookup' && field.lookup_type && field.lookup_column) {
//                 try {
//                     const [lookupData] = await db.execute(
//                         `SELECT id, ${field.lookup_column} FROM ${field.lookup_type} WHERE active = 1`
//                     );

//                     // Format lookup data as choices
//                     field.choices = lookupData.map(item => ({
//                         id: item.id,
//                         contactfield_id: field.id,
//                         custom_option: item[field.lookup_column]
//                     }));

//                 } catch (error) {
//                     console.error(`Error fetching lookup data for field ${field.field_name}:`, error);
//                     field.choices = [];
//                 }
//             }

//             return field;
//         }));

//         const contact = {
//             id: id.toString(),
//             contacts,
//             tasks: [],
//             sales_account: null,
//             meetings: [],
//             files:[],
//             notes: [],
//             count:{}
//         };
        
//         rows.forEach(row => {
//             // Add task if exists
//             if ((row.task_id)&&(row.task_status!=0)) {
//                 if (!contact.tasks.some(task => task.id === row.task_id)) {
//                     contact.tasks.push({
//                         id: row.task_id,
//                         title: row.task_title,
//                         description: row.task_description,
//                         status: row.task_status
//                     });
//                 }
//             }
            
//             // Add sales account if exists
//             if (row.sales_account_id && !contact.sales_account) {
//                 contact.sales_account = {
//                     id: row.sales_account_id,
//                     name: row.sales_account_name
//                 };
//             }
            
//             // Add meeting if exists
//             if (row.meeting_id) {
//                 if (!contact.meetings.some(meeting => meeting.id === row.meeting_id)) {
//                     contact.meetings.push({
//                         id: row.meeting_id,
//                         title: row.meeting_title,
//                         description: row.meeting_description,
//                         status: row.meeting_status,
//                         start_date: row.start_date,
//                         start_time: row.start_time,
//                         end_date: row.end_date,
//                         end_time: row.end_time
//                     });
//                 }
//             }

//             // Add note if exists
//             if (row.note_id && !row.is_deleted) {
//                 if (!contact.notes.some(note => note.id === row.note_id)) {
//                     contact.notes.push({
//                         id: row.note_id,
//                         text: row.notes,
//                         voice_note: row.voicenote,
//                         created_at: row.note_created_at,
//                     });
//                 }
//             }

//             // Add file if exists
//             if (row.file_id) {
//                 if (!contact.files.some(file => file.id === row.file_id)) {
//                     contact.files.push({
//                         id: row.file_id,
//                         filename: row.filename,
//                         file_url: row.file_url, 
//                     }) 
//                 }  
//             }
//         });

//         // Sort notes by note_time in descending order (newest first)
//         contact.notes.sort((a, b) => new Date(b.note_time) - new Date(a.note_time));
//         // Count tasks, meetings, notes, and files
//         contact.count = {
//             tasks: contact.tasks.length,
//             meetings: contact.meetings.length,
//             notes: contact.notes.length,
//             sales_accounts: contact.sales_account? 1 : 0,
//             files: contact.files.length
//         };
//         return contact;
//     } catch (error) {
//         console.error('Error in findById:', error);
//         throw new Error(`Failed to fetch contact: ${error.message}`);
//     }
// }

async function findById(id) {
    try {
     
        const query =  `SELECT * FROM contacts WHERE id = ?`
        const [rows] = await db.execute(query, [id]);
        if (!rows.length) return null;

        const [taskCount] = await db.execute('SELECT COUNT(*) AS count FROM tasks WHERE related_to = ? AND related_type = "contacts" AND active = 1',  [id]);
        const [meetingCount] = await db.execute('SELECT COUNT(*) AS count FROM salesactivitydatas WHERE salesactivities_id = 3 AND active = 1 AND targetable_id = ? AND targetable_type = "contacts"', [id]);
        // const [fileCount] = await db.execute('SELECT COUNT(*) AS count FROM files WHERE belongsto_id = ? AND belongsto = "contact" AND is_deleted = 0', [id]);
        const [noteCount] = await db.execute('SELECT COUNT(*) AS count FROM notes WHERE belongsto_id = ? AND belongsto = "contacts" AND is_deleted = 0', [id]);     

        const [fields] = await db.execute('SELECT * FROM contactfields WHERE active=1 ORDER BY id');
        const [choices] = await db.execute('SELECT * FROM customselectoptions');
        
        // Process fields with lookup handling
        const contacts = await Promise.all(fields.map(async field => {
            // Handle regular choices
            field.choices = choices.filter(choice => choice.contactfield_id === field.id);
            
            // Set field value
            field.field_value = rows[0][field.field_name];
            
            // Handle custom fields
            if (field.custom_field === 1 && rows[0].custom_field) {
                const customFields = JSON.parse(rows[0].custom_field);
                field.field_value = customFields.find(cf => 
                    cf.hasOwnProperty(field.field_name))?.[field.field_name] || null;
            }

            // Handle lookup type fields
            if (field.field_type.toLowerCase() === 'lookup' && field.lookup_type && field.lookup_column) {
                try {
                    const [lookupData] = await db.execute(
                        `SELECT id, ${field.lookup_column} FROM ${field.lookup_type} WHERE active = 1`
                    );

                    // Format lookup data as choices
                    field.choices = lookupData.map(item => ({
                        id: item.id,
                        contactfield_id: field.id,
                        custom_option: item[field.lookup_column]
                    }));

                } catch (error) {
                    console.error(`Error fetching lookup data for field ${field.field_name}:`, error);
                    field.choices = [];
                }
            }

            return field;
        }));

        const contact = {
            id: id.toString(),
            contacts,
            sales_account: null,
            count:{}
        };
        if(rows[0].sales_account_id){            
            const [sales_accounts] = await db.execute('SELECT * FROM sales_accounts WHERE id = ? AND active = 1', [rows[0].sales_account_id]);  
            contact.sales_account = sales_accounts[0];
        }
        contact.count = {
            tasks: taskCount[0].count,
            meetings: meetingCount[0].count,
            notes: noteCount[0].count,
            sales_accounts: rows[0].sales_account_id? 1 : 0,
            // files: fileCount[0].count
        };
        console.log(contact);
        // rows.forEach(row => {
        //     // Add task if exists
        //     if ((row.task_id)&&(row.task_status!=0)) {
        //         if (!contact.tasks.some(task => task.id === row.task_id)) {
        //             contact.tasks.push({
        //                 id: row.task_id,
        //                 title: row.task_title,
        //                 description: row.task_description,
        //                 status: row.task_status
        //             });
        //         }
        //     }
            
        //     // Add sales account if exists
        //     if (row.sales_account_id && !contact.sales_account) {
        //         contact.sales_account = {
        //             id: row.sales_account_id,
        //             name: row.sales_account_name
        //         };
        //     }
            
        //     // Add meeting if exists
        //     if (row.meeting_id) {
        //         if (!contact.meetings.some(meeting => meeting.id === row.meeting_id)) {
        //             contact.meetings.push({
        //                 id: row.meeting_id,
        //                 title: row.meeting_title,
        //                 description: row.meeting_description,
        //                 status: row.meeting_status,
        //                 start_date: row.start_date,
        //                 start_time: row.start_time,
        //                 end_date: row.end_date,
        //                 end_time: row.end_time
        //             });
        //         }
        //     }

        //     // Add note if exists
        //     if (row.note_id && !row.is_deleted) {
        //         if (!contact.notes.some(note => note.id === row.note_id)) {
        //             contact.notes.push({
        //                 id: row.note_id,
        //                 text: row.notes,
        //                 voice_note: row.voicenote,
        //                 created_at: row.note_created_at,
        //             });
        //         }
        //     }

        //     // Add file if exists
        //     if (row.file_id) {
        //         if (!contact.files.some(file => file.id === row.file_id)) {
        //             contact.files.push({
        //                 id: row.file_id,
        //                 filename: row.filename,
        //                 file_url: row.file_url, 
        //             }) 
        //         }  
        //     }
        // });

        // // Sort notes by note_time in descending order (newest first)
        // contact.notes.sort((a, b) => new Date(b.note_time) - new Date(a.note_time));
        // // Count tasks, meetings, notes, and files
        // contact.count = {
        //     tasks: contact.tasks.length,
        //     meetings: contact.meetings.length,
        //     notes: contact.notes.length,
        //     sales_accounts: contact.sales_account? 1 : 0,
        //     files: contact.files.length
        // };
        return contact;
    } catch (error) {
        console.error('Error in findById:', error);
        throw new Error(`Failed to fetch contact: ${error.message}`);
    }
}


async function update(id, Data) {
    const { contact: allFields } = Data;

    const uniqueFields = Array.from(allFields).filter(field => field.unique_field == 1 || field.unique_field == 'on');
    for (const field of allFields) {
        if (uniqueFields.some(uf => uf.field_name === field.field_name) && field.field_value) {
            // Check if value already exists
            const [existing] = await db.execute(
                `SELECT id FROM contacts WHERE ${field.field_name} = ? AND active = 1`,
                [field.field_value]
            );
            console.log(existing, "existing");
            if (existing.length > 0 && existing[0].id != id) {
                console.log(`Contact with ${field.field_name} "${field.field_value}" already exists`);
                return {
                    status: 409,
                    success: false,
                    message: `Contact with ${field.field_name} "${field.field_value}" already exists`,
                };
            }
        }
    }

    const custom_fields = Array.from(allFields).filter(field => field.custom_field == 1).map(field => ({ [field.field_name]: field.field_value }))
    const standardFields = Array.from(allFields).filter(field => field.custom_field == 0)

    // Prepare standard fields for update
    const updateFields = standardFields
        .map(field => `${field.field_name} = ?`)
        .concat(['custom_field = ?'])
        .join(', ');
    const updateValues = [...standardFields.map(field => field.field_value), JSON.stringify(custom_fields)];
    const [result] = await db.execute(
        `UPDATE contacts SET ${updateFields} WHERE id = ?`,
        [...updateValues, id]
    );
    return result.affectedRows > 0;
}

async function deleteRecord (id) {
    const [result] = await db.execute('UPDATE contacts SET active = 0 WHERE id = ?', [id]);
    return result.affectedRows > 0;
}
async function findByContactId(contactId) {
    try {
        const [rows] = await db.execute(
            `SELECT t.*,  
                u.name AS owner_name,
                COALESCE(t.task_type, 'General') AS task_type_name
            FROM tasks t
            LEFT JOIN users u ON t.user_id = u.id
            WHERE t.active=1 and LOWER(t.related_type) = 'contacts'
            AND t.related_to = ?
            ORDER BY t.due_date DESC, t.due_time DESC`,
            [contactId]
        );

        return {
            status: 200,
            message: "Tasks fetched successfully",
            tasks: rows.map(task => ({
                id: task.id,
                title: task.title,
                description: task.description,
                task_type: task.task_type_name,
                related_type: task.related_type,
                related_to: task.related_to,
                mobile_number: task.mobile_number,
                due_date: formatDate(task.due_date) ? formatDate(task.due_date).replaceAll('/', '-').split('-').reverse().join('-') : null,
                due_time: formatTime(task.due_time),
                mark_as_complete: task.mark_as_complete,
                completed_date: formatDate(task.completed_date) ? formatDate(task.completed_date).replaceAll('/', '-').split('-').reverse().join('-') : null,
                completed_time: formatTime(task.completed_time),
                collaboratoes: task.collaboratoes,
                owner: task.user_id,
                user_id: task.user_id,
                owner_name: task.owner_name || '',
                outcome: task.outcome,
                active: task.active,
                snooze_date: formatDate(task.snooze_date) ? formatDate(task.snooze_date).replaceAll('/', '-').split('-').reverse().join('-') : null,
                snooze_time: formatTime(task.snooze_time),
                created_at: formatDateTime(task.created_at),
                updated_at: formatDateTime(task.updated_at),
                status: task.mark_as_complete ? 'Completed' : 'Pending'
            }))
        };
    } catch (error) {
        console.error('Detailed error:', error);
        throw new Error(`Failed to fetch contact tasks: ${error.message}`);
    }
}
async function findContactMeetings(contactId) {
    try {
        const [meetings] = await db.execute(
            `SELECT 
                sad.*,
                u.name as owner_name,
                COALESCE(sao.outcome, '') as outcome_name,
                sa.activity_name,
                sa.activity_icon
             FROM salesactivitydatas sad
             LEFT JOIN users u ON sad.owner_id = u.id
             LEFT JOIN salesactivityoutcomes sao ON sad.outcome = sao.id
             LEFT JOIN salesactivities sa ON sad.salesactivities_id = sa.id
             WHERE sad.targetable_type = 'contacts'
             AND sad.targetable_id = ?
             AND sad.salesactivities_id = 3 
             AND sad.active = 1
             ORDER BY sad.start_date DESC, sad.start_time DESC`,
             [contactId]
             //  AND sad.active=0
        );

        return meetings.map(meeting => ({
            id: meeting.id,
            title: meeting.title,
            description: meeting.description,
            activity_name: meeting.activity_name,
            activity_icon: meeting.activity_icon,
            activity_title: meeting.activity_title,
            start_date: meeting.start_date,
            end_date: meeting.end_date,
            start_time: formatTime(meeting.start_time),
            end_time: formatTime(meeting.end_time),
            duration: meeting.duration,
            outcome: meeting.outcome_name,
            status: meeting.status,
            priority: meeting.priority,
            location: meeting.location,
            owner_id: meeting.owner_id,
            owner_name: meeting.owner_name,
            owner_email: meeting.owner_email,
            owner_mobile: meeting.owner_mobile,
            salesactivities_id: meeting.salesactivities_id,
            targetable_type: meeting.targetable_type,
            targetable_id: meeting.targetable_id,
            transport_mode: meeting.transport_mode,
            distance: meeting.distance,
            claim_amount: meeting.claim_amount,
            checkin_location: meeting.checkin_location,
            checkin_datetime: meeting.checkin_datetime,
            checkout_location: meeting.checkout_location,
            checkout_datetime: meeting.checkout_datetime,
            custom_field: meeting.custom_field ? JSON.parse(meeting.custom_field) : null,
            allow_reschedule: meeting.allow_reschedule,
            allow_complete: meeting.allow_complete,
            // active: meeting.active,
            created_at: meeting.created_at,
            updated_at: meeting.updated_at
        }));

    } catch (error) {
        console.error('Error in findContactMeetings:', error);
        throw new Error(`Failed to fetch contact meetings: ${error.message}`);
    }
}

// Helper functions for date/time formatting
function formatDate(date) {
    if (!date) return null;
    return new Date(date).toLocaleDateString('en-GB', {
        day: '2-digit',
        month: '2-digit',
        year: 'numeric'
    });
}

function formatTime(time) {
    if (!time) return null;
    return new Date(`1970-01-01T${time}`).toLocaleTimeString('en-US', {
        hour: '2-digit',
        minute: '2-digit',
        hour12: true
    });
}
function formatDateTime(datetime) {
    if (!datetime) return null;
    return new Date(datetime).toISOString();
}

// Function to get contact relations (accounts and deals)
// This function retrieves related accounts and deals for a given contact ID
async function getContactRelations(contactId) {
    try {
        // Get related account
        const accountQuery = `
            SELECT 
                a.id,
                a.name,
                a.website,
                a.phone,
                a.created_at
            FROM accounts a
            INNER JOIN contacts c ON c.sales_accounts = a.id
            WHERE c.id = ? AND c.active = 1 AND a.active = 1`;

        // Get related deals
        const dealsQuery = `
            SELECT 
                d.id,
                d.name,
                d.amount,
                d.probability,
                d.deal_stage_id,
                ds.deal_stage as stage_name,
                d.created_at,
                a.name as account_name
            FROM deals d
            LEFT JOIN dealstages ds ON d.deal_stage_id = ds.id
            LEFT JOIN accounts a ON d.sales_account_id = a.id
            WHERE d.contacts = ? AND d.active = 1
            ORDER BY d.created_at DESC`;

        // Execute both queries
        const [accounts] = await db.execute(accountQuery, [contactId]);
        const [deals] = await db.execute(dealsQuery, [contactId]);

        // Get deals summary
        const [dealsSummary] = await db.execute(
            `SELECT 
                COUNT(*) as total_deals,
                SUM(amount) as total_value,
                SUM(CASE WHEN deal_stage_id NOT IN (SELECT id FROM dealstages WHERE deal_stage IN ('Won', 'Lost')) 
                    THEN 1 ELSE 0 END) as open_deals,
                SUM(CASE WHEN deal_stage_id NOT IN (SELECT id FROM dealstages WHERE deal_stage IN ('Won', 'Lost')) 
                    THEN amount ELSE 0 END) as open_value,
                SUM(CASE WHEN deal_stage_id IN (SELECT id FROM dealstages WHERE deal_stage = 'Won') 
                    THEN 1 ELSE 0 END) as won_deals,
                SUM(CASE WHEN deal_stage_id IN (SELECT id FROM dealstages WHERE deal_stage = 'Won') 
                    THEN amount ELSE 0 END) as won_value
            FROM deals 
            WHERE contacts = ? AND active = 1`,
            [contactId]
        );

        return {
            account: accounts.length > 0 ? {
                id: accounts[0].id,
                name: accounts[0].name,
                website: accounts[0].website,
                phone: accounts[0].phone,
                created_at: formatDateTime(accounts[0].created_at)
            } : null,
            deals: {
                summary: {
                    total_deals: dealsSummary[0].total_deals || 0,
                    total_value: dealsSummary[0].total_value || 0,
                    open_deals: dealsSummary[0].open_deals || 0,
                    open_value: dealsSummary[0].open_value || 0,
                    won_deals: dealsSummary[0].won_deals || 0,
                    won_value: dealsSummary[0].won_value || 0
                },
                deals: deals.map(deal => ({
                    id: deal.id,
                    name: deal.name,
                    amount: deal.amount,
                    probability: deal.probability,
                    stage: {
                        id: deal.deal_stage_id,
                        name: deal.stage_name
                    },
                    account_name: deal.account_name,
                    created_at: formatDateTime(deal.created_at)
                }))
            }
        };

    } catch (error) {
        console.error('Error in getContactRelations:', error);
        throw new Error(`Failed to fetch contact relations: ${error.message}`);
    }
}

async function getLifecycles() {
    const [rows] = await db.execute(
        'SELECT * FROM lifecyclestages WHERE active = 1 ORDER BY id DESC'
    );
    return rows;
}

async function createLifecycle(data) {
    const [result] = await db.execute(
        `INSERT INTO lifecyclestages (
            name,
            description,
            active,
            created_at,
            updated_at
        ) VALUES (?, ?, ?, NOW(), NOW())`,
        [
            data.name,
            data.description || null,
            data.active || 1
        ]
    );
    return result.insertId;
}

async function updateLifecycle(id, data) {
    const [result] = await db.execute(
        `UPDATE lifecyclestages 
         SET name = ?,
             description = ?,
             active = ?,
             updated_at = NOW()
         WHERE id = ?`,
        [
            data.name,
            data.description || null,
            data.active || 1,
            id
        ]
    );
    return result.affectedRows > 0;
}

async function deleteLifecycle(id) {
    const [result] = await db.execute(
        'UPDATE lifecyclestages SET active = 0 WHERE id = ?',
        [id]
    );
    return result.affectedRows > 0;
}

async function getLifecycleStages() {
    // Get all lifecycles with their stages
    const [rows] = await db.execute(`
        SELECT 
            ls.id as lifecycle_id,
            ls.stage as lifecycle_name,
            lst.id as stage_id,
            lst.status,
            lst.order,
            lst.active
        FROM lifecyclestages ls
        LEFT JOIN lifecyclestatuses lst ON ls.id = lst.lifecyclestage_id
        WHERE ls.active = 1
        ORDER BY ls.id, lst.order
    `);

    // Transform the flat results into grouped structure
    const groupedStages = rows.reduce((acc, row) => {
        if (!acc[row.lifecycle_id]) {
            acc[row.lifecycle_id] = {
                lifecycle_id: row.lifecycle_id,
                lifecycle_name: row.lifecycle_name,
                stages: []
            };
        }
        
        if (row.stage_id) {  // Only add if stage exists
            acc[row.lifecycle_id].stages.push({
                id: row.stage_id,
                status: row.status,
                order: row.order,
                active: row.active
            });
        }
        
        return acc;
    }, {});

    // Convert to array and return
    return Object.values(groupedStages);
}

async function createLifecycleStage(data) {
    const [result] = await db.execute(
        `INSERT INTO lifecyclestatuses (
            status,
            lifecyclestage_id,
            probability,
            order,
            active,
            created_at,
            updated_at
        ) VALUES (?, ?, ?, ?, ?, NOW(), NOW())`,
        [
            data.status,
            data.lifecyclestage_id,
            data.probability,
            data.order,
            data.active || 1
        ]
    );
    return result.insertId;
}

async function updateLifecycleStages(inputData) {
    const connection = await db.getConnection();
    
    try {
        await connection.beginTransaction();

        // Convert input to array if it's a single object
        const dataArray = Array.isArray(inputData) ? inputData : [inputData];
        const finalResults = [];

        for (const data of dataArray) {
            const { lifecycle, stages } = data;
            let lifecycleId = lifecycle?.id;

            // Handle lifecycle update/creation first
            if (lifecycle) {
                if (lifecycle.id) {
                    // Update existing lifecycle
                    const [lifecycleResult] = await connection.execute(
                        `UPDATE lifecyclestages 
                        SET stage = ?,
                            active = ?,
                            updated_at = NOW()
                        WHERE id = ?`,
                        [
                            lifecycle.stage,
                            lifecycle.active ?? 1,
                            lifecycle.id
                        ]
                    );
                } else {
                    // Create new lifecycle
                    const [lifecycleResult] = await connection.execute(
                        `INSERT INTO lifecyclestages (
                            stage,
                            active,
                            created_at,
                            updated_at
                        ) VALUES (?, ?, NOW(), NOW())`,
                        [
                            lifecycle.stage,
                            lifecycle.active ?? 1
                        ]
                    );
                    lifecycleId = lifecycleResult.insertId;
                }
            }

            // Update stages with the correct lifecycle ID
            const updatedStages = stages.map(stage => ({
                status: stage.status,
                lifecyclestage_id: stage.lifecyclestage_id || lifecycleId,
                order: parseInt(stage.order),
                active: stage.active ?? 1,
                id: stage.id || null
            }));

            // Group stages by lifecycle
            const lifecycleStages = updatedStages.reduce((acc, stage) => {
                if (!acc[stage.lifecyclestage_id]) {
                    acc[stage.lifecyclestage_id] = [];
                }
                acc[stage.lifecyclestage_id].push(stage);
                return acc;
            }, {});

            // Validate sequential orders for each lifecycle
            for (const lId in lifecycleStages) {
                const lifecycleStagesList = lifecycleStages[lId];
                
                // Create array of orders for checking sequence
                const orders = lifecycleStagesList.map(stage => stage.order);
                
                // Check if orders are sequential and in order
                for (let i = 0; i < orders.length; i++) {
                    if (orders[i] !== i + 1) {
                        const expectedOrder = i + 1;
                        const foundOrder = orders[i];
                        const stageName = lifecycleStagesList.find(s => s.order === foundOrder)?.status;
                        
                        throw new Error(
                            `Invalid stage order in lifecycle ${lId}. ` +
                            `Found stage "${stageName}" at position ${i + 1} with order ${foundOrder}. ` +
                            `Expected order ${expectedOrder}. ` +
                            `Stages must be in order: 1,2,3,4,5...`
                        );
                    }
                }

                // Check for gaps
                const maxOrder = Math.max(...orders);
                if (maxOrder !== orders.length) {
                    throw new Error(
                        `Invalid stage sequence in lifecycle ${lId}. ` +
                        `Found ${orders.length} stages but highest order is ${maxOrder}. ` +
                        `Orders must be sequential from 1 to ${orders.length} without gaps.`
                    );
                }
            }

            // Check for duplicate orders within the same lifecycle
            const orderMap = new Map();
            updatedStages.forEach(stage => {
                const key = `${stage.lifecyclestage_id}-${stage.order}`;
                if (orderMap.has(key)) {
                    const existingStage = updatedStages.find(s => 
                        s.lifecyclestage_id === stage.lifecyclestage_id && 
                        s.order === stage.order && 
                        s !== stage
                    );
                    throw new Error(
                        `Duplicate order ${stage.order} found in lifecycle ${stage.lifecyclestage_id}. ` +
                        `Both "${existingStage.status}" and "${stage.status}" have the same order.`
                    );
                }
                orderMap.set(key, true);
            });

            const results = {
                lifecycle: null,
                stages: []
            };
            
            // Get lifecycle details
            if (lifecycleId) {
                const [lifecycleDetails] = await connection.execute(
                    'SELECT * FROM lifecyclestages WHERE id = ?',
                    [lifecycleId]
                );
                results.lifecycle = lifecycleDetails[0];
            }

            // First, get all existing stages for the affected lifecycles
            const lifecycleIds = [...new Set(updatedStages.map(s => s.lifecyclestage_id))];
            const [existingStages] = await connection.execute(
                `SELECT id, lifecyclestage_id, \`order\` 
                FROM lifecyclestatuses 
                WHERE lifecyclestage_id IN (${lifecycleIds.join(',')})
                AND id NOT IN (${updatedStages.filter(s => s.id).map(s => s.id).join(',') || 0})`
            );

            // Temporarily update orders of existing stages to avoid conflicts
            if (existingStages.length > 0) {
                await connection.execute(
                    `UPDATE lifecyclestatuses 
                    SET \`order\` = \`order\` + 1000 
                    WHERE id IN (${existingStages.map(s => s.id).join(',')})`
                );
            }
            
            // Process each stage
            for (const stage of updatedStages) {
                let result;
                if (stage.id) {
                    // Update existing stage
                    [result] = await connection.execute(
                        `UPDATE lifecyclestatuses 
                        SET status = ?,
                            lifecyclestage_id = ?,
                            \`order\` = ?,
                            active = ?,
                            updated_at = NOW()
                        WHERE id = ?`,
                        [
                            stage.status,
                            stage.lifecyclestage_id,
                            stage.order,
                            stage.active,
                            stage.id
                        ]
                    );

                    // Get updated stage details
                    const [updatedStageDetails] = await connection.execute(
                        'SELECT * FROM lifecyclestatuses WHERE id = ?',
                        [stage.id]
                    );

                    results.stages.push({
                        ...updatedStageDetails[0],
                        isNewStage: false
                    });
                } else {
                    // Create new stage
                    [result] = await connection.execute(
                        `INSERT INTO lifecyclestatuses (
                            status,
                            lifecyclestage_id,
                            \`order\`,
                            active,
                            created_at,
                            updated_at
                        ) VALUES (?, ?, ?, ?, NOW(), NOW())`,
                        [
                            stage.status,
                            stage.lifecyclestage_id,
                            stage.order,
                            stage.active || 1
                        ]
                    );

                    // Get new stage details
                    const [newStageDetails] = await connection.execute(
                        'SELECT * FROM lifecyclestatuses WHERE id = ?',
                        [result.insertId]
                    );

                    results.stages.push({
                        ...newStageDetails[0],
                        isNewStage: true
                    });
                }
            }

            // Delete any remaining stages that weren't in the update
            await connection.execute(
                `DELETE FROM lifecyclestatuses 
                WHERE lifecyclestage_id IN (${lifecycleIds.join(',')})
                AND id NOT IN (${results.stages.map(r => r.id).join(',')})`
            );

            finalResults.push(results);
        }

        await connection.commit();
        return finalResults;

    } catch (error) {
        await connection.rollback();
        throw error;
    } finally {
        connection.release();
    }
}

async function deleteLifecycleStage(id) {
    const [result] = await db.execute('DELETE FROM lifecyclestatuses WHERE id = ?', [id]);
    return result;
}


async function createFilterView(viewData) {
    try {
                // Insert product view
        const [viewResult] = await db.execute(
            `INSERT INTO contactviews 
            (name, share_with, accessibility, users) 
            VALUES (?, ?, ?, ?)`,
            [viewData.title, viewData.share_with, viewData.accessibility, viewData.selected_users.join(',') || null]
        );

        const viewId = viewResult.insertId;

        // Insert filters if any
        if (viewData.filter && Array.isArray(viewData.filter)) {
            for (const filter of viewData.filter) {
                if (filter.field_label && filter.field_label.trim() !== '') {
                    await db.execute(
                        `INSERT INTO contactfilters 
                        (view_id, name, field_id, contains, filter, active) 
                        VALUES (?, ?, ?, ?, ?, ?)`,
                        [
                            viewId,
                            filter.field_label,
                            filter.field_name,
                            filter.contains,
                            JSON.stringify(filter.value_id),
                            1
                        ]
                    );
                }
            }
        }

       

        // Get the created view
        const [views] = await db.execute(
            'SELECT * FROM contactviews WHERE id = ?',
            [viewId]
        );

        return views[0];

    } catch (error) {
        console.error('Error in contactviews:', error);
        throw new Error(`Failed to create account view: ${error.message}`);
    }
}

async function deleteFilterView(viewId) {
    try {
        //await db.execute('START TRANSACTION');

        // Delete related filters first
        await db.execute(
            'DELETE FROM contactfilters WHERE view_id = ?',
            [viewId]
        );

        // Delete the view
        const [result] = await db.execute(
            'DELETE FROM contactviews WHERE id = ?',
            [viewId]
        );

        //await db.execute('COMMIT');

        return result.affectedRows > 0;

    } catch (error) {
       // await db.execute('ROLLBACK');
        console.error('Error in contactviews:', error);
        throw new Error(`Failed to delete product view: ${error.message}`);
    }
}

async function getAllFiltersList() {
    try {
        console.log("=== Starting getAllFiltersList Model Function ===");
        console.log("Executing database query for contactviews");
        
        // Get the views with their filters
        const [views] = await db.execute(`
            SELECT cv.*, cf.id as filter_id, cf.name as filter_name, 
                   cf.field_id, cf.contains, cf.filter as filter_value
                            FROM contactviews cv
            LEFT JOIN contactfilters cf ON cv.id = cf.view_id
            `
        );
        
        // Transform the data to match findAll format
        const transformedViews = views.reduce((acc, row) => {
            if (!acc[row.id]) {
                acc[row.id] = {
                    id: row.id,
                    name: row.name,
                    share_with: row.share_with,
                    accessibility: row.accessibility,
                    users: row.users ? row.users.split(',') : [],
                    owner_name: row.owner_name,
                    created_at: formatDateTime(row.created_at),
                    updated_at: formatDateTime(row.updated_at),
                    filters: []
                };
            }
            
            if (row.filter_id) {
                acc[row.id].filters.push({
                    id: row.filter_id,
                    name: row.filter_name,
                    field_id: row.field_id,
                    contains: row.contains,
                    filter: row.filter_value ? JSON.parse(row.filter_value) : [],
                });
            }
            
            return acc;
        }, {});

        const result = Object.values(transformedViews);
        
        console.log("Number of views found:", result.length);
        return result;
        
    } catch (error) {
        console.error("=== Error in getAllFiltersList Model Function ===");
        console.error("Error details:", error);
        throw new Error(`Failed to fetch contact filters: ${error.message}`);
    }
}


// filter
async function getContactListWithFilters(filters, userId, role_id) {
    const connection = await db.getConnection();
    try {
        // Step 1: Get user role and access scope
        const [userRows] = await connection.execute(
            `SELECT r.access_scope, r.id as role_id, u.territory_id 
             FROM users u 
             JOIN roles r ON u.roles_id = r.id 
             WHERE u.id = ?`,
            [userId]
        );

        console.log('User role and access scope:', userRows);
        console.log({filters})

        if (!userRows.length) throw new Error('User role not found');
        const { access_scope, territory_id } = userRows[0];
        console.log('Access scope:', access_scope);
        const { view } = JSON.parse(access_scope)?.contact || {};
        console.log('View:', view);

        let baseQuery = `SELECT p.*, u.name as owner_name 
                         FROM contacts p 
                         LEFT JOIN users u ON p.owner_id = u.id 
                         WHERE p.active = 1`;

        let params = [];

        // Step 2: Apply permission-based filters
        switch (view) {
            case 'owned':
                baseQuery += ' AND p.owner_id = ?';
                params.push(userId);
                break;

            case 'territory':
                if (!territory_id) return [];
                if (role_id === 2) {
                    const territories = territory_id.split(',');
                    const placeholders = territories.map(() => '?').join(',');
                    console.log('Territory IDs:', territories);
                    console.log('Placeholders:', placeholders);
                    baseQuery += ` AND (p.territory_id IN (${placeholders}) OR p.owner_id = ?)`;
                    params.push(...territories);
                    params.push(userId);
                    break;
                } else {
                    const territories = territory_id.split(',');
                    const placeholders = territories.map(() => '?').join(',');
                    console.log('Territory IDs:', territories);
                    console.log('Placeholders:', placeholders);
                    baseQuery += ` AND p.territory_id IN (${placeholders})`;
                    params.push(...territories);
                    break;
                }


            case 'global':
            case 'restricted':
            default:
                if (view === 'restricted') return [];
        }

        // Step 3: Map filter fields
        const fieldMapping = {
            owner: 'p.owner_id',
            created_at: 'p.created_at',
            updated_at: 'p.updated_at'
        };

        // Step 4: Apply filters
        if (filters?.length > 0) {
            for (const filter of filters) {
                const { contains, field_name, value_id } = filter;
                const mappedField = fieldMapping[field_name] || `p.${field_name}`;
                baseQuery = applyFilterToQuery({ baseQuery, contains, mappedField, value_id, params });
            }
        }
        // console.log({baseQuery, params})

        baseQuery += ` ORDER BY p.created_at DESC`; // Prevent overload

        // console.log({baseQuery, params})
        // Step 5: Execute final query
        const [contacts] = await connection.execute(baseQuery, params);

        return contacts;

    } catch (error) {
        console.error('Error in getContactListWithFilters:', error);
        throw error;
    } finally {
        connection.release(); // Always release connection
    }
}

// Helper function to avoid repeating condition logic
function applyFilterToQuery({ baseQuery, contains, mappedField, value_id, params }) {
    switch (contains) {
        case 'contains':
            if (Array.isArray(value_id) && value_id.length) {
                const placeholders = value_id.map(() => '?').join(',');
                baseQuery += ` AND ${mappedField} IN (${placeholders}) `;
                params.push(...value_id);
            }
            break;

        case 'does not contain':
            if (Array.isArray(value_id) && value_id.length) {
                const placeholders = value_id.map(() => '?').join(',');
                baseQuery += ` AND ${mappedField} NOT IN (${placeholders})`;
                params.push(...value_id);
            }
            break;

        case 'is not empty':
            baseQuery += ` AND ${mappedField} IS NOT NULL AND ${mappedField} != ''`;
            break;

        case 'is empty':
            baseQuery += ` AND (${mappedField} IS NULL OR ${mappedField} = '')`;
            break;

        case '< less than':
            baseQuery += ` AND ${mappedField} < ?`;
            params.push(value_id);
            break;

        case '> greater than':
            baseQuery += ` AND ${mappedField} > ?`;
            params.push(value_id);
            break;

        case '= equal to':
            baseQuery += ` AND ${mappedField} = ?`;
            params.push(value_id);
            break;

        case '!= not equal to':
            baseQuery += ` AND ${mappedField} != ?`;
            params.push(value_id);
            break;

        case 'between':
            if (Array.isArray(value_id) && value_id.length === 2) {
                baseQuery += ` AND ${mappedField} BETWEEN ? AND ?`;
                params.push(value_id[0], value_id[1]);
            }
            break;

        case 'last 30 minutes':
            baseQuery += ` AND ${mappedField} >= ?`;
            params.push(dayjs().subtract(30, 'minute').format('YYYY-MM-DD HH:mm:ss'));
            break;

        case 'last 60 minutes':
            baseQuery += ` AND ${mappedField} >= ?`;
            params.push(dayjs().subtract(60, 'minute').format('YYYY-MM-DD HH:mm:ss'));
            break;

        case 'last 2 hours':
            baseQuery += ` AND ${mappedField} >= ?`;
            params.push(dayjs().subtract(2, 'hour').format('YYYY-MM-DD HH:mm:ss'));
            break;

        case 'last 24 hours':
            baseQuery += ` AND ${mappedField} >= ?`;
            params.push(dayjs().subtract(24, 'hour').format('YYYY-MM-DD HH:mm:ss'));
            break;

        case 'last 7 days':
            baseQuery += ` AND ${mappedField} >= ?`;
            params.push(dayjs().subtract(7, 'day').format('YYYY-MM-DD HH:mm:ss'));
            break;

        case 'last 30 days':
            baseQuery += ` AND ${mappedField} >= ?`;
            params.push(dayjs().subtract(30, 'day').format('YYYY-MM-DD HH:mm:ss'));
            break;

        case 'last 90 days':
            baseQuery += ` AND ${mappedField} >= ?`;
            params.push(dayjs().subtract(90, 'day').format('YYYY-MM-DD HH:mm:ss'));
            break;

        case 'last 12 months':
            baseQuery += ` AND ${mappedField} >= ?`;
            params.push(dayjs().subtract(12, 'month').format('YYYY-MM-DD HH:mm:ss'));
            break;
    }
    // console.log({baseQuery, params})
    return baseQuery
}


// get all lifecycles by id
async function getLifecyclesById(id) {
    const [rows] = await db.execute('SELECT * FROM lifecyclestatuses WHERE lifecyclestage_id = ? ORDER BY \`order\`', [id]);
    return rows;
}


async function updateFilterView(viewData) {
    console.log("updateFilterView inits "+JSON.stringify(viewData));
    try {
                // Insert product view
                const selectedUsers = Array.isArray(viewData.users) && viewData.users.length > 0
                ? viewData.users.join(',')
                : null;
              
              const [viewResult] = await db.execute(
                `UPDATE contactviews 
                 SET name = ?, share_with = ?, accessibility = ?, users = ?
                 WHERE id = ?`,
                [viewData.title, viewData.share_with, viewData.accessibility, selectedUsers, viewData.id]
              );

        const viewId = viewResult.insertId;

       console.log("viewResultsds "+viewResult);

       

        // Get the created view
        const [views] = await db.execute(
            'SELECT * FROM contactviews WHERE id = ?',
            [viewData.id]
        );

        return views[0];

    } catch (error) {
        console.error('Error in contactviews:', error);
        throw new Error(`Failed to create account view: ${error.message}`);
    }
}



async function removerelated(id, module, related, related_id) {
    try {
        const modules = module.toLowerCase();
        let query;
        let params = [id, related_id];

        switch (modules) {
            case 'deals':
                // For deals, remove related contact or account
                if (related === 'contact') {
                    query = `UPDATE deals SET contacts = NULL WHERE id = ? AND contacts = ? AND active = 1`;
                    params = [id, related_id];
                } else if (related === 'account') {
                    query = `UPDATE deals SET sales_account_id = NULL WHERE id = ? AND sales_account_id = ? AND active = 1`;
                    params = [id, related_id];
                } else {
                    throw new Error(`Invalid related field for deals: ${related}. Supported: contact, account`);
                }
                break;

            case 'contacts':
                // For contacts, remove related account
                if (related === 'account') {
                    query = `UPDATE contacts SET sales_accounts = NULL WHERE id = ? AND sales_accounts = ? AND active = 1`;
                    params = [id, related_id];
                } else if (related === 'deals') {
                    query = `UPDATE deals SET contacts = NULL WHERE contacts = ? AND id = ? AND active = 1`;
                    params = [id, related_id];
                } else {
                    throw new Error(`Invalid related field for contacts: ${related}. Supported: account`);
                }
                break;

            case 'accounts':
                // For accounts, we can't directly remove contacts/deals, but we can update the relationship
                if (related === 'contact') {
                    // Remove specific contact from account
                    query = `UPDATE contacts SET sales_accounts = NULL WHERE sales_accounts = ? AND id = ? AND active = 1`;
                    params = [id, related_id];
                } else if (related === 'deal') {
                    // Remove specific deal from account
                    query = `UPDATE deals SET sales_account_id = NULL WHERE sales_account_id = ? AND id = ? AND active = 1`;    
                    params = [id, related_id];
                } else {
                    throw new Error(`Invalid related field for accounts: ${related}. Supported: contact, deal`);
                }
                break;

            default:
                throw new Error(`Unsupported module type: ${module}`);
        }

        const [result] = await db.execute(query, params);
    
        return {
            success: result.affectedRows > 0,
            affectedRows: result.affectedRows,
            message: result.affectedRows > 0 
                ? `${related} relationship removed successfully from ${module}`
                : `No ${module} found with ID ${id} or relationship already removed`
        };

    } catch (error) {
        console.error('Error removing related module:', error);
        throw new Error(`Error removing related module: ${error.message}`);
    }
}

// Bulk import contacts with custom fields
async function bulkImport(csvData, columnMappings, userId) {
    const connection = await db.getConnection();
    try {
        await connection.beginTransaction();
        console.log("bulkImport inits "+JSON.stringify(csvData));

        const results = {
            total: csvData.length,
            successful: 0,
            failed: 0,
            errors: [],
            errorCsvData: [],
            errorCsvFile: null
        };

        // Pre-fetch validation data
        const [users] = await connection.execute('SELECT id, record_id FROM users WHERE active = 1');
        const [accounts] = await connection.execute('SELECT id FROM accounts WHERE active = 1');
        
        // Create lookup maps
        const userMap = new Map();
        users.forEach(user => {
            if (user.record_id) {
                userMap.set(user.record_id.toString(), user.id);
            }
        });
        
        const accountIds = new Set(accounts.map(acc => acc.id.toString()));

        for (let i = 0; i < csvData.length; i++) {
            const row = csvData[i];
            const rowNumber = i + 2; // +2 because CSV has header and arrays are 0-indexed

            try {
                // Prepare contact data
                const contactData = {
                    owner_id: userId,
                    created_at: new Date().toISOString().slice(0, 19).replace('T', ' '),
                    updated_at: new Date().toISOString().slice(0, 19).replace('T', ' ')
                };

                // Separate standard fields and custom fields
                const standardFields = [];
                const customFields = [];

                // Process each mapped column
                for (const mapping of columnMappings) {
                    if (mapping.isMapped && mapping.apiField) {
                        const csvValue = row[mapping.csvColumn];
                        const field = mapping.apiField;

                        // Skip if no value provided
                        if (csvValue === undefined || csvValue === null || csvValue === '') {
                            continue;
                        }

                        let processedValue = csvValue.toString().trim();

                        // Handle different field types based on field_name and field_type
                        switch (field.field_type.toLowerCase()) {
                            case 'text field':
                            case 'textarea':
                                processedValue = csvValue.toString().trim();
                                break;

                            case 'number':
                                const numValue = parseFloat(csvValue);
                                if (!isNaN(numValue)) {
                                    processedValue = numValue;
                                }
                                break;

                            case 'datepicker':
                                // Handle various date formats
                                if (csvValue) {
                                    const date = new Date(csvValue);
                                    if (!isNaN(date.getTime())) {
                                        processedValue = date.toISOString().split('T')[0];
                                    }
                                }
                                break;

                            case 'dropdown':
                            case 'multiselect':
                                // For dropdown/multiselect, query the choices table directly
                                if (field.field_name) {
                                    const [choices] = await connection.execute(
                                        'SELECT id, custom_option FROM customselectoptions WHERE contactfield_id  = ? AND custom_option = ?',
                                        [field.id, csvValue.toString().trim()]
                                    );
                                    
                                    if (choices.length > 0) {
                                        processedValue = choices[0].id;
                                    } else {
                                        throw new Error(`Invalid dropdown value: "${csvValue}" for field "${field.field_label}" - Choice not found`);
                                    }
                                }
                                break;

                            case 'lookup':
                                // Handle lookup fields based on field_name
                                if (field.field_name === 'owner_id' || field.field_name === 'created_by' || field.field_name === 'modified_by') {
                                    // For user lookup fields, get user ID from users table based on record_id
                                    const userRecordId = csvValue.toString().trim();
                                    const mappedUserId = userMap.get(userRecordId);
                                    if (!mappedUserId) {
                                        throw new Error(`Invalid ${field.field_name}: "${userRecordId}" - User not found`);
                                    }
                                    processedValue = mappedUserId;
                                } else if (field.field_name === 'sales_accounts' || field.field_name === 'sales_account_id') {
                                    // For sales account lookup, check if the value looks like a record_id
                                    if (csvValue.toString().includes('zcrm_') || /^\d+$/.test(csvValue.toString())) {
                                        // This looks like a record_id, try to find by record_id first
                                        const [accountResult] = await connection.execute(
                                            'SELECT id FROM accounts WHERE record_id = ? AND active = 1',
                                            [csvValue.toString().trim()]
                                        );
                                        
                                        if (accountResult.length > 0) {
                                            processedValue = accountResult[0].id;
                                        } else {
                                            // If not found by record_id, try by name
                                            const [accountResult2] = await connection.execute(
                                                'SELECT id FROM accounts WHERE name = ? AND active = 1',
                                                [csvValue.toString().trim()]
                                            );
                                            
                                            if (accountResult2.length > 0) {
                                                processedValue = accountResult2[0].id;
                                            } else {
                                                throw new Error(`Invalid sales account: "${csvValue}" - Account not found`);
                                            }
                                        }
                                    } else {
                                        // This looks like a display name, query by name
                                        const [accountResult] = await connection.execute(
                                            'SELECT id FROM accounts WHERE name = ? AND active = 1',
                                            [csvValue.toString().trim()]
                                        );
                                        
                                        if (accountResult.length > 0) {
                                            processedValue = accountResult[0].id;
                                        } else {
                                            throw new Error(`Invalid sales account: "${csvValue}" - Account not found`);
                                        }
                                    }
                                } else if (field.lookup_type && field.lookup_column) {
                                    // For other lookup fields, check if the value looks like a record_id
                                    if (csvValue.toString().includes('zcrm_') || /^\d+$/.test(csvValue.toString())) {
                                        // This looks like a record_id, try to find by record_id first
                                        const [lookupResult] = await connection.execute(
                                            `SELECT id FROM ${field.lookup_type} WHERE record_id = ? AND active = 1`,
                                            [csvValue.toString().trim()]
                                        );
                                        
                                        if (lookupResult.length > 0) {
                                            processedValue = lookupResult[0].id;
                                        } else {
                                            // If not found by record_id, try by the lookup_column
                                            const [lookupResult2] = await connection.execute(
                                                `SELECT id FROM ${field.lookup_type} WHERE ${field.lookup_column} = ? AND active = 1`,
                                                [csvValue.toString().trim()]
                                            );
                                            
                                            if (lookupResult2.length > 0) {
                                                processedValue = lookupResult2[0].id;
                                            } else {
                                                throw new Error(`Invalid ${field.field_label}: "${csvValue}" - Record not found in ${field.lookup_type}`);
                                            }
                                        }
                                    } else {
                                        // This looks like a display name, query by lookup_column
                                        const [lookupResult] = await connection.execute(
                                            `SELECT id FROM ${field.lookup_type} WHERE ${field.lookup_column} = ? AND active = 1`,
                                            [csvValue.toString().trim()]
                                        );
                                        
                                        if (lookupResult.length > 0) {
                                            processedValue = lookupResult[0].id;
                                        } else {
                                            throw new Error(`Invalid ${field.field_label}: "${csvValue}" - Record not found in ${field.lookup_type}`);
                                        }
                                    }
                                } else {
                                    throw new Error(`Lookup field "${field.field_label}" has no lookup configuration`);
                                }
                                break;

                            default:
                                processedValue = csvValue.toString().trim();
                        }

                        // Separate standard and custom fields
                        if (field.custom_field === 1) {
                            customFields.push({ [field.field_name]: processedValue });
                        } else {
                            contactData[field.field_name] = processedValue;
                        }
                    }
                }

                // Add custom fields as JSON string - format as array of objects
                if (customFields.length > 0) {
                    contactData.custom_field = JSON.stringify(customFields);
                }

                // Add active status
                contactData.active = 1;

                // Insert the contact
                const keys = Object.keys(contactData);
                const values = Object.values(contactData);
                const placeholders = keys.map(() => '?').join(', ');

                const [result] = await connection.execute(
                    `INSERT INTO contacts (${keys.join(', ')}) VALUES (${placeholders})`,
                    values
                );

                results.successful++;

            } catch (error) {
                results.failed++;
                const errorRecord = {
                    row: rowNumber,
                    error: error.message,
                    data: row
                };
                results.errors.push(errorRecord);
                
                // Add error record to CSV data
                const errorCsvRow = {
                    ...row,
                    'Error Row': rowNumber,
                    'Error Message': error.message
                };
                results.errorCsvData.push(errorCsvRow);
            }
        }

        await connection.commit();

        // Generate error CSV file if there are errors
        if (results.errorCsvData.length > 0) {
            const fs = require('fs');
            const path = require('path');
            
            // Create filename with timestamp
            const timestamp = new Date().toISOString().replace(/[:.]/g, '-').slice(0, 19);
            const filename = `contacts_bulk_import_errors_${timestamp}.csv`;
            const filepath = path.join(__dirname, '../public/uploads', filename);
            
            // Ensure uploads directory exists
            const uploadsDir = path.dirname(filepath);
            if (!fs.existsSync(uploadsDir)) {
                fs.mkdirSync(uploadsDir, { recursive: true });
            }
            
            // Get CSV headers from the original CSV data (first row) plus error columns
            const originalHeaders = Object.keys(csvData[0] || {});
            const errorHeaders = ['Error Row', 'Error Message'];
            const headers = [...originalHeaders, ...errorHeaders];
            
            // Create CSV content
            let csvContent = headers.join(',') + '\n';
            
            results.errorCsvData.forEach(row => {
                const values = headers.map(header => {
                    let value = '';
                    if (header === 'Error Row') {
                        value = row['Error Row'] || '';
                    } else if (header === 'Error Message') {
                        value = row['Error Message'] || '';
                    } else {
                        value = row[header] || '';
                    }
                    // Escape quotes and wrap in quotes if contains comma or quote
                    const escapedValue = value.toString().replace(/"/g, '""');
                    return `"${escapedValue}"`;
                });
                csvContent += values.join(',') + '\n';
            });
            
            // Write CSV file
            fs.writeFileSync(filepath, csvContent);
            
            // Set the file path for response
            results.errorCsvFile = `/uploads/${filename}`;
        }

        return results;

    } catch (error) {
        await connection.rollback();
        throw error;
    } finally {
        connection.release();
    }
}



module.exports = {
    create,
    findAll,
    findById,
    update,
    deleteRecord,
    findContactMeetings,
    findByContactId,
    getContactRelations,
    getLifecycles,
    createLifecycle,
    updateLifecycle,
    deleteLifecycle,
    getLifecycleStages,
    createLifecycleStage,
    updateLifecycleStages,
    deleteLifecycleStage,
    getAllFiltersList,
    createFilterView,
    deleteFilterView,
    getContactListWithFilters,
    getLifecyclesById,
    updateFilterView,
    removerelated,
    bulkImport
};