const db = require('../config/db');
const { CONTAINS_LIST } = require('../config/utils');

async function create(fieldData) {
    // Provide default values and handle undefined
    const {
        custom_field = false,
        field_label = '',
        field_name = '',
        field_type = '',
        required =  '',
        quick_add = '',
        tool_tip = null,
        placeholder = null,
        choices = [],
        read_only = '',
        has_dependent = false,
        contactgroups_id = null,
        active = true,
        order = 0,
        lookup_type = null,
        lookup_column = null
    } = fieldData;

    // Validate required fields
    // if (!field_label || !field_name || !field_type) {
    //     throw new Error('field_label, field_name, and field_type are required');
    // }

    // Convert choices array to JSON string if it exists
    const choicesJson = Array.isArray(choices) && choices.length > 0 ? JSON.stringify(choices) : null;

    // Insert into contactfields table with choices as JSON
    const [result] = await db.execute(
        `INSERT INTO contactfields (
            field_label, field_name, field_type, required, quick_add,
            tool_tip, placeholder, choices, read_only, has_dependent,
            contactgroups_id, active, \`order\`, lookup_type, lookup_column,
            custom_field, created_at, updated_at
        ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW(), NOW())`,
        [
            field_label,
            field_name,
            field_type,
            required ? 1 : 0,
            quick_add ? 1 : 0,
            tool_tip,
            placeholder,
            choicesJson,
            read_only ? 1 : 0,
            has_dependent ? 1 : 0,
            contactgroups_id,
            active ? 1 : 0,
            order,
            lookup_type,
            lookup_column,
            custom_field 
        ]
    );

    // Insert choices into customselectoptions if choices exist
    if (Array.isArray(choices) && choices.length > 0) {
        for (const choice of choices) {
            await db.execute(
                `INSERT INTO customselectoptions (contactfield_id, custom_option, created_at, updated_at) VALUES (?, ?, NOW(), NOW())`,
                [result.insertId, choice]
            );
        }
    }

    return result.insertId;
}

async function findAll(userId) {
    const connection = await db.getConnection();
    try {
        // Get user's role and access scope
        const [userRole] = await connection.execute(
            `SELECT r.access_scope, r.id as role_id, u.territory_id 
             FROM users u 
             JOIN roles r ON u.roles_id = r.id 
             WHERE u.id = ?`,
            [userId]
        );

        if (!userRole || !userRole.length) {
            throw new Error('User role not found');
        }

        const accessScope = JSON.parse(userRole[0].access_scope);

        let [rows] = await db.execute('SELECT * FROM contactfields where active = 1 ORDER BY `order` ASC');
        const [choices] = await db.execute('SELECT * FROM customselectoptions');

        // Process each field
        const processedRows = await Promise.all(rows.map(async field => {
            // Handle regular choices
            field.choices = choices.filter(choice => choice.contactfield_id === field.id);
            field.field_value = null;

            // Handle lookup type fields
            if (field.field_type.toLowerCase() === 'lookup' && field.lookup_type && field.lookup_column) {
                try {
                    let query = `SELECT id, ${field.lookup_column} FROM ${field.lookup_type} WHERE active = 1`;
                    const params = [];

                    // Get view permission for the specific module
                    const modulePermission = field.lookup_type === 'contacts' ? accessScope?.contact?.view :
                                          field.lookup_type === 'accounts' ? accessScope?.sales_account?.view :
                                          field.lookup_type === 'deals' ? accessScope?.deal?.view : 'global';

                    switch (modulePermission) {
                        case 'owned':
                            query += ' AND owner_id = ?';
                            params.push(userId);
                            break;

                        case 'territory':
                            if (userRole[0].territory_id) {
                                const territories = userRole[0].territory_id.split(',');
                                const placeholders = territories.map(() => '?').join(',');
                                query += ` AND territory_id IN (${placeholders})`;
                                params.push(...territories);
                            } else {
                                field.choices = [];
                                return field;
                            }
                            break;

                        case 'restricted':
                            field.choices = [];
                            return field;

                        case 'global':
                        default:
                            // No additional filters for global access
                            break;
                    }

                    // Get data from lookup table with permissions applied
                    const [lookupData] = await connection.execute(query, params);

                    // Format lookup data as choices
                    field.choices = lookupData.map(item => ({
                        id: item.id,
                        contactfield_id: field.id,
                        custom_option: item[field.lookup_column]
                    }));
                } catch (error) {
                    console.error(`Error fetching lookup data for field ${field.field_name}:`, error);
                    field.choices = [];
                }
            }

            return field;
        }));

        return processedRows;

    } catch (error) {
        console.error('Error in findAll:', error);
        throw error;
    } finally {
        connection.release();
    }
}

async function viewAllFilters(userId) {
    try {
        // Validate userId
        if (!userId) {
            throw new Error('User ID is required');
        }

        // Get all product fields
        const [fields] = await db.execute(
            'SELECT * FROM contactfields WHERE active = 1 ORDER BY id'
        );

        if (!fields || fields.length === 0) {
            throw new Error('No product fields found');
        }

        // Get users for lookup fields
        const [users] = await db.execute(
            'SELECT id, name FROM users WHERE active = 1'
        );

        // Get all field choices
        const [choices] = await db.execute(
            'SELECT * FROM customselectoptions'
        );

        // Process each field
        const processedFields = fields.map(field => {
            let values = [];
            let contains_list = [];

            // Set contains_list based on field type
            switch (field.field_type.toLowerCase()) {
                case 'dropdown':
                    contains_list = CONTAINS_LIST.DROPDOWN;
                    // Add dropdown values from choices with id and custom_option
                    values = choices
                        .filter(choice => choice.contactfield_id === field.id)
                        .map(choice => ({
                            id: choice.id,
                            value: choice.value,
                            value: choice.custom_option || 0
                        }));
                    break;

                case 'number':
                    contains_list =CONTAINS_LIST.NUMBERS;
                    break;

                case 'datepicker':
                    contains_list = CONTAINS_LIST.DATE; 
                    break;

                case 'lookup':
                    contains_list = CONTAINS_LIST.LOOKUP;
                    // Add lookup values
                    values = [
                        { id: null, value: "unassigned" },
                        ...users.map(user => ({
                            id: user.id,
                            value: user.id === userId ? "Me" : user.name
                        }))
                    ];
                    break;

                default:
                    contains_list = CONTAINS_LIST.DEFAULT;
            }

            return {
                id: field.id,
                custom_field: field.custom_field || 0,
                field_label: field.field_label,
                field_name: field.field_name,
                field_type: field.field_type,
                values: values,
                contains_list: contains_list
            };
        });

        return processedFields;

    } catch (error) {
        console.error('Error in getAccountFields:', error);
        throw new Error(`Failed to fetch saccount fields: ${error.message}`);
    }
}


async function findById(id) {
    let [rows] = await db.execute('SELECT * FROM contactfields WHERE id = ?', [id]);
    const [choices] = await db.execute('SELECT * FROM customselectoptions');

    if (rows.length === 0) {
        return null;
    }

    const field = rows[0];

    // Handle regular choices
    field.choices = choices.filter(choice => choice.contactfield_id === field.id);
    field.field_value = null;

    // Handle lookup type fields
    if (field.field_type.toLowerCase() === 'lookup' && field.lookup_type && field.lookup_column) {
        try {
            // Get data from lookup table
            const [lookupData] = await db.execute(
                `SELECT id, ${field.lookup_column} FROM ${field.lookup_type} WHERE active = 1`
            );

            // Format lookup data as choices
            field.choices = lookupData.map(item => ({
                id: item.id,
                contactfield_id: field.id,
                custom_option: item[field.lookup_column]
            }));
        } catch (error) {
            console.error(`Error fetching lookup data for field ${field.field_name}:`, error);
            field.choices = [];
        }
    }

    return field;
}
async function update(id, fieldData) {
    const {
        custom_field = false,
        field_label,
        field_name,
        field_type,
        required = false,
        quick_add = false,
        tool_tip = null,
        placeholder = null,
        choices = [],
        read_only = false,
        has_dependent = false,
        contactgroups_id = null,
        active = true,
        order = 0,
        lookup_type = null,
        lookup_column = null
    } = fieldData;

    // Validate required fields
    // if (!field_label || !field_name || !field_type) {
    //     throw new Error('field_label, field_name, and field_type are required');
    // }

    // Convert choices array to JSON string if it exists
    const choicesJson = Array.isArray(choices) && choices.length > 0 ? JSON.stringify(choices) : null;

    const [result] = await db.execute(
        `UPDATE contactfields SET 
            field_label = ?,
            field_name = ?,
            field_type = ?,
            required = ?,
            quick_add = ?,
            tool_tip = ?,
            placeholder = ?,
            choices = ?,
            read_only = ?,
            has_dependent = ?,
            contactgroups_id = ?,
            active = ?,
            \`order\` = ?,
            lookup_type = ?,
            lookup_column = ?,
            custom_field = ?,
            updated_at = NOW()
        WHERE id = ?`,
        [
            field_label,
            field_name,
            field_type,
            required ? 1 : 0,
            quick_add ? 1 : 0,
            tool_tip,
            placeholder,
            choicesJson,
            read_only ? 1 : 0,
            has_dependent ? 1 : 0,
            contactgroups_id,
            active ? 1 : 0,
            order,
            lookup_type,
            lookup_column,
            custom_field ? 1 : 0,
            id
        ]
    );

    return result.affectedRows > 0;
}

async function deleteRecord(id) {
    const [result] = await db.execute('UPDATE contactfields SET active = 0 WHERE id = ?', [id]);
    return result.affectedRows > 0;
}

async function order(fields) {

    try {
        for (const field of fields) {
            await db.execute(
                `UPDATE contactfields 
                 SET \`order\` = ?, 
                     updated_at = NOW() 
                 WHERE id = ?`,
                [field.order, field.id]
            );
        }
        
        return {
            status: 200,
            success: true,
            message: 'Field order updated successfully'
        };
} catch (error) {
    console.error('Error updating field order:', error);
    throw new Error(`Failed to update field order: ${error.message}`);
}

}


module.exports = {
    create,
    findAll,
    findById,
    update,
    deleteRecord,
    viewAllFilters,
    order
};