const db = require('../config/db');

/**
 * Get conveyance data based on filter
 * @param {Object} data - Filter data (filter, month, year)
 * @param {Number} userId - Current user ID
 */
async function getConveyance(data, userId) {
    try {
        const { filter, month, year } = data;
        let activities = [];
        let claimApply = false;
        
        // Get connection for potentially complex queries
        // const connection = await db.getConnection();
        
        try {
            // Different date filtering based on the filter parameter
            if (filter === "Today") {
                const date = new Date().toISOString().split('T')[0]; // Today's date YYYY-MM-DD
                
                // Get activities for today with salesactivities_id 3 or 10
                const [rows] = await db.execute(`
                    SELECT 
                        sad.start_date, 
                        SUM(sad.distance) AS distance, 
                        SUM(sad.claim_amount) AS claim_amount, 
                        SUM(sad.timespent) AS timespent
                    FROM 
                        salesactivitydatas sad
                    WHERE 
                        sad.start_date = ?
                        AND sad.owner_id = ?
                        AND sad.status = 'Closed'
                        AND sad.salesactivities_id IN (3, 10)
                    GROUP BY 
                        sad.start_date
                `, [date, userId]);
                
                activities = rows;
            } 
            else if (filter === "Weekly") {
                const today = new Date();
                const start = new Date(today);
                start.setDate(today.getDate() - 7);
                
                const startDate = start.toISOString().split('T')[0];
                const endDate = today.toISOString().split('T')[0];
                
                // Get activities for the past week
                const [rows] = await db.execute(`
                    SELECT 
                        sad.start_date, 
                        SUM(sad.distance) AS distance, 
                        SUM(sad.claim_amount) AS claim_amount, 
                        SUM(sad.timespent) AS timespent
                    FROM 
                        salesactivitydatas sad
                    WHERE 
                        sad.start_date BETWEEN ? AND ?
                        AND sad.owner_id = ?
                        AND sad.status = 'Closed'
                        AND sad.salesactivities_id IN (3, 10)
                    GROUP BY 
                        sad.start_date
                    ORDER BY 
                        sad.start_date
                `, [startDate, endDate, userId]);
                
                activities = rows;
                
                // Check if claim exists
                const [claimCount] = await db.execute(`
                    SELECT COUNT(*) as count
                    FROM salesactivitydatas
                    WHERE start_date BETWEEN ? AND ?
                    AND owner_id = ?
                    AND status = 'Closed'
                    AND salesactivities_id IN (3, 10)
                    AND claim_status != ''
                `, [startDate, endDate, userId]);
                
                if (claimCount.length > 0 && claimCount[0].count > 0) {
                    claimApply = true;
                }
            }
            else if (filter === "Monthly" && month && year) {
                try {
                    // Ensure month and year are integers
                    const numMonth = parseInt(month, 10);
                    const numYear = parseInt(year, 10);
                    
                    // Calculate first and last day of the month
                    const firstDay = new Date(numYear, numMonth - 1, 1);
                    const lastDay = new Date(numYear, numMonth, 0);
                    
                    const startDate = firstDay.toISOString().split('T')[0]; // First day of month
                    const endDate = lastDay.toISOString().split('T')[0]; // Last day of month
                    
                    console.log(`Fetching monthly data for ${startDate} to ${endDate} for user ${userId}`);
                    
                    // Format month name for display (YYYY-MM)
                    const monthDisplay = `${numYear}-${numMonth.toString().padStart(2, '0')}`;
                    
                    // Get activities for specific month and year with day-by-day breakdown and total
                    const [rows] = await db.execute(`
                        SELECT 
                            CASE 
                                WHEN sad.start_date IS NULL THEN 'Total'
                                ELSE DATE_FORMAT(sad.start_date, '%Y-%m-%d')
                            END AS conveyance_date,
                            SUM(IFNULL(sad.distance, 0)) AS distance, 
                            SUM(IFNULL(sad.claim_amount, 0)) AS claim_amount, 
                            SUM(IFNULL(sad.timespent, 0)) AS timespent,
                            IF(sad.start_date IS NULL, 1, 0) AS is_total
                        FROM 
                            salesactivitydatas sad
                        WHERE 
                            sad.start_date BETWEEN ? AND ?
                            AND sad.owner_id = ?
                            AND sad.status = 'Closed'
                            AND sad.salesactivities_id IN (3, 10)
                        GROUP BY 
                            sad.start_date WITH ROLLUP
                    `, [startDate, endDate, userId]);
                    
                    console.log(`Found ${rows.length} rows of data`);
                    
                    // If no actual data found for the month, create a placeholder total record
                    if (rows.length === 0 || (rows.length === 1 && rows[0].is_total === 1 && parseFloat(rows[0].distance) === 0)) {
                        console.log(`No data found for month ${monthDisplay}, creating placeholder`);
                        activities = [{
                            conveyance_date: monthDisplay,
                            distance: 0,
                            claim_amount: 0,
                            timespent: 0,
                            is_total: 1
                        }];
                    } else {
                        activities = rows;
                    }
                    
                    // Check if claim exists for monthly filter
                    const [claimCount] = await db.execute(`
                        SELECT COUNT(*) as count
                        FROM 
                            salesactivitydatas
                        WHERE 
                            start_date BETWEEN ? AND ?
                            AND owner_id = ?
                            AND status = 'Closed'
                            AND salesactivities_id IN (3, 10)
                            AND claim_status = 'Claimed'
                    `, [startDate, endDate, userId]);
                    
                    if (claimCount.length > 0 && claimCount[0].count > 0) {
                        claimApply = true;
                    }
                } catch (error) {
                    console.error('Error in monthly filter query:', error);
                    // Create a placeholder total record when there's an error
                    const numMonth = parseInt(month, 10);
                    const numYear = parseInt(year, 10);
                    const monthDisplay = `${numYear}-${numMonth.toString().padStart(2, '0')}`;
                    
                    console.log(`Creating placeholder record for ${monthDisplay} due to error`);
                    
                    activities = [{
                        conveyance_date: monthDisplay,
                        distance: 0,
                        claim_amount: 0,
                        timespent: 0,
                        is_total: 1
                    }];
                }
            }
            
            // Get conveyance charge config
            const [conveyanceConfig] = await db.execute(`
                SELECT * FROM conveyanceconfigs LIMIT 1
            `);
            
            // Return empty if no activities found
            if (activities.length === 0) {
                return {
                    status: false,
                    message: 'Conveyance info not exists',
                    data: []
                };
            }
            
            // Format the activities data for response
            const conveyances = activities.map(activity => {
                // Handle monthly data with totals
                if (filter === "Monthly") {
                    return {
                        conveyance_date: activity.conveyance_date,
                        distance: parseFloat(activity.distance) || 0,
                        amount: parseFloat(activity.claim_amount) || 0,
                        timespent: activity.timespent ? parseInt(activity.timespent) : 0,
                        is_total: activity.is_total === 1
                    };
                }
                
                // Standard mapping for daily data
                return {
                    conveyance_date: new Date(activity.start_date).toISOString().split('T')[0],
                    distance: parseFloat(activity.distance) || 0,
                    amount: parseFloat(activity.claim_amount) || 0,
                    timespent: activity.timespent ? parseInt(activity.timespent) : 0
                };
            });
            
            return {
                status: true,
                message: 'Conveyance info',
                data: {
                    conveyance: conveyances,
                    claimapply: claimApply
                }
            };
            
        } finally {
            connection.release();
        }
    } catch (error) {
        console.error('Error in getConveyance:', error);
        return {
            status: false,
            message: `Error getting conveyance data: ${error.message}`,
            data: []
        };
    }
}

function formatTime(time) {
    if (!time) return null;
    const date = new Date(time);

    // IST offset in minutes
    const IST_OFFSET = 5.5 * 60;
    // Convert to IST
    const utc = date.getTime() + (date.getTimezoneOffset() * 60000);
    const istDate = new Date(utc + (IST_OFFSET * 60000));

    // Format in 12-hour time and force AM/PM uppercase
    let formatted = istDate.toLocaleTimeString('en-US', {
        hour: '2-digit',
        minute: '2-digit',
        hour12: true
    });

    // Ensure AM/PM is always uppercase and with a space
    formatted = formatted.replace(/([AP]M)$/, ' $1').toUpperCase();

    return formatted;
}

// Example usage:
console.log(formatTime('2024-07-03T05:15:00Z')); // "10:45 AM"


/**
 * Get detailed conveyance data for a specific date
 * @param {String} date - Date in YYYY-MM-DD format
 * @param {Number} userId - Current user ID
 * @returns {Object} Conveyance details for the date
 */
async function getConveyanceByDate(date, userId) {
    try {
        // Get transport modes from conveyance configs
        const [transportModes] = await db.execute(
            `SELECT transport_mode, conveyance_charge 
             FROM conveyanceconfigs 
             `
        );
        console.log(date, userId)
        // Get activities for the date
        const [rows] = await db.execute(`
            SELECT 
                sad.*,
                u.name as employee_name,
                t.activity_name,
                a.name as account_name,
                c.first_name as contact_name,
                cc.conveyance_charge as rate_per_km
            FROM salesactivitydatas sad
            LEFT JOIN users u ON sad.owner_id = u.id
            LEFT JOIN salesactivities t ON sad.salesactivities_id = t.id
            LEFT JOIN accounts a ON sad.targetable_id = a.id AND sad.targetable_type = 'accounts'
            LEFT JOIN contacts c ON sad.targetable_id = c.id AND sad.targetable_type = 'contacts'
            LEFT JOIN conveyanceconfigs cc ON sad.transport_mode = cc.transport_mode
            WHERE sad.start_date = ? 
            AND sad.owner_id = ? 
            AND sad.status = 'Closed'
	    AND sad.claim_status = 'Pending' 
            AND sad.salesactivities_id IN (3, 10)
        `, [date, userId]);

        // Format activities data
        const activities = rows.map(row => ({
            id: row.id,
            activity_name: row.activity_name,
            start_time: formatTime(row.checkin_time),
            end_time: formatTime(row.checkout_time),
            checkin_time: formatTime(row.checkin_time),
            checkout_time: formatTime(row.checkout_time),
            transport_mode: row.transport_mode,
            distance: parseFloat(row.distance || 0),
            rate_per_km: parseFloat(row.rate_per_km || 0),
            claim_amount: parseFloat(row.claim_amount || 0),
	    claim_status: row.claim_status,
            target_name: row.targetable_type === 'accounts' ? row.account_name : row.contact_name,
            target_type: row.targetable_type,
	    checkin_latitude: row.checkin_latitude,
	    checkin_longitude: row.checkin_longitude,
            checkout_latitude: row.checkout_latitude,
	    checkout_longitude: row.checkout_longitude,
            remarks: row.remarks
        }));

        // Calculate totals
        const totals = activities.reduce((acc, curr) => {
            acc.total_distance += curr.distance;
            acc.total_amount += curr.claim_amount;
            return acc;
        }, { total_distance: 0, total_amount: 0 });

        return {
            status: true,
            message: 'Conveyance data retrieved successfully',
            data: {
                date: date,
                activities: activities,
                transport_modes: transportModes.map(mode => ({
                    mode: mode.transport_mode,
                    rate: parseFloat(mode.conveyance_charge)
                })),
                summary: {
                    total_distance: totals.total_distance.toFixed(2),
                    total_amount: totals.total_amount.toFixed(2)
                }
            }
        };

    } catch (error) {
        console.error('Error in getConveyanceByDate:', error);
        return {
            status: false,
            message: `Error getting conveyance data: ${error.message}`,
            data: {
                date: date,
                activities: [],
                transport_modes: [],
                summary: {
                    total_distance: "0.00",
                    total_amount: "0.00"
                }
            }
        };
    }
}

// claim apply
/**
 * Apply for claim
 * @param {String} startDate - Start date in YYYY-MM-DD format
 * @param {String} endDate - End date in YYYY-MM-DD format
 * @param {Number} userId - Current user ID
 * @returns {Object} Result of the claim application
 */
async function claimapply(startDate, endDate, userId) {
    try {
        // const connection = await db.getConnection();
        const [rows] = await db.execute(`
            UPDATE salesactivitydatas
            SET claim_status = 'Applied'
            WHERE start_date BETWEEN ? AND ?
            AND owner_id = ?
            AND status = 'Closed'
            AND salesactivities_id IN (3, 10)
        `, [startDate, endDate, userId]);

        return rows;
    } catch (error) {
        console.error('Error in claimapply:', error);
    }
}

// claim history 
async function claimhistory(userId, startDate, endDate) {
    try {
        // Check if startDate and endDate are provided
        if (startDate && endDate) {
            // const connection = await db.getConnection();
            const [rows] = await db.execute(`
                SELECT * FROM salesactivitydatas
                    WHERE owner_id = ?
                    AND start_date BETWEEN ? AND ?
                    AND claim_status = 'Pending'
                    AND salesactivities_id IN (3, 10)
                `, [userId, startDate, endDate]);

            return rows;
        } else {
            // If startDate and endDate are not provided, fetch all records
            // const connection = await db.getConnection();
            const [rows] = await db.execute(`
                SELECT * FROM salesactivitydatas
                    WHERE owner_id = ?
                    AND claim_status = 'Pending'
                    AND salesactivities_id IN (3, 10)
                `, [userId]);

            return rows;
        }
    } catch (error) {
        console.error('Error in claimhistory:', error);
    }
}


/**
 * Edit claim by ID
 * @param {Number} id - Claim ID
 * @param {Object} data - Claim update data
 * @param {String} data.transport_mode - Mode of transport (bike/car/public)
 * @param {Number} data.distance - Distance traveled
 * @param {String} data.claim_image - Path to claim image
 * @param {Number} userId - Current user ID
 */
async function editClaimById(id, data, userId) {
    try {
        console.log('Editing claim:', { id, data, userId });
        // Get existing claim
        const [existingClaim] = await db.execute(
            `SELECT * FROM salesactivitydatas WHERE id = ? AND owner_id = ?`,
            [id, userId]
        );

        if (!existingClaim.length) {
            throw new Error('Claim not found or you do not have permission to edit this claim');
        }

        let claimAmount = 0;
        let updateFields = [];
        let updateValues = [];

        if (data.transport_mode === 'Public Transport') {
            // Validate required fields for public transport
            if (!data.amount || !data.claim_images || data.claim_images.length === 0) {
                throw new Error('Amount and bill attachment are required for Public Transport');
            }

            // Use manual amount for public transport
            claimAmount = parseFloat(data.amount);
            if (isNaN(claimAmount) || claimAmount <= 0) {
                throw new Error('Invalid amount for Public Transport');
            }

            // Ensure claim_images is an array and join multiple images with comma
            const claimImagesArray = Array.isArray(data.claim_images) ? data.claim_images : [data.claim_images].filter(Boolean);
            const claimImages = claimImagesArray.join(',');

            updateFields = [
                'transport_mode = ?',
                'claim_amount = ?',
                'claim_image = ?',
                'updated_by = ?',
                'updated_at = CURRENT_TIMESTAMP'
            ];
            updateValues = [
                data.transport_mode,
                claimAmount,
                claimImages,
                userId
            ];
        } else {
            // Get conveyance configuration for other transport modes
            const [config] = await db.execute(
                'SELECT * FROM conveyanceconfigs WHERE transport_mode = ?',
                [data.transport_mode]
            );

            if (!config.length) {
                throw new Error('Invalid transport mode or configuration not found');
            }

            // Calculate claim amount based on distance and conveyance charge
            claimAmount = parseFloat(existingClaim[0].distance) * parseFloat(config[0].conveyance_charge);

            // If there are new images, use them, otherwise keep existing ones
            let claimImages;
            if (data.claim_images && data.claim_images.length > 0) {
                // Ensure claim_images is an array and join
                const claimImagesArray = Array.isArray(data.claim_images) ? data.claim_images : [data.claim_images].filter(Boolean);
                claimImages = claimImagesArray.join(',');
            } else {
                claimImages = existingClaim[0].claim_image;
            }

            updateFields = [
                'transport_mode = ?',
                'claim_amount = ?',
                'claim_image = ?',
                'updated_by = ?',
                'updated_at = CURRENT_TIMESTAMP'
            ];
            updateValues = [
                data.transport_mode,
                claimAmount,
                claimImages,
                userId
            ];
        }

        // Update claim
        const [result] = await db.execute(
            `UPDATE salesactivitydatas 
             SET ${updateFields.join(', ')}
             WHERE id = ? AND owner_id = ?`,
            [...updateValues, id, userId]
        );

        if (result.affectedRows === 0) {
            throw new Error('Failed to update claim');
        }

        return {
            status: true,
            message: 'Claim updated successfully',
            data: {
                id,
                transport_mode: data.transport_mode,
                distance: existingClaim[0].distance || 0,
                claim_amount: claimAmount,
                claim_images: data.claim_images || []
            }
        };

    } catch (error) {
        console.error('Error in editClaimById:', error);
        throw new Error(`Failed to update claim: ${error.message}`);
    }
}

// Reapply claim
async function reapplyClaim(id, data, userId) {
    try {
        console.log(id, data, userId);
        // Get existing claim
        const [existingClaim] = await db.execute(
            `SELECT * FROM salesactivitydatas WHERE id = ?`,
            [id]
        );

        if (!existingClaim.length) {
            throw new Error('Claim not found');
        }

        let claimAmount = 0;
        let updateFields = [];
        let updateValues = [];

        if (data.transport_mode === 'Public Transport') {
            // Validate required fields for public transport
            if (!data.amount || !data.claim_images || data.claim_images.length === 0) {
                throw new Error('Amount and bill attachment are required for Public Transport');
            }

            // Use manual amount for public transport
            claimAmount = parseFloat(data.amount);
            if (isNaN(claimAmount) || claimAmount <= 0) {
                throw new Error('Invalid amount for Public Transport');
            }

            // Ensure claim_images is an array and join multiple images with comma
            const claimImagesArray = Array.isArray(data.claim_images) ? data.claim_images : [data.claim_images].filter(Boolean);
            const claimImages = claimImagesArray.join(',');

            updateFields = [
                'transport_mode = ?',
                'claim_amount = ?',
                'claim_image = ?',
                'claim_status = ?',
                'remarks = ?',
                'updated_by = ?',
                'updated_at = CURRENT_TIMESTAMP'
            ];
            updateValues = [
                data.transport_mode,
                claimAmount,
                claimImages,
                'Reapplied',
                data.remarks,
                userId
            ];
        } else {
            // Get conveyance configuration for other transport modes
            const [config] = await db.execute(
                'SELECT * FROM conveyanceconfigs WHERE transport_mode = ?',
                [data.transport_mode]
            );

            if (!config.length) {
                throw new Error('Invalid transport mode or configuration not found');
            }

            // Calculate claim amount based on distance and conveyance charge
            claimAmount = parseFloat(existingClaim[0].distance) * parseFloat(config[0].conveyance_charge);

            // If there are new images, use them, otherwise keep existing ones
            let claimImages;
            if (data.claim_images && data.claim_images.length > 0) {
                // Ensure claim_images is an array and join
                const claimImagesArray = Array.isArray(data.claim_images) ? data.claim_images : [data.claim_images].filter(Boolean);
                claimImages = claimImagesArray.join(',');
            } else {
                claimImages = existingClaim[0].claim_image;
            }

            updateFields = [
                'transport_mode = ?',
                'claim_amount = ?',
                'claim_image = ?',
                'status = ?',
                'claim_status = ?',
                'remarks = ?',
                'updated_by = ?',
                'updated_at = CURRENT_TIMESTAMP'
            ];
            updateValues = [
                data.transport_mode,
                claimAmount,
                claimImages,
                'Reapplied',
                'Reapplied',
                data.remarks,
                userId
            ];
        }

        // Update claim
        const [result] = await db.execute(
            `UPDATE salesactivitydatas 
             SET ${updateFields.join(', ')}
             WHERE id = ?`,
            [...updateValues, id]
        );

        if (result.affectedRows === 0) {
            throw new Error('Failed to update claim');
        }

        return {
            data: {
                id,
                transport_mode: data.transport_mode,
                distance: existingClaim[0].distance || 0,
                claim_amount: claimAmount,
                claim_images: data.claim_images || []
            }
        };

    } catch (error) {
        console.error('Error in reapplyClaim:', error);
        throw new Error(`Failed to reapply claim: ${error.message}`);
    }
}

/**
 * Get claim history by status
 * @param {Number} userId - Current user ID
 * @param {String} startDate - Start date in YYYY-MM-DD format (optional)
 * @param {String} endDate - End date in YYYY-MM-DD format (optional)
 * @returns {Object} Claim history data
 */
async function claimhistoryByStatus(userId, startDate = null, endDate = null) {
    try {
        let query = `
            SELECT 
                sad.*,
                sa.activity_name,
                a.name as account_name,
                c.first_name as contact_name,
                u.name as employee_name
            FROM salesactivitydatas sad
            LEFT JOIN salesactivities sa ON sad.salesactivities_id = sa.id
            LEFT JOIN accounts a ON sad.targetable_id = a.id AND sad.targetable_type = 'accounts'
            LEFT JOIN contacts c ON sad.targetable_id = c.id AND sad.targetable_type = 'contacts'
            LEFT JOIN users u ON sad.owner_id = u.id
            WHERE sad.owner_id = ?
            AND sad.claim_status IN ('Applied', 'Rejected', 'Reapplied', 'Approved', 'Manager Approved', 'Completed', 'Finance Rejected', 'Manager Rejected')
            AND sad.salesactivities_id IN (3, 10)`;

        const params = [userId];

        if (startDate && endDate) {
            query += ` AND sad.start_date BETWEEN ? AND ?`;
            params.push(startDate, endDate);
        }

        query += ` ORDER BY sad.start_date DESC, sad.start_time DESC`;

        const [rows] = await db.execute(query, params);

        return rows.map(row => ({
            id: row.id,
            activity_name: row.activity_name,
            start_date: row.start_date,
            start_time: row.start_time,
            end_date: row.end_date,
            end_time: row.end_time,
            transport_mode: row.transport_mode,
            distance: parseFloat(row.distance || 0),
            claim_amount: parseFloat(row.claim_amount || 0),
            status: row.claim_status,
            target_name: row.targetable_type === 'accounts' ? row.account_name : row.contact_name,
            target_type: row.targetable_type,
            employee_name: row.employee_name,
            remarks: row.remarks,
            claim_image: row.claim_image
        }));

    } catch (error) {
        console.error('Error in claimhistoryByStatus:', error);
        throw new Error(`Failed to get claim history: ${error.message}`);
    }
}

module.exports = {
    getConveyance,
    getConveyanceByDate,
    claimapply,
    claimhistory,
    editClaimById,
    claimhistoryByStatus,
    reapplyClaim
};