const db = require('../config/db');

/**
 * Get conveyance data based on filter
 * @param {Object} data - Filter data (filter, month, year)
 * @param {Number} userId - Current user ID
 */
async function getConveyance(data, userId) {
    try {
        const { filter, month, year } = data;
        let activities = [];
        let claimApply = false;
        
        // Get connection for potentially complex queries
        // const connection = await db.getConnection();
        
        try {
            // Different date filtering based on the filter parameter
            if (filter === "Today") {
                const date = new Date().toISOString().split('T')[0]; // Today's date YYYY-MM-DD
                
                // Get activities for today with salesactivities_id 3 or 10
                const [rows] = await db.execute(`
                    SELECT 
                        sad.start_date, 
                        SUM(sad.distance) AS distance, 
                        SUM(sad.claim_amount) AS claim_amount, 
                        SUM(sad.timespent) AS timespent
                    FROM 
                        salesactivitydatas sad
                    WHERE 
                        sad.start_date = ?
                        AND sad.owner_id = ?
                        AND sad.status = 'Closed'
                        AND sad.salesactivities_id IN (3, 10)
                    GROUP BY 
                        sad.start_date
                `, [date, userId]);
                
                activities = rows;
            } 
            else if (filter === "Weekly") {
                const today = new Date();
                const start = new Date(today);
                start.setDate(today.getDate() - 7);
                
                const startDate = start.toISOString().split('T')[0];
                const endDate = today.toISOString().split('T')[0];
                
                // Get activities for the past week
                const [rows] = await db.execute(`
                    SELECT 
                        sad.start_date, 
                        SUM(sad.distance) AS distance, 
                        SUM(sad.claim_amount) AS claim_amount, 
                        SUM(sad.timespent) AS timespent
                    FROM 
                        salesactivitydatas sad
                    WHERE 
                        sad.start_date BETWEEN ? AND ?
                        AND sad.owner_id = ?
                        AND sad.status = 'Closed'
                        AND sad.salesactivities_id IN (3, 10)
                    GROUP BY 
                        sad.start_date
                    ORDER BY 
                        sad.start_date
                `, [startDate, endDate, userId]);
                
                activities = rows;
                
                // Check if claim exists
                const [claimCount] = await db.execute(`
                    SELECT COUNT(*) as count
                    FROM salesactivitydatas
                    WHERE start_date BETWEEN ? AND ?
                    AND owner_id = ?
                    AND status = 'Closed'
                    AND salesactivities_id IN (3, 10)
                    AND claim_status != ''
                `, [startDate, endDate, userId]);
                
                if (claimCount.length > 0 && claimCount[0].count > 0) {
                    claimApply = true;
                }
            }
            else if (filter === "Monthly" && month && year) {
                try {
                    // Ensure month and year are integers
                    const numMonth = parseInt(month, 10);
                    const numYear = parseInt(year, 10);
                    
                    // Calculate first and last day of the month
                    const firstDay = new Date(numYear, numMonth - 1, 1);
                    const lastDay = new Date(numYear, numMonth, 0);
                    
                    const startDate = firstDay.toISOString().split('T')[0]; // First day of month
                    const endDate = lastDay.toISOString().split('T')[0]; // Last day of month
                    
                    console.log(`Fetching monthly data for ${startDate} to ${endDate} for user ${userId}`);
                    
                    // Format month name for display (YYYY-MM)
                    const monthDisplay = `${numYear}-${numMonth.toString().padStart(2, '0')}`;
                    
                    // Get activities for specific month and year with day-by-day breakdown and total
                    const [rows] = await db.execute(`
                        SELECT 
                            CASE 
                                WHEN sad.start_date IS NULL THEN 'Total'
                                ELSE DATE_FORMAT(sad.start_date, '%Y-%m-%d')
                            END AS conveyance_date,
                            SUM(IFNULL(sad.distance, 0)) AS distance, 
                            SUM(IFNULL(sad.claim_amount, 0)) AS claim_amount, 
                            SUM(IFNULL(sad.timespent, 0)) AS timespent,
                            IF(sad.start_date IS NULL, 1, 0) AS is_total
                        FROM 
                            salesactivitydatas sad
                        WHERE 
                            sad.start_date BETWEEN ? AND ?
                            AND sad.owner_id = ?
                            AND sad.status = 'Closed'
                            AND sad.salesactivities_id IN (3, 10)
                        GROUP BY 
                            sad.start_date WITH ROLLUP
                    `, [startDate, endDate, userId]);
                    
                    console.log(`Found ${rows.length} rows of data`);
                    
                    // If no actual data found for the month, create a placeholder total record
                    if (rows.length === 0 || (rows.length === 1 && rows[0].is_total === 1 && parseFloat(rows[0].distance) === 0)) {
                        console.log(`No data found for month ${monthDisplay}, creating placeholder`);
                        activities = [{
                            conveyance_date: monthDisplay,
                            distance: 0,
                            claim_amount: 0,
                            timespent: 0,
                            is_total: 1
                        }];
                    } else {
                        activities = rows;
                    }
                    
                    // Check if claim exists for monthly filter
                    const [claimCount] = await db.execute(`
                        SELECT COUNT(*) as count
                        FROM 
                            salesactivitydatas
                        WHERE 
                            start_date BETWEEN ? AND ?
                            AND owner_id = ?
                            AND status = 'Closed'
                            AND salesactivities_id IN (3, 10)
                            AND claim_status = 'Claimed'
                    `, [startDate, endDate, userId]);
                    
                    if (claimCount.length > 0 && claimCount[0].count > 0) {
                        claimApply = true;
                    }
                } catch (error) {
                    console.error('Error in monthly filter query:', error);
                    // Create a placeholder total record when there's an error
                    const numMonth = parseInt(month, 10);
                    const numYear = parseInt(year, 10);
                    const monthDisplay = `${numYear}-${numMonth.toString().padStart(2, '0')}`;
                    
                    console.log(`Creating placeholder record for ${monthDisplay} due to error`);
                    
                    activities = [{
                        conveyance_date: monthDisplay,
                        distance: 0,
                        claim_amount: 0,
                        timespent: 0,
                        is_total: 1
                    }];
                }
            }
            
            // Get conveyance charge config
            const [conveyanceConfig] = await db.execute(`
                SELECT * FROM conveyanceconfigs LIMIT 1
            `);
            
            // Return empty if no activities found
            if (activities.length === 0) {
                return {
                    status: false,
                    message: 'Conveyance info not exists',
                    data: []
                };
            }
            
            // Format the activities data for response
            const conveyances = activities.map(activity => {
                // Handle monthly data with totals
                if (filter === "Monthly") {
                    return {
                        conveyance_date: activity.conveyance_date,
                        distance: parseFloat(activity.distance) || 0,
                        amount: parseFloat(activity.claim_amount) || 0,
                        timespent: activity.timespent ? parseInt(activity.timespent) : 0,
                        is_total: activity.is_total === 1
                    };
                }
                
                // Standard mapping for daily data
                return {
                    conveyance_date: new Date(activity.start_date).toISOString().split('T')[0],
                    distance: parseFloat(activity.distance) || 0,
                    amount: parseFloat(activity.claim_amount) || 0,
                    timespent: activity.timespent ? parseInt(activity.timespent) : 0
                };
            });
            
            return {
                status: true,
                message: 'Conveyance info',
                data: {
                    conveyance: conveyances,
                    claimapply: claimApply
                }
            };
            
        } finally {
            connection.release();
        }
    } catch (error) {
        console.error('Error in getConveyance:', error);
        return {
            status: false,
            message: `Error getting conveyance data: ${error.message}`,
            data: []
        };
    }
}

/**
 * Get detailed conveyance data for a specific date
 * @param {String} date - Date in YYYY-MM-DD format
 * @param {Number} userId - Current user ID
 * @returns {Object} Conveyance details for the date
 */
async function getConveyanceByDate(date, userId) {
    try {
        // Get connection for potentially complex queries
        // const connection = await db.getConnection();
    
        const [rows] = await db.execute(`
            SELECT * FROM salesactivitydatas 
                WHERE start_date = ? 
                AND owner_id = ? 
                AND status = 'Closed' 
                AND salesactivities_id IN (3, 10)
        `, [date, userId]);

        const result = {
            date: date,
            distance: rows[0].distance,
            amount: rows[0].claim_amount, 
            timespent: rows[0].timespent,
            date: rows 
        }

        return result;
    } catch (error) {
        console.error('Error in getConveyanceByDate:', error);
        return {
            status: false,
            message: `Error getting conveyance data: ${error.message}`,
            data: []
        };
    }
}

// claim apply
/**
 * Apply for claim
 * @param {String} startDate - Start date in YYYY-MM-DD format
 * @param {String} endDate - End date in YYYY-MM-DD format
 * @param {Number} userId - Current user ID
 * @returns {Object} Result of the claim application
 */
async function claimapply(startDate, endDate, userId) {
    try {
        // const connection = await db.getConnection();
        const [rows] = await db.execute(`
            UPDATE salesactivitydatas
            SET status = 'Applied'
            WHERE start_date BETWEEN ? AND ?
            AND owner_id = ?
            AND status = 'Closed'
            AND salesactivities_id IN (3, 10)
        `, [startDate, endDate, userId]);

        return rows;
    } catch (error) {
        console.error('Error in claimapply:', error);
    }
}

// claim history 
async function claimhistory(userId, startDate, endDate) {
    try {
        // Check if startDate and endDate are provided
        if (startDate && endDate) {
            // const connection = await db.getConnection();
            const [rows] = await db.execute(`
                SELECT * FROM salesactivitydatas
                    WHERE owner_id = ?
                    AND start_date BETWEEN ? AND ?
                    AND status = 'Applied'
                    AND salesactivities_id IN (3, 10)
                `, [userId, startDate, endDate]);

            return rows;
        } else {
            // If startDate and endDate are not provided, fetch all records
            // const connection = await db.getConnection();
            const [rows] = await db.execute(`
                SELECT * FROM salesactivitydatas
                    WHERE owner_id = ?
                    AND status = 'Applied'
                    AND salesactivities_id IN (3, 10)
                `, [userId]);

            return rows;
        }
    } catch (error) {
        console.error('Error in claimhistory:', error);
    }
}

module.exports = {
    getConveyance,
    getConveyanceByDate,
    claimapply,
    claimhistory
};  