const db = require('../config/db');

async function totalcounts(req) {
    const { start_date, end_date } = req.body;

    // Validate and format dates
    if (!start_date || !end_date) {
        throw new Error('Both start_date and end_date are required');
    }

    const [result] = await db.execute(
        `SELECT 
            (SELECT COUNT(*) FROM accounts 
             WHERE 
             DATE(created_at) BETWEEN DATE(?) AND DATE(?)) as total_accounts,
            
            (SELECT COUNT(*) FROM contacts 
             WHERE
             DATE(created_at) BETWEEN DATE(?) AND DATE(?)) as total_contacts,
            
            (SELECT COUNT(*) FROM deals 
             WHERE 
             DATE(created_at) BETWEEN DATE(?) AND DATE(?)) as total_deals,
            
            (SELECT COUNT(*) FROM salesactivitydatas sad 
             INNER JOIN salesactivities sa ON sad.salesactivities_id = sa.id 
             WHERE 
             sa.activity_name = 'meeting'
             AND DATE(sad.created_at) BETWEEN DATE(?) AND DATE(?)) as total_meetings,
            
            (SELECT COUNT(*) FROM salesactivitydatas sad 
             INNER JOIN salesactivities sa ON sad.salesactivities_id = sa.id 
             WHERE 
             sa.activity_name = 'task'
             AND DATE(sad.created_at) BETWEEN DATE(?) AND DATE(?)) as total_tasks`,
        [
            start_date, end_date,  // for accounts
            start_date, end_date,  // for contacts
            start_date, end_date,  // for deals
            start_date, end_date,  // for meetings
            start_date, end_date   // for tasks
        ]
    );

    const [task] = await db.execute(
        `SELECT COUNT(*) as total_tasks FROM tasks t 
        WHERE DATE(t.created_at) BETWEEN DATE(?) AND DATE(?)`,
        [start_date, end_date]
    );


    return {
        total_accounts: parseInt(result[0].total_accounts),
        total_contacts: parseInt(result[0].total_contacts),
        total_deals: parseInt(result[0].total_deals),
        total_meetings: parseInt(result[0].total_meetings),
        total_tasks: parseInt(task[0].total_tasks)
    };
}

async function getDailyAttendance(req) {
    const { start_date, end_date } = req.body;
    
    if (!start_date || !end_date) {
        throw new Error('Both start_date and end_date are required');
    }

    // First get total number of non-admin users
    const [userCount] = await db.execute(
        `SELECT COUNT(*) as total_users 
         FROM users 
         WHERE active = 1 AND roles_id != 1`
    );
    const totalUsers = parseInt(userCount[0].total_users);

    // Set timezone to UTC explicitly
    await db.execute('SET time_zone = "+00:00"');

    const [result] = await db.execute(
        `SELECT 
            DATE(sad.start_date) as date,
            COUNT(DISTINCT u.id) as present_count
        FROM salesactivitydatas sad
        INNER JOIN salesactivities sa ON sad.salesactivities_id = sa.id
        INNER JOIN users u ON sad.owner_id = u.id
        WHERE sa.activity_name = 'attendance'
        AND sad.checkin_time IS NOT NULL
        AND u.roles_id != 1
        AND u.active = 1
        AND DATE(sad.start_date) = DATE(?)
        GROUP BY DATE(sad.start_date)`,
        [start_date]
    );

    const presentCount = result.length > 0 ? parseInt(result[0].present_count) : 0;
    const absentCount = totalUsers - presentCount;

    return {
        date: start_date,
        total_users: totalUsers,
        present: presentCount,
        absent: absentCount,
        attendance_percentage: totalUsers > 0 ? Math.round((presentCount / totalUsers) * 100) : 0
    };
}

async function getWeeklyAttendance(req) {
    const { start_date, end_date } = req.body;
    
    if (!start_date || !end_date) {
        throw new Error('Both start_date and end_date are required');
    }

    // First get total number of non-admin users
    const [userCount] = await db.execute(
        `SELECT COUNT(*) as total_users 
         FROM users 
         WHERE active = 1 AND roles_id != 1`
    );
    const totalUsers = parseInt(userCount[0].total_users);

    // Set timezone to UTC explicitly
    await db.execute('SET time_zone = "+00:00"');

    // Generate all dates in the range
    const [dates] = await db.execute(
        `WITH RECURSIVE date_range AS (
            SELECT DATE(?) as date
            UNION ALL
            SELECT DATE_ADD(date, INTERVAL 1 DAY)
            FROM date_range
            WHERE date < DATE(?)
        )
        SELECT date FROM date_range`,
        [start_date, end_date]
    );

    // Get attendance data
    const [attendance] = await db.execute(
        `SELECT 
            DATE(sad.start_date) as date,
            COUNT(DISTINCT u.id) as present_count,
            GROUP_CONCAT(DISTINCT CONCAT(u.name, ' (', 
                TIME_FORMAT(sad.start_date, '%H:%i'), ' - ', 
                CASE 
                    WHEN sad.end_date IS NOT NULL THEN TIME_FORMAT(sad.end_date, '%H:%i')
                    ELSE 'Not checked out'
                END, 
            ')') ORDER BY sad.start_date) as present_users
        FROM salesactivitydatas sad
        INNER JOIN salesactivities sa ON sad.salesactivities_id = sa.id
        INNER JOIN users u ON sad.owner_id = u.id
        WHERE sa.activity_name = 'attendance'
        AND u.roles_id != 1
        AND u.active = 1
        AND DATE(sad.start_date) BETWEEN DATE(?) AND DATE(?)
        GROUP BY DATE(sad.start_date)`,
        [start_date, end_date]
    );

    // Create a map of attendance data with explicit date handling
    const attendanceMap = new Map(
        attendance.map(row => {
            const date = new Date(row.date);
            date.setUTCHours(0, 0, 0, 0);
            return [
                date.toISOString().split('T')[0],
                {
                    present: parseInt(row.present_count),
                    present_users: row.present_users ? row.present_users.split(',') : []
                }
            ];
        })
    );

    // Generate report for each day with explicit date handling
    return dates.map(row => {
        const date = new Date(row.date);
        date.setUTCHours(0, 0, 0, 0);
        const dateStr = date.toISOString().split('T')[0];
        const dayData = attendanceMap.get(dateStr) || { present: 0, present_users: [] };
        const presentCount = dayData.present;
        const absentCount = totalUsers - presentCount;

        return {
            date: dateStr,
            total_users: totalUsers,
            present: presentCount,
            absent: absentCount,
            attendance_percentage: totalUsers > 0 ? Math.round((presentCount / totalUsers) * 100) : 0,
            present_users: dayData.present_users,
            day_of_week: date.toLocaleDateString('en-US', { weekday: 'long' })
        };
    });
}

// Helper functions for date formatting
function formatDate(date) {
    if (!date) return null;
    return new Date(date).toLocaleDateString('en-GB', {
        day: '2-digit',
        month: '2-digit',
        year: 'numeric'
    });
}

function formatTime(time) {
    if (!time) return null;
    return new Date(time).toLocaleTimeString('en-US', {
        hour: '2-digit',
        minute: '2-digit',
        hour12: true
    });
}


function formatDateTime(datetime) {
    if (!datetime) return null;
    const date = new Date(datetime);
    return date.toLocaleString('en-GB', {
        day: '2-digit',
        month: '2-digit',
        year: 'numeric',
        hour: '2-digit',
        minute: '2-digit',
        second: '2-digit',
        hour12: false
    });
}

// last 5 meetings and tasks
async function activitylist() {
    const [meetings] = await db.execute(`
        SELECT 
            sad.*,
            owner.name AS owner_name,
            creator.name AS creator_name,
            updater.name AS updater_name,
            CASE 
                WHEN sad.targetable_type = 'contacts' THEN CONCAT(c.first_name, ' ', c.last_name)
                WHEN sad.targetable_type = 'accounts' THEN ac.name
                WHEN sad.targetable_type = 'deals' THEN d.name
                ELSE NULL
            END as target_name,
            CASE 
                WHEN sad.targetable_type = 'contacts' THEN c.emails
                WHEN sad.targetable_type = 'accounts' THEN ac.website
                WHEN sad.targetable_type = 'deals' THEN d.amount
                ELSE NULL
            END as target_details
        FROM salesactivitydatas sad
        LEFT JOIN users AS owner ON sad.owner_id = owner.id
        LEFT JOIN users AS creator ON sad.creater_id = creator.id
        LEFT JOIN users AS updater ON sad.updater_id = updater.id
        LEFT JOIN contacts c ON sad.targetable_type = 'contacts' AND sad.targetable_id = c.id
        LEFT JOIN accounts ac ON sad.targetable_type = 'accounts' AND sad.targetable_id = ac.id
        LEFT JOIN deals d ON sad.targetable_type = 'deals' AND sad.targetable_id = d.id
        WHERE sad.salesactivities_id = 3
        ORDER BY sad.created_at DESC
        lIMIT 5`
    );

    // Process each meeting to get attendee details
    for (let meeting of meetings) {
        // Get attendee details
        if (meeting.attendees) {
            const attendeeIds = meeting.attendees.split(',').map(id => id.trim());
            if (attendeeIds.length > 0) {
                const placeholders = attendeeIds.map(() => '?').join(',');
                const [attendeeRows] = await db.execute(`
                    SELECT id, name, email, mobile
                    FROM users
                    WHERE id IN (${placeholders})`,
                    attendeeIds
                );
                meeting.attendee_details = attendeeRows;
            } else {
                meeting.attendee_details = [];
            }
        } else {
            meeting.attendee_details = [];
        }

        // Format dates and times
        meeting.start_date = formatDate(meeting.start_date);
        meeting.end_date = formatDate(meeting.end_date);
        meeting.start_time = formatTime(meeting.start_time);
        meeting.end_time = formatTime(meeting.end_time);
        meeting.checkin_time = formatDateTime(meeting.checkin_time);
        meeting.checkout_time = formatDateTime(meeting.checkout_time);

        // Structure target information
        meeting.target = {
            type: meeting.targetable_type,
            id: meeting.targetable_id,
            name: meeting.target_name,
        };

        // Remove the raw fields
        delete meeting.targetable_type;
        delete meeting.targetable_id;
        delete meeting.target_name;
        delete meeting.target_details;
    }

    const [tasks] = await db.execute(
        ` SELECT t.*, u.name AS user_name, u2.name AS collaborator_name
        FROM tasks t
        LEFT JOIN users u ON t.user_id = u.id
        LEFT JOIN users u2 ON t.collaboratoes = u2.id
        ORDER BY t.id DESC LIMIT 5`
    );

    return {
        meetings,
        tasks
    };
}

module.exports = { 
    totalcounts, 
    getDailyAttendance,
    getWeeklyAttendance,
    activitylist
 };