const { json } = require('express');
const db = require('../config/db');
const { object } = require('@google/maps/lib/internal/validate');
const { CONTAINS_LIST } = require('../config/utils');
const dayjs = require('dayjs');

async function create(Data) {
    let connection = null;
    try {
        connection = await db.getConnection();
        await connection.beginTransaction();

        const { deal: allFields, user_id } = Data;
        if (!allFields || !Array.isArray(allFields)) {
            throw new Error('Invalid input data format');
        }

        // Get unique fields from dealfields table
        const [uniqueFields] = await connection.execute(
            'SELECT field_name FROM dealfields WHERE unique_field = "on"'
        );

        // Check for uniqueness constraints
        for (const field of allFields) {
            if (uniqueFields.some(uf => uf.field_name === field.field_name) && field.field_value) {
                // Check if value already exists
                const [existing] = await connection.execute(
                    `SELECT id FROM deals WHERE ${field.field_name} = ? AND active = 1`,
                    [field.field_value]
                );

                if (existing.length > 0) {
                    throw new Error(`Deal with ${field.field_name} "${field.field_value}" already exists`);
                }
            }
        }

        // Check pipeline_id and stage_id
        const pipelineField = allFields.find(field => field.field_name === 'deal_pipeline_id');
        const stageField = allFields.find(field => field.field_name === 'deal_stage_id');

        if (!pipelineField?.field_value || !stageField?.field_value) {
            const defaults = await getDefaultPipelineAndStage();

            if (!pipelineField) {
                allFields.push({
                    field_name: 'deal_pipeline_id',
                    field_value: defaults.pipeline_id,
                    custom_field: 0
                });
            } else if (!pipelineField.field_value) {
                pipelineField.field_value = defaults.pipeline_id;
            }

            if (!stageField) {
                allFields.push({
                    field_name: 'deal_stage_id',
                    field_value: defaults.stage_id,
                    custom_field: 0
                });
            } else if (!stageField.field_value) {
                stageField.field_value = defaults.stage_id;
            }
        }

        const custom_fields = Array.from(allFields)
            .filter(field => field.custom_field == 1)
            .map(field => ({ [field.field_name]:  field.field_name !== 'created_by' ? field.field_value :  user_id}));

        const standardFields = Array.from(allFields)
            .filter(field => field.custom_field == 0);

        const deal = [
            ...standardFields.map(field => field.field_value),
            JSON.stringify(custom_fields),
            1 // active status
        ];

        const keys = standardFields
            .map(field => field.field_name)
            .concat(['custom_field', 'active'])
            .join(', ');
        
        const placeholders = '?' + ', ?'.repeat(deal.length - 1);

        const [result] = await connection.execute(
            `INSERT INTO deals (${keys}) VALUES (${placeholders})`,
            deal
        );

        await connection.commit();        
        return result.insertId;

    } catch (error) {
        if (connection) {
            await connection.rollback();
        }
        throw new Error(`Failed to create deal: ${error.message}`);
    } finally {
        if (connection) {
            connection.release();
        }
    }
}

async function findAll(filters = {}) {
    let query = 'SELECT * FROM deals WHERE active = 1';
    const params = [];

    // Add search functionality
    if (filters.search) {
        query += ' AND (name LIKE ? OR amount LIKE ? OR stage LIKE ?)';
        const searchTerm = `%${filters.search}%`;
        params.push(searchTerm, searchTerm, searchTerm);
    }

    // Add specific field filters
    if (filters.name) {
        query += ' AND name = ?';
        params.push(filters.name);
    }
    if (filters.stage) {
        query += ' AND stage = ?';
        params.push(filters.stage);
    }

    // Add sorting
    query += ' ORDER BY created_at DESC';

    const [rows] = await db.execute(query, params);
    return rows;
}

// async function findById(id) {
//     try {
//         // Fetch deal with related entities
//         const dealQuery = `
//             SELECT 
//                 d.*, t.id as task_id, m.id as meeting_id,
//                 t.active as task_status, m.active as meeting_status,
//                 t.title as task_title, t.description as task_description,
//                 sa.id as sales_account_id, sa.name as sales_account_name,
//                 c.id as contact_id, CONCAT(c.first_name, ' ', c.last_name) as contact_name,
//                 dp.id as pipeline_id, dp.pipeline_name,
//                 ds.id as stage_id, ds.deal_stage, ds.probability, ds.index, ds.active,
//                 n.id as note_id, n.notes, n.voicenote, n.belongsto_id, n.belongsto,
//                 n.users_id, n.note_time, n.active as note_active, n.is_deleted,
//                 n.created_at as note_created_at, n.updated_at as note_updated_at,
//                 u.name as user_name
//             FROM deals d
//             LEFT JOIN tasks t ON t.related_to = d.id AND t.related_type = 'deals' AND t.active = 1
//             LEFT JOIN salesactivitydatas m ON m.salesactivities_id = 3 AND m.active = 1 
//                 AND m.targetable_id = d.id AND m.targetable_type = 'deals'
//             LEFT JOIN accounts sa ON sa.id = d.sales_account_id
//             LEFT JOIN contacts c ON c.id = d.contacts
//             LEFT JOIN dealpipelines dp ON dp.id = d.deal_pipeline_id
//             LEFT JOIN dealstages ds ON ds.id = d.deal_stage_id
//             LEFT JOIN notes n ON n.belongsto_id = d.id AND n.belongsto = 'deals' AND n.is_deleted = 0
//             LEFT JOIN users u ON u.id = n.users_id
//             WHERE d.id = ? AND d.active = 1
//         `;

//         const [rows] = await db.execute(dealQuery, [id]);
//         if (!rows.length) return null;

//         const mainDeal = rows[0];

//         // Fetch all active fields and choices
//         const [fields] = await db.execute('SELECT * FROM dealfields WHERE active = 1 ORDER BY `order` ASC');
//         const [choices] = await db.execute('SELECT * FROM dealchoices');

//         // Process fields with custom and lookup logic
//         const processedFields = await processFields(fields, choices, mainDeal);

//         // Fetch all stages for this pipeline
//         const [pipelineStages] = await db.execute(`
//             SELECT id, deal_stage AS name, probability, \`index\`, active
//             FROM dealstages 
//             WHERE dealpipelines_id = ? AND active = 1
//             ORDER BY \`index\` ASC
//         `, [mainDeal.deal_pipeline_id]);

//         // Format tasks, notes, etc.
//         const tasks = deduplicate(rows, 'task_id').map(row => ({
//             id: row.task_id,
//             title: row.task_title,
//             description: row.task_description
//         }));

//         const notes = deduplicate(rows, 'note_id').map(row => ({
//             id: row.note_id,
//             text: row.notes,
//             voice_note: row.voicenote
//         }));

//         const meetings = deduplicate(rows, 'meeting_id');

//         return {
//             id: id.toString(),
//             deals: processedFields,
//             tasks,
//             notes,
//             notes_count: notes.length,
//             task_count: tasks.length,
//             meeting_count: meetings.length,
//             sales_account: mainDeal.sales_account_id ? {
//                 id: mainDeal.sales_account_id,
//                 name: mainDeal.sales_account_name
//             } : null,
//             contact: mainDeal.contact_id ? {
//                 id: mainDeal.contact_id,
//                 name: mainDeal.contact_name
//             } : null,
//             pipeline: mainDeal.pipeline_id ? {
//                 id: mainDeal.pipeline_id,
//                 name: mainDeal.pipeline_name
//             } : null,
//             current_stage: mainDeal.stage_id ? {
//                 id: mainDeal.stage_id,
//                 name: mainDeal.deal_stage,
//                 probability: mainDeal.probability,
//                 index: mainDeal.index,
//                 active: mainDeal.active
//             } : null,
//             pipeline_stages: pipelineStages
//         };

//     } catch (error) {
//         console.error('Error in findById:', error);
//         throw new Error(`Failed to fetch deal: ${error.message}`);
//     }
// }
async function findById(id) {
    const connection = await db.getConnection();
    try {
        // 1. Main deal
        const [dealRows] = await connection.execute(`SELECT * FROM deals WHERE id = ? AND active = 1`, [id]);
        if (!dealRows.length) return null;
        const mainDeal = dealRows[0];

        // 2. Get user's role and access scope for field processing
        const [userRole] = await connection.execute(
            `SELECT r.access_scope, r.id as role_id, u.territory_id 
             FROM users u 
             JOIN roles r ON u.roles_id = r.id 
             WHERE u.id = ?`,
            [mainDeal.owner_id]
        );

        const accessScope = userRole.length ? JSON.parse(userRole[0].access_scope) : {};
        const territories = userRole.length && userRole[0].territory_id ? userRole[0].territory_id.split(',') : [];

        // 3. Fields and Choices
        const [fields] = await connection.execute(`SELECT * FROM dealfields WHERE active = 1 ORDER BY \`order\` ASC`);
        const [choices] = await connection.execute(`SELECT * FROM dealchoices`);
        const [pipeline_id] = await connection.execute('SELECT id, pipeline_name FROM dealpipelines');
        const [stage_id] = await connection.execute('SELECT id, deal_stage FROM dealstages');

        // Process fields similar to dealField findAll
        const processedFields = await Promise.all(fields.map(async field => {
            // Handle regular choices
            field.choices = choices.filter(choice => choice.dealfield_id === field.id);
            field.field_value = mainDeal[field.field_name];

            // Handle custom fields
            if (field.custom_field === 1 && mainDeal.custom_field) {
                const customFields = JSON.parse(mainDeal.custom_field || '{}');
                const fieldName = field.field_name;
                const match = customFields.find(c => Object.hasOwn(c, fieldName));
                field.field_value = match ? match[fieldName] : null;
            }

            // Handle lookup type fields
            if (field.field_type === 'lookup' && field.lookup_type && field.lookup_column) {
                try {
                    let query = `SELECT id, ${field.lookup_column} FROM ${field.lookup_type} WHERE active = 1`;
                    const params = [];

                    if (field.lookup_type === 'contacts') {
                        query += ' AND owner_id = ?';
                        params.push(mainDeal.owner_id);
                    } else if (field.lookup_type === 'sales_accounts') {
                        query += ' AND owner_id = ?';
                        params.push(mainDeal.owner_id);
                    } else if (field.lookup_type === 'deals') {
                        query += ' AND owner_id = ?';
                        params.push(mainDeal.owner_id);
                    }

                    // Get view permission for the specific module
                    const modulePermission = field.lookup_type === 'contacts' ? accessScope?.contact?.view :
                                          field.lookup_type === 'accounts' ? accessScope?.sales_account?.view :
                                          field.lookup_type === 'deals' ? accessScope?.deal?.view : 'global';

                    switch (modulePermission) {
                        case 'owned':
                            query += ' AND owner_id = ?';
                            params.push(mainDeal.owner_id);
                            break;

                        case 'territory':
                            if (territories.length) {
                                const placeholders = territories.map(() => '?').join(',');
                                query += ` AND territory_id IN (${placeholders})`;
                                params.push(...territories);
                            } else {
                                field.choices = [];
                                return field;
                            }
                            break;

                        case 'restricted':
                            field.choices = [];
                            return field;

                        case 'global':
                        default:
                            // No additional filters for global access
                            break;
                    }

                    // Execute filtered query
                    const [lookupData] = await connection.execute(query, params);

                    // Format lookup data as choices
                    field.choices = lookupData.map(item => ({
                        id: item.id,
                        dealfield_id: field.id,
                        custom_option: field.lookup_column.includes(', ') ? 
                            item[field.lookup_column.split(', ')[0]] + ' ' + item[field.lookup_column.split(', ')[1]] : 
                            item[field.lookup_column],
                    }));

                    if (field.lookup_type === 'contacts') {
                        field.choices = field.choices.map(item => ({
                            ...item,
                            related_id: lookupData.find(lookup => lookup.id === item.id)?.sales_accounts
                        }));
                    }
                } catch (error) {
                    console.error(`Error fetching lookup data for field ${field.field_name}:`, error);
                    field.choices = [];
                }
            }

            // Handle pipeline field
            if (field.field_name.toLowerCase() === 'deal_pipeline_id') {
                try {
                    field.choices = pipeline_id.map(item => ({
                        id: item.id,
                        custom_option: item['pipeline_name']
                    }));
                } catch (error) {
                    console.error(`Error fetching pipeline data for field ${field.field_name}:`, error);
                    field.choices = [];
                }
            }

            // Handle stage field
            if (field.field_name.toLowerCase() === 'deal_stage_id') {
                try {
                    field.choices = stage_id.map(item => ({
                        id: item.id,
                        custom_option: item['deal_stage']
                    }));
                } catch (error) {
                    console.error(`Error fetching stage data for field ${field.field_name}:`, error);
                    field.choices = [];
                }
            }

            return {
                id: field.id,
                custom_field: field.custom_field,
                field_label: field.field_label,
                field_name: field.field_name,
                field_type: field.field_type,
                required: field.required,
                quick_add: field.quick_add,
                tool_tip: field.tool_tip,
                placeholder: field.placeholder,
                choices: field.choices,
                read_only: field.read_only,
                has_dependent: field.has_dependent,
                dealgroups_id: field.dealgroups_id,
                active: field.active,
                order: field.order,
                lookup_type: field.lookup_type,
                lookup_column: field.lookup_column,
                created_at: field.created_at,
                updated_at: field.updated_at,
                field_value: field.field_value
            };
        }));

        // Get pipeline ID from custom fields if not in standard fields
        let pipelineId = mainDeal.deal_pipeline_id;
        if (!pipelineId && mainDeal.custom_field) {
            try {
                const customFields = JSON.parse(mainDeal.custom_field);
                const pipelineField = customFields.find(field => Object.keys(field)[0] === 'deal_pipeline_id');
                if (pipelineField) {
                    pipelineId = pipelineField.deal_pipeline_id;
                }
            } catch (error) {
                console.error('Error parsing custom fields for pipeline ID:', error);
            }
        }
        
        console.log('pipeline ID:', pipelineId);
        
        // 4. Pipeline stages
        const [pipelineStages] = await connection.execute(`
            SELECT id, dealpipelines_id, deal_stage AS name, probability, \`index\`, active
            FROM dealstages 
            WHERE dealpipelines_id = ? AND active = 1 
            ORDER BY \`index\` ASC
        `, [pipelineId]);

        // 5. Tasks, Notes, Meetings
        const [tasks] = await connection.execute(`
            SELECT * FROM tasks 
            WHERE related_to = ? AND related_type = 'deals' AND active = 1
        `, [id]);

        const [notes] = await connection.execute(`
            SELECT * FROM notes 
            WHERE belongsto_id = ? AND belongsto = 'deals' AND is_deleted = 0
        `, [id]);

        const [meetings] = await connection.execute(`
            SELECT * FROM salesactivitydatas 
            WHERE salesactivities_id = 3 AND active = 1 AND targetable_id = ? AND targetable_type = 'deals'
        `, [id]);

        // 6. Sales account
        const salesAccount = mainDeal.sales_account_id
            ? (await connection.execute(`SELECT id, name FROM accounts WHERE id = ? AND active = 1`, [mainDeal.sales_account_id]))[0][0]
            : null;

        // 7. Contact
        const contact = mainDeal.contacts
            ? (await connection.execute(`SELECT id, CONCAT(first_name, ' ', last_name) AS name FROM contacts WHERE id = ? AND active = 1`, [mainDeal.contacts]))[0][0]
            : null;

        // 8. Pipeline (optional - if name needed)
        const pipeline = pipelineId
            ? (await connection.execute(`SELECT id, pipeline_name as name FROM dealpipelines WHERE id = ? AND active = 1`, [pipelineId]))[0][0]
            : null;

        // 9. Current stage
        const current_stage = pipelineStages.find(stage => stage.id === mainDeal.deal_stage_id) || null;

        return {
            id: id.toString(),
            deals: processedFields,
            notes_count: notes.length,
            task_count: tasks.length,
            meeting_count: meetings.length,
            sales_account: salesAccount,
            contact,
            pipeline,
            current_stage,
            pipeline_stages: pipelineStages
        };

    } catch (error) {
        console.error('Error in findById:', error);
        throw new Error(`Failed to fetch deal: ${error.message}`);
    } finally {
        connection.release();
    }
}

async function update(id, Data) {
    console.log("Data", Data);
    let connection;
    try {
        connection = await db.getConnection();
        await connection.beginTransaction();

        const { deal: allFields, user_id } = Data;
        console.log("allFields", allFields);
        const [response] = await connection.execute('SELECT * FROM deals WHERE id = ?', [id]);
        
        if (!response.length) {
            await connection.rollback();
            return false;
        }
        
        if (!allFields || !Array.isArray(allFields)) {
            throw new Error('Invalid input data format');
        }
        
        // Get unique fields from dealfields table
        const [uniqueFields] = await connection.execute(
            'SELECT field_name FROM dealfields WHERE unique_field = "on"'
        );

        // Check for uniqueness constraints
        for (const field of allFields) {
            if (uniqueFields.some(uf => uf.field_name === field.field_name) && field.field_value) {
                // Check if value already exists for other deals
                const [existing] = await connection.execute(
                    `SELECT id FROM deals WHERE ${field.field_name} = ? AND active = 1 AND id != ?`,
                    [field.field_value, id]
                );

                if (existing.length > 0) {
                    await connection.rollback();
                    console.log(`Deal with ${field.field_name} "${field.field_value}" already exists`)
                    // throw new Error(`Deal with ${field.field_name} "${field.field_value}" already exists`);
                    return {
                        status: 400,
                        success: false,
                        message: `Deal with ${field.field_name} "${field.field_value}" already exists`
                    }
                }
            }
        }

        // Check and set pipeline_id and stage_id
        const pipelineField = allFields.find(field => field.field_name === 'deal_pipeline_id');
        const stageField = allFields.find(field => field.field_name === 'deal_stage_id');

        if (!pipelineField?.field_value || !stageField?.field_value) {
            const defaults = await getDefaultPipelineAndStage();

            if (!pipelineField) {
                allFields.push({
                    field_name: 'deal_pipeline_id',
                    field_value: defaults.pipeline_id,
                    custom_field: 0
                });
            } else if (!pipelineField.field_value) {
                pipelineField.field_value = defaults.pipeline_id;
            }

            if (!stageField) {
                allFields.push({
                    field_name: 'deal_stage_id',
                    field_value: defaults.stage_id,
                    custom_field: 0
                });
            } else if (!stageField.field_value) {
                stageField.field_value = defaults.stage_id;
            }
        }

        // const custom_fields = allFields
        //     .filter(field => field.custom_field === 1)
        //     .reduce((acc, field) => ({ ...acc, [field.field_name]: field.field_name !== 'modified_by' ? field.field_value : user_id }), {});

        const custom_fields = Array.from(allFields)
        .filter(field => field.custom_field == 1)
        .map(field => ({ [field.field_name]:  field.field_name !== 'modified_by' ? field.field_value :  user_id}));

        const standardFields = allFields.filter(field => field.custom_field === 0);

        // Construct UPDATE query dynamically
        const updateFields = standardFields
            .map(field => `${field.field_name} = ?`)
            .concat(['custom_field = ?', 'active = ?'])
            .join(', ');

        const updateValues = [
            ...standardFields.map(field => field.field_value),
            JSON.stringify(custom_fields),
            1
        ];

        const [result] = await connection.execute(
            `UPDATE deals SET ${updateFields} WHERE id = ?`,
            [...updateValues, id]
        );

        await connection.commit();
        return result.affectedRows > 0;
    } catch (err) {
        if (connection) await connection.rollback();
        throw err;
    } finally {
        if (connection) connection.release();
    }
}

async function deleteRecord(id) {
    const [result] = await db.execute('UPDATE deals SET active = 0 WHERE id = ?', [id]);
    return result.affectedRows > 0;
}

async function findDealMeetings(dealId) {
    try {
        const [meetings] = await db.execute(
            `SELECT 
                sad.*,
                u.name as owner_name,
                COALESCE(sao.outcome, '') as outcome_name,
                sa.activity_name,
                sa.activity_icon
             FROM salesactivitydatas sad
             LEFT JOIN users u ON sad.owner_id = u.id
             LEFT JOIN salesactivityoutcomes sao ON sad.outcome = sao.id
             LEFT JOIN salesactivities sa ON sad.salesactivities_id = sa.id
             WHERE sad.active=1 and (sad.targetable_type = 'deals' OR sad.targetable_type = 'Deals')
             AND sad.targetable_id = ?
             AND sad.salesactivities_id = 3
             AND sad.active = 1 
             ORDER BY sad.start_date DESC, sad.start_time DESC`,
             [dealId]
            );
            // AND sad.active=0

        return meetings.map(meeting => ({
            id: meeting.id,
            title: meeting.title,
            description: meeting.description,
            activity_name: meeting.activity_name,
            activity_icon: meeting.activity_icon,
            activity_title: meeting.activity_title,
            start_date: meeting.start_date,
            end_date: meeting.end_date,
            start_time: formatTime(meeting.start_time),
            end_time: formatTime(meeting.end_time),
            duration: meeting.duration,
            outcome: meeting.outcome_name,
            status: meeting.status,
            priority: meeting.priority,
            location: meeting.location,
            owner_id: meeting.owner_id,
            salesactivities_id: meeting.salesactivities_id,
            targetable_type: meeting.targetable_type,
            targetable_id: meeting.targetable_id,
            transport_mode: meeting.transport_mode,
            distance: meeting.distance,
            claim_amount: meeting.claim_amount,
            checkin_location: meeting.checkin_location,
            checkin_datetime: formatDateTime(meeting.checkin_datetime),
            checkout_location: meeting.checkout_location,
            checkout_datetime: formatDateTime(meeting.checkout_datetime),
            custom_field: meeting.custom_field ? JSON.parse(meeting.custom_field) : null,
            allow_reschedule: meeting.allow_reschedule,
            allow_complete: meeting.allow_complete,
            active: meeting.active,
            created_at: formatDateTime(meeting.created_at),
            updated_at: formatDateTime(meeting.updated_at)
        }));

    } catch (error) {
        console.error('Error in findDealMeetings:', error);
        throw new Error(`Failed to fetch deal meetings: ${error.message}`);
    }
}

async function findByDeals(dealid) {
    try {
        const [rows] = await db.execute(
            `SELECT t.*, 
                u.name as owner_name,
                COALESCE(t.task_type, 'General') as task_type_name
             FROM tasks t
             LEFT JOIN users u ON t.user_id = u.id
             WHERE t.active=1 and (t.related_type = "Deals" OR t.related_type = "deals")
             AND t.related_to = ?
             ORDER BY t.due_date DESC, t.due_time DESC`,
            [dealid]
        );

        return {
            status: 200,
            message: "Tasks fetched successfully",
            tasks: rows.map(task => ({
                id: task.id,
                title: task.title,
                description: task.description,
                task_type: task.task_type_name,
                related_type: task.related_type,
                related_to: task.related_to,
                mobile_number: task.mobile_number,
                due_date: formatDate(task.due_date) ? formatDate(task.due_date).replaceAll('/', '-').split('-').reverse().join('-') : null,
                due_time: formatTime(task.due_time),
                mark_as_complete: task.mark_as_complete,
                completed_date: formatDate(task.completed_date) ? formatDate(task.completed_date).replaceAll('/', '-').split('-').reverse().join('-') : null,
                completed_time: formatTime(task.completed_time),
                collaboratoes: task.collaboratoes,
                owner: task.owner,
                user_id: task.user_id,
                owner_name: task.owner_name || '',
                outcome: task.outcome,
                active: task.active,
                snooze_date: formatDate(task.snooze_date) ? formatDate(task.snooze_date).replaceAll('/', '-').split('-').reverse().join('-') : null,
                snooze_time: formatTime(task.snooze_time),
                created_at: formatDateTime(task.created_at),
                updated_at: formatDateTime(task.updated_at),
                status: task.mark_as_complete ? 'Completed' : 'Pending'
            }))
        };
    } catch (error) {
        console.error('Detailed error:', error);
        throw new Error(`Failed to fetch deal tasks: ${error.message}`);
    }
}

// Helper functions for date/time formatting
function formatDate(date) {
    if (!date) return null;
    return new Date(date).toLocaleDateString('en-GB', {
        day: '2-digit',
        month: '2-digit',
        year: 'numeric'
    });
}

function formatTime(time) {
    if (!time) return null;
    return new Date(`1970-01-01T${time}`).toLocaleTimeString('en-US', {
        hour: '2-digit',
        minute: '2-digit',
        hour12: true
    });
}

function formatDateTime(datetime) {
    if (!datetime) return null;
    return new Date(datetime).toISOString();
}

async function addProducts(dealId, products, dealValue, userId) {
    try {
        // Start transaction
        await db.execute('START TRANSACTION');

        // Get deal
        const [deals] = await db.execute(
            'SELECT * FROM deals WHERE id = ?',
            [dealId]
        );

        if (deals.length === 0) {
            return { success: false, message: 'Deal not found' };
        }

        const deal = deals[0];
        let totalPrice = 0;
        let totalItems = 0;

        // Process each product
        for (const product of products) {
            const totalAmount = product.unitprice * product.quantity;

            if (product.id) {
                // Update existing deal product
                await db.execute(
                    `UPDATE dealproducts SET 
                        product_id = ?,
                        unitprice = ?,
                        quantity = ?,
                        discount_type = ?,
                        discount = ?,
                        totalprice = ?,
                        setupfee = ?,
                        billing_cycle = ?,
                        no_of_billing_cycle = ?
                     WHERE id = ?`,
                    [
                        product.product_id,
                        product.unitprice,
                        product.quantity,
                        product.discount_type,
                        product.discount,
                        totalAmount,
                        product.setupfee || null,
                        product.billing_cycle || null,
                        product.no_of_billing_cycle || null,
                        product.id
                    ]
                );
            } else {
                // Insert new deal product
                await db.execute(
                    `INSERT INTO dealproducts 
                     (deal_id, product_id, unitprice, quantity, discount_type, 
                      discount, totalprice, setupfee, billing_cycle, 
                      no_of_billing_cycle, user_id) 
                     VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`,
                    [
                        dealId,
                        product.product_id,
                        product.unitprice,
                        product.quantity,
                        product.discount_type,
                        product.discount,
                        totalAmount,
                        product.setupfee || null,
                        product.billing_cycle || null,
                        product.no_of_billing_cycle || null,
                        userId
                    ]
                );
            }

            totalPrice += totalAmount;
            totalItems++;
        }

        // Update deal amount
        await db.execute(
            'UPDATE deals SET amount = ? WHERE id = ?',
            [dealValue, dealId]
        );

        // Update contact deal statistics
        const [contactStats] = await db.execute(
            `SELECT 
                COUNT(*) as deal_count,
                SUM(CASE WHEN deal_stage IN ('Won', 'Lost') THEN amount ELSE 0 END) as open_deal_sum,
                COUNT(CASE WHEN deal_stage IN ('Won', 'Lost') THEN 1 END) as open_deal_count
             FROM deals 
             WHERE contacts = ?`,
            [deal.contacts]
        );

        await db.execute(
            `UPDATE contacts SET 
                dealcount = ?,
                open_dealvalue = ?,
                open_dealcount = ?
             WHERE id = ?`,
            [
                contactStats[0].deal_count,
                contactStats[0].open_deal_sum || 0,
                contactStats[0].open_deal_count || 0,
                deal.contacts
            ]
        );

                return { success: true };

    } catch (error) {
        console.error('Error in addProducts:', error);
        throw new Error(`Failed to add deal products: ${error.message}`);
    }
}

// get deals stages api
async function getdealstages() {
    // First, get all pipelines with their stages
    const [rows] = await db.execute(`
        SELECT 
            dp.id as pipeline_id,
            dp.pipeline_name,
            ds.id as stage_id,
            ds.deal_stage,
            ds.probability,
            ds.\`index\`,
            ds.active
        FROM dealpipelines dp
        LEFT JOIN dealstages ds ON dp.id = ds.dealpipelines_id
        WHERE dp.active = 1
        ORDER BY dp.id, ds.\`index\`
    `);

    // Transform the flat results into grouped structure
    const groupedStages = rows.reduce((acc, row) => {
        if (!acc[row.pipeline_id]) {
            acc[row.pipeline_id] = {
                pipeline_id: row.pipeline_id,
                pipeline_name: row.pipeline_name,
                stages: []
            };
        }
        
        if (row.stage_id) {  // Only add if stage exists
            acc[row.pipeline_id].stages.push({
                id: row.stage_id,
                deal_stage: row.deal_stage,
                probability: row.probability,
                index: row.index,
                active: row.active
            });
        }
        
        return acc;
    }, {});

    // Convert to array and return
    return Object.values(groupedStages);
}

async function createdealstages(data) {
    const [result] = await db.execute(
        `INSERT INTO dealstages (deal_stage, dealpipelines_id, probability, \`index\`, active, created_at, updated_at) 
         VALUES (?, ?, ?, ?, ?, NOW(), NOW())`,
        [
          data.deal_stage,
          data.dealpipelines_id,
          data.probability,
          data.index,
          data.active || 1 
        ]
      );
      
    return result;
}

async function updatedealstages(inputData) {
    const connection = await db.getConnection();
    
    try {
        await connection.beginTransaction();

        // Convert input to array if it's a single object
        const dataArray = Array.isArray(inputData) ? inputData : [inputData];
        const finalResults = [];

        for (const data of dataArray) {
            const { pipeline, stages } = data;
            let pipelineId = pipeline?.id;

            // Handle pipeline update/creation first
            if (pipeline) {
                if (pipeline.id) {
                    // Update existing pipeline
                    const [pipelineResult] = await connection.execute(
                        `UPDATE dealpipelines 
                        SET pipeline_name = ?,
                            active = ?,
                            updated_at = NOW()
                        WHERE id = ?`,
                        [
                            pipeline.pipeline_name,
                            pipeline.active ?? 1,
                            pipeline.id
                        ]
                    );
                } else {
                    // Create new pipeline
                    const [pipelineResult] = await connection.execute(
                        `INSERT INTO dealpipelines (
                            pipeline_name,
                            active,
                            created_at,
                            updated_at
                        ) VALUES (?, ?, NOW(), NOW())`,
                        [
                            pipeline.pipeline_name,
                            pipeline.active ?? 1
                        ]
                    );
                    pipelineId = pipelineResult.insertId;
                }
            }

            // Validate stages is not undefined
            if (!stages || !Array.isArray(stages)) {
                throw new Error('Stages array is required');
            }

            // Update stages with the correct pipeline ID
            const updatedStages = stages.map(stage => {
                // Ensure stage.probability is always a string or number before converting
                let probability = 0;
                if (stage.probability !== undefined && stage.probability !== null) {
                    probability = typeof stage.probability === 'string' 
                        ? parseInt(stage.probability.replace('%', '')) 
                        : parseInt(String(stage.probability).replace('%', ''));
                }

                return {
                    deal_stage: stage.deal_stage || '',
                    dealpipelines_id: stage.dealpipelines_id || pipelineId,
                    probability: probability,
                    index: parseInt(stage.index || '0'),
                    active: stage.active ?? 1,
                    id: stage.id || null
                };
            });

            // Group stages by pipeline
            const pipelineStages = {};
            updatedStages.forEach(stage => {
                const pId = stage.dealpipelines_id;
                if (!pId) {
                    throw new Error('Pipeline ID is required for all stages');
                }
                
                if (!pipelineStages[pId]) {
                    pipelineStages[pId] = [];
                }
                pipelineStages[pId].push(stage);
            });

            // Validate sequential indexes for each pipeline
            for (const pId in pipelineStages) {
                const pipelineStagesList = pipelineStages[pId];
                
                // Create array of indexes for checking order
                const indexes = pipelineStagesList.map(stage => stage.index);
                
                // Check if indexes are sequential and in order
                for (let i = 0; i < indexes.length; i++) {
                    if (indexes[i] !== i + 1) {
                        const expectedIndex = i + 1;
                        const foundIndex = indexes[i];
                        const stageName = pipelineStagesList.find(s => s.index === foundIndex)?.deal_stage || 'Unknown';
                        
                        throw new Error(
                            `Invalid stage order in pipeline ${pId}. ` +
                            `Found stage "${stageName}" at position ${i + 1} with index ${foundIndex}. ` +
                            `Expected index ${expectedIndex}. ` +
                            `Stages must be in order: 1,2,3,4,5...`
                        );
                    }
                }

                // Check for gaps
                const maxIndex = Math.max(...indexes);
                if (maxIndex !== indexes.length) {
                    throw new Error(
                        `Invalid stage sequence in pipeline ${pId}. ` +
                        `Found ${indexes.length} stages but highest index is ${maxIndex}. ` +
                        `Indexes must be sequential from 1 to ${indexes.length} without gaps.`
                    );
                }
            }

            // Check for duplicate indexes within the same pipeline
            const indexMap = new Map();
            updatedStages.forEach(stage => {
                const key = `${stage.dealpipelines_id}-${stage.index}`;
                if (indexMap.has(key)) {
                    const existingStage = updatedStages.find(s => 
                        s.dealpipelines_id === stage.dealpipelines_id && 
                        s.index === stage.index && 
                        s !== stage
                    );
                    throw new Error(
                        `Duplicate index ${stage.index} found in pipeline ${stage.dealpipelines_id}. ` +
                        `Both "${existingStage?.deal_stage || 'Unknown'}" and "${stage.deal_stage || 'Unknown'}" have the same index.`
                    );
                }
                indexMap.set(key, true);
            });

            const results = {
                pipeline: null,
                stages: []
            };
            
            // Get pipeline details
            if (pipelineId) {
                const [pipelineDetails] = await connection.execute(
                    'SELECT * FROM dealpipelines WHERE id = ?',
                    [pipelineId]
                );
                results.pipeline = pipelineDetails[0] || null;
            }

            // First, get all existing stages for the affected pipelines
            const pipelineIds = Array.from(new Set(updatedStages.map(s => s.dealpipelines_id).filter(Boolean)));
            
            if (pipelineIds.length === 0) {
                throw new Error('No valid pipeline IDs found in stages');
            }
            
            // Get existing stages that aren't in our update list
            const stageIdsToUpdate = updatedStages.filter(s => s.id).map(s => s.id).filter(Boolean);
            let existingSql = `
                SELECT id, dealpipelines_id, \`index\` 
                FROM dealstages 
                WHERE dealpipelines_id IN (${pipelineIds.join(',')})
            `;
            
            if (stageIdsToUpdate.length > 0) {
                existingSql += ` AND id NOT IN (${stageIdsToUpdate.join(',')})`;
            }
            
            const [existingStages] = await connection.execute(existingSql);

            // Temporarily update indexes of existing stages to avoid conflicts
            if (existingStages.length > 0) {
                await connection.execute(
                    `UPDATE dealstages 
                    SET \`index\` = \`index\` + 1000 
                    WHERE id IN (${existingStages.map(s => s.id).join(',')})`
                );
            }
            
            // Process each stage
            for (const stage of updatedStages) {
                let result;
                try {
                    if (stage.id) {
                        // Update existing stage
                        [result] = await connection.execute(
                            `UPDATE dealstages 
                            SET deal_stage = ?,
                                dealpipelines_id = ?,
                                probability = ?,
                                \`index\` = ?,
                                active = ?,
                                updated_at = NOW()
                            WHERE id = ?`,
                            [
                                stage.deal_stage,
                                stage.dealpipelines_id,
                                stage.probability,
                                stage.index,
                                stage.active,
                                stage.id
                            ]
                        );

                        // Get updated stage details
                        const [updatedStageDetails] = await connection.execute(
                            'SELECT * FROM dealstages WHERE id = ?',
                            [stage.id]
                        );

                        if (updatedStageDetails && updatedStageDetails.length > 0) {
                            results.stages.push({
                                ...updatedStageDetails[0],
                                isNewStage: false
                            });
                        }
                    } else {
                        // Create new stage
                        [result] = await connection.execute(
                            `INSERT INTO dealstages (
                                deal_stage,
                                dealpipelines_id,
                                probability,
                                \`index\`,
                                active,
                                created_at,
                                updated_at
                            ) VALUES (?, ?, ?, ?, ?, NOW(), NOW())`,
                            [
                                stage.deal_stage,
                                stage.dealpipelines_id,
                                stage.probability,
                                stage.index,
                                stage.active || 1,
                            ]
                        );

                        // Get new stage details
                        const [newStageDetails] = await connection.execute(
                            'SELECT * FROM dealstages WHERE id = ?',
                            [result.insertId]
                        );

                        if (newStageDetails && newStageDetails.length > 0) {
                            results.stages.push({
                                ...newStageDetails[0],
                                isNewStage: true
                            });
                        }
                    }
                } catch (error) {
                    console.error('Error processing stage:', error);
                    throw new Error(`Failed to process stage "${stage.deal_stage}": ${error.message}`);
                }
            }

            // Delete any remaining stages that weren't in the update
            if (results.stages.length > 0) {
                const stageIds = results.stages.map(r => r.id).filter(Boolean);
                
                if (stageIds.length > 0 && pipelineIds.length > 0) {
                    try {
                        await connection.execute(
                            `DELETE FROM dealstages 
                            WHERE dealpipelines_id IN (${pipelineIds.join(',')})
                            AND id NOT IN (${stageIds.join(',')})`
                        );
                    } catch (error) {
                        console.error('Error deleting unused stages:', error);
                        // Don't throw here, as we've already processed the main stages
                    }
                }
            }

            finalResults.push(results);
        }

        await connection.commit();
        return finalResults;

    } catch (error) {
        await connection.rollback();
        throw error;
    } finally {
        connection.release();
    }
}

// deals satge list by id   
async function getDealStageslist(id) {
    const [rows] = await db.execute('SELECT id, deal_stage, dealpipelines_id, probability, \`index\` FROM dealstages WHERE dealpipelines_id = ? ORDER BY \`index\`', [id]);
    return rows;
}

// delete deal stages
// async function deletedealstages(id) {
//     const [result] = await db.execute('DELETE FROM dealstages WHERE id = ?', [id]);
//     return result;
// }

// deals pipeline 
async function getdealPipelines() {
    const [rows] = await db.execute(
        'SELECT * FROM dealpipelines WHERE active = 1 ORDER BY id DESC'
    );
    return rows;
}

// get deal relations`
async function getDealRelations(dealId) {
    try {
        // Get deal's related account and contact
        const query = `
            SELECT 
            d.id AS deal_id,
            d.sales_account_id,
            d.contacts AS contact_id,
            
            a.name AS account_name,
            a.website AS account_website,
            a.phone AS account_phone,
            CONCAT(c.first_name, ' ', c.last_name) AS contact_name,
            c.emails AS contact_email,
            c.mobile_number AS phone_number
            FROM deals d
            LEFT JOIN accounts a ON d.sales_account_id = a.id
            LEFT JOIN contacts c ON d.contacts = c.id
            WHERE d.id = ? AND d.active = 1`;

        const [rows] = await db.execute(query, [dealId]);

        if (!rows.length) {
            return null;
        }

        const result = {
            deal_id: rows[0].deal_id,
            account: [rows[0].sales_account_id ? {
                id: rows[0].sales_account_id,
                name: rows[0].account_name,
                website: rows[0].account_website,
                phone: rows[0].account_phone,
            } : null],
            contact: [rows[0].contact_id ? {
                id: rows[0].contact_id,
                name: rows[0].contact_name,
                email: rows[0].contact_email,
                phone: rows[0].phone_number
            } : null]
        };

        return result;

    } catch (error) {
        console.error('Error in getDealRelations:', error);
        throw new Error(`Failed to fetch deal relations: ${error.message}`);
    }
}
async function getDefaultPipelineAndStage() {
    try {
        // Get default pipeline (first active pipeline)
        const [pipelines] = await db.execute(
            'SELECT id FROM dealpipelines WHERE active = 1 and is_default=1 ORDER BY id ASC LIMIT 1'
        );

        if (!pipelines.length) {
            throw new Error('No active pipeline found');
        }

        // Get first stage of the default pipeline
        const [stages] = await db.execute(
            'SELECT id FROM dealstages WHERE dealpipelines_id = ? AND active = 1 ORDER BY `index` ASC LIMIT 1',
            [pipelines[0].id]
        );

        if (!stages.length) {
            throw new Error('No active stages found for the pipeline');
        }

        return {
            pipeline_id: pipelines[0].id,
            stage_id: stages[0].id
        };
    } catch (error) {
        throw new Error(`Failed to get default pipeline and stage: ${error.message}`);
    }
}
async function updateDealStage(dealId, stageId, userId) {
    try {
        //await db.execute('START TRANSACTION');

        try {
            // Get stage details
            const [stages] = await db.execute(
                'SELECT * FROM dealstages WHERE id = ? AND active = 1',
                [stageId]
            );

            if (!stages.length) {
                throw new Error('Invalid stage ID');
            }

            // Update deal stage
            const [result] = await db.execute(
                `UPDATE deals 
                 SET deal_stage_id = ?,
                     stage_updated_at = CURRENT_TIMESTAMP,
                     updated_by = ? 
                 WHERE id = ? AND active = 1`,
                [stageId, userId, dealId]
            );

            if (result.affectedRows === 0) {
                throw new Error('Deal not found or already inactive');
            }

            // Add stage history
            await db.execute(
                `INSERT INTO dealstagehistory 
                 (deal_id, stage_id, changed_by, created_at)
                 VALUES (?, ?, ?, CURRENT_TIMESTAMP)`,
                [dealId, stageId, userId]
            );

            //await db.execute('COMMIT');
            return true;

        } catch (error) {
            //await db.execute('ROLLBACK');
            throw error;
        }

    } catch (error) {
        throw new Error(`Failed to update deal stage: ${error.message}`);
    }
}

async function createFilterView(viewData) {
    try {
                // Insert product view
        const [viewResult] = await db.execute(
            `INSERT INTO dealviews 
            (name, share_with, accessibility, users) 
            VALUES (?, ?, ?, ?)`,
            [viewData.title, viewData.share_with, viewData.accessibility, viewData.selected_users.join(',') || null]
        );

        const viewId = viewResult.insertId;

        // Insert filters if any
        if (viewData.filter && Array.isArray(viewData.filter)) {
            for (const filter of viewData.filter) {
                if (filter.field_label && filter.field_label.trim() !== '') {
                    await db.execute(
                        `INSERT INTO dealfilters 
                        (view_id, name, field_id, contains, filter, active) 
                        VALUES (?, ?, ?, ?, ?, ?)`,
                        [
                            viewId,
                            filter.field_label,
                            filter.field_name,
                            filter.contains,
                            JSON.stringify(filter.value_id),
                            1
                        ]
                    );
                }
            }
        }

       

        // Get the created view
        const [views] = await db.execute(
            'SELECT * FROM dealviews WHERE id = ?',
            [viewId]
        );

        return views[0];

    } catch (error) {
        console.error('Error in dealviews:', error);
        throw new Error(`Failed to create account view: ${error.message}`);
    }
}

async function deleteFilterView(viewId) {
    try {
        //await db.execute('START TRANSACTION');

        // Delete related filters first
        await db.execute(
            'DELETE FROM dealfilters WHERE view_id = ?',
            [viewId]
        );

        // Delete the view
        const [result] = await db.execute(
            'DELETE FROM dealviews WHERE id = ?',
            [viewId]
        );

        //await db.execute('COMMIT');

        return result.affectedRows > 0;

    } catch (error) {
       // await db.execute('ROLLBACK');
        console.error('Error in dealviews:', error);
        throw new Error(`Failed to delete product view: ${error.message}`);
    }
}

async function getAllFiltersList() {
    try {
        console.log("=== Starting getAllFiltersList Model Function ===");
        console.log("Executing database query for contactviews");
        
        // Get the views with their filters
        const [views] = await db.execute(`
            SELECT cv.*, cf.id as filter_id, cf.name as filter_name, 
                   cf.field_id, cf.contains, cf.filter as filter_value
            FROM dealviews cv
            LEFT JOIN dealfilters cf ON cv.id = cf.view_id
            `
        );
        
        // Transform the data to match findAll format
        const transformedViews = views.reduce((acc, row) => {
            if (!acc[row.id]) {
                acc[row.id] = {
                    id: row.id,
                    name: row.name,
                    share_with: row.share_with,
                    accessibility: row.accessibility,
                    users: row.users ? row.users.split(',') : [],
                    owner_name: row.owner_name,
                    created_at: formatDateTime(row.created_at),
                    updated_at: formatDateTime(row.updated_at),
                    filters: []
                };
            }
            
            if (row.filter_id) {
                acc[row.id].filters.push({
                    id: row.filter_id,
                    name: row.filter_name,
                    field_id: row.field_id,
                    contains: row.contains,
                    filter: row.filter_value ? JSON.parse(row.filter_value) : [],
                });
            }
            
            return acc;
        }, {});

        const result = Object.values(transformedViews);
        
        console.log("Number of views found:", result.length);
        return result;
        
    } catch (error) {
        console.error("=== Error in getAllFiltersList Model Function ===");
        console.error("Error details:", error);
        throw new Error(`Failed to fetch contact filters: ${error.message}`);
    }
}


async function getFieldsForFilter(userId) {
    try {
        // Validate userId
        if (!userId) {
            throw new Error('User ID is required');
        }

        // Get all product fields
        const [fields] = await db.execute(
            'SELECT * FROM dealfields WHERE active = 1 ORDER BY id'
        );

        if (!fields || fields.length === 0) {
            throw new Error('No product fields found');
        }

        // Get users for lookup fields
        const [users] = await db.execute(
            'SELECT id, name FROM users WHERE active = 1'
        );

        // Get all field choices
        const [choices] = await db.execute(
            'SELECT * FROM dealchoices'
        );

        // Process each field
        const processedFields = fields.map(field => {
            let values = [];
            let contains_list = [];

            // Set contains_list based on field type
            switch (field.field_type.toLowerCase()) {
                case 'dropdown':
                    contains_list = CONTAINS_LIST.DROPDOWN;
                    // Add dropdown values from choices with id and custom_option
                    values = choices
                        .filter(choice => choice.productfield_id === field.id)
                        .map(choice => ({
                            id: choice.id,
                            value: choice.value,
                            value: choice.custom_option || 0
                        }));
                    break;

                case 'number':
                    contains_list =CONTAINS_LIST.NUMBERS;
                    break;

                case 'datepicker':
                    contains_list = CONTAINS_LIST.DATE;
                    break;

                case 'lookup':
                    contains_list = CONTAINS_LIST.LOOKUP;
                    // Add lookup values
                    values = [
                        { id: null, value: "unassigned" },
                        ...users.map(user => ({
                            id: user.id,
                            value: user.id === userId ? "Me" : user.name
                        }))
                    ];
                    break;
                    case "Text field":
                        contains_list = CONTAINS_LIST.DEFAULT;
                        break;           

                default:
                    contains_list = CONTAINS_LIST.DEFAULT;
            }

            return {
                id: field.id,
                custom_field: field.custom_field || 0,
                field_label: field.field_label,
                field_name: field.field_name,
                field_type: field.field_type,
                values: values,
                contains_list: contains_list
            };
        });

        return processedFields;

    } catch (error) {
        console.error('Error in getProductFields:', error);
        throw new Error(`Failed to fetch product fields: ${error.message}`);
    }
}



// const dayjs = require('dayjs');

async function getDealListWithFilters(filters, userId) {
    try {
        // Step 1: Get user role and access scope
        const [userRows] = await db.execute(
            `SELECT r.access_scope, r.id as role_id, u.territory_id 
             FROM users u 
             JOIN roles r ON u.roles_id = r.id 
             WHERE u.id = ?`,
            [userId]
        );

        if (!userRows.length) throw new Error('User role not found');
        const { access_scope, territory_id } = userRows[0];
        const { view } = JSON.parse(access_scope)?.deal || {};

        let baseQuery = `SELECT d.*, u.name as owner_name 
                         FROM deals d 
                         LEFT JOIN users u ON d.owner_id = u.id 
                         WHERE d.active = 1`;

        let params = [];

        // Step 2: Apply permission-based filters
        switch (view) {
            case 'owned':
                baseQuery += ' AND d.owner_id = ?';
                params.push(userId);
                break;

            case 'territory':
                if (!territory_id) return [];
                const territories = territory_id.split(',');
                const placeholders = territories.map(() => '?').join(',');
                baseQuery += ` AND d.territory_id IN (${placeholders})`;
                params.push(...territories);
                break;

            case 'global':
            case 'restricted':
            default:
                if (view === 'restricted') return [];
        }

        // Step 3: Apply filter conditions
        const fieldMapping = {
            owner: 'd.owner_id',
            created_at: 'd.created_at',
            updated_at: 'd.updated_at',
            deal_stage: 'd.deal_stage_id',
            pipeline: 'd.deal_pipeline_id'
        };

        if (filters?.length > 0) {
            for (const filter of filters) {
                const { contains, field_name, value_id } = filter;
                const mappedField = fieldMapping[field_name] || `d.${field_name}`;
                applyFilterToQuery({ baseQuery, contains, mappedField, value_id, params });
            }
        }

        baseQuery += ` ORDER BY d.created_at DESC LIMIT 1000`; // Prevent overload

        // Step 4: Execute main query
        const [deals] = await db.execute(baseQuery, params);

        // Step 5: Process custom fields for each deal
        const processedDeals = deals.map(deal => {
            const processedDeal = { ...deal };
            
            // Parse custom fields if they exist
            if (deal.custom_field) {
                try {
                    const customFields = JSON.parse(deal.custom_field);
                    // Add custom fields to the deal object
                    customFields.forEach(customField => {
                        const fieldName = Object.keys(customField)[0];
                        const fieldValue = customField[fieldName];
                        processedDeal[fieldName] = fieldValue;
                    });
                    // Remove the original custom_field property since we've parsed it
                    delete processedDeal.custom_field;
                } catch (error) {
                    console.error('Error parsing custom fields for deal:', deal.id, error);
                }
            }
            
            return processedDeal;
        });

        // Step 6: Fetch pipeline and stage names for each deal
        const processedDealsWithNames = await Promise.all(processedDeals.map(async (deal) => {
            const dealWithNames = { ...deal };
            
            // Fetch pipeline name if deal_pipeline_id exists
            if (deal.deal_pipeline_id) {
                try {
                    const [pipelines] = await db.execute(
                        'SELECT id, pipeline_name as name FROM dealpipelines WHERE active = 1 AND id = ?',
                        [deal.deal_pipeline_id]
                    );
                    if (pipelines.length > 0) {
                        dealWithNames.deal_pipeline_name = pipelines[0].name;
                    }
                } catch (error) {
                    console.error('Error fetching pipeline name for deal:', deal.id, error);
                }
            }
            
            // Fetch stage name if deal_stage_id exists
            console.log("deal.deal_stage_id", deal.deal_stage_id);
            if (deal.deal_stage_id) {
                try {
                    const [stages] = await db.execute(
                        'SELECT id, deal_stage as name FROM dealstages WHERE active = 1 AND id = ?',
                        [deal.deal_stage_id]
                    );
                    if (stages.length > 0) {
                        dealWithNames.deal_stage_name = stages[0].name;
                    }
                } catch (error) {
                    console.error('Error fetching stage name for deal:', deal.id, error);
                }
            }
            
            return dealWithNames;
        }));

        return processedDealsWithNames;


    } catch (error) {
        console.error('Error in getDealListWithFilters:', error);
        throw error;
    }
}

// Helper function to avoid repeating condition logic
function applyFilterToQuery({ baseQuery, contains, mappedField, value_id, params }) {
    switch (contains) {
        case 'contains':
            if (Array.isArray(value_id) && value_id.length) {
                const placeholders = value_id.map(() => '?').join(',');
                baseQuery += ` AND ${mappedField} IN (${placeholders})`;
                params.push(...value_id);
            }
            break;

        case 'does not contain':
            if (Array.isArray(value_id) && value_id.length) {
                const placeholders = value_id.map(() => '?').join(',');
                baseQuery += ` AND ${mappedField} NOT IN (${placeholders})`;
                params.push(...value_id);
            }
            break;

        case 'is not empty':
            baseQuery += ` AND ${mappedField} IS NOT NULL AND ${mappedField} != ''`;
            break;

        case 'is empty':
            baseQuery += ` AND (${mappedField} IS NULL OR ${mappedField} = '')`;
            break;

        case '< less than':
            baseQuery += ` AND ${mappedField} < ?`;
            params.push(value_id);
            break;

        case '> greater than':
            baseQuery += ` AND ${mappedField} > ?`;
            params.push(value_id);
            break;

        case '= equal to':
            baseQuery += ` AND ${mappedField} = ?`;
            params.push(value_id);
            break;

        case '!= not equal to':
            baseQuery += ` AND ${mappedField} != ?`;
            params.push(value_id);
            break;

        case 'between':
            if (Array.isArray(value_id) && value_id.length === 2) {
                baseQuery += ` AND ${mappedField} BETWEEN ? AND ?`;
                params.push(value_id[0], value_id[1]);
            }
            break;

        case 'last 30 minutes':
            baseQuery += ` AND ${mappedField} >= ?`;
            params.push(dayjs().subtract(30, 'minute').format('YYYY-MM-DD HH:mm:ss'));
            break;

        case 'last 60 minutes':
            baseQuery += ` AND ${mappedField} >= ?`;
            params.push(dayjs().subtract(60, 'minute').format('YYYY-MM-DD HH:mm:ss'));
            break;

        case 'last 2 hours':
            baseQuery += ` AND ${mappedField} >= ?`;
            params.push(dayjs().subtract(2, 'hour').format('YYYY-MM-DD HH:mm:ss'));
            break;

        case 'last 24 hours':
            baseQuery += ` AND ${mappedField} >= ?`;
            params.push(dayjs().subtract(24, 'hour').format('YYYY-MM-DD HH:mm:ss'));
            break;

        case 'last 7 days':
            baseQuery += ` AND ${mappedField} >= ?`;
            params.push(dayjs().subtract(7, 'day').format('YYYY-MM-DD HH:mm:ss'));
            break;

        case 'last 30 days':
            baseQuery += ` AND ${mappedField} >= ?`;
            params.push(dayjs().subtract(30, 'day').format('YYYY-MM-DD HH:mm:ss'));
            break;

        case 'last 90 days':
            baseQuery += ` AND ${mappedField} >= ?`;
            params.push(dayjs().subtract(90, 'day').format('YYYY-MM-DD HH:mm:ss'));
            break;

        case 'last 12 months':
            baseQuery += ` AND ${mappedField} >= ?`;
            params.push(dayjs().subtract(12, 'month').format('YYYY-MM-DD HH:mm:ss'));
            break;
    }
    return baseQuery;
}



async function updateFilterView(viewData) {
    console.log("updateFilterView inits "+JSON.stringify(viewData));
    try {
                // Insert product view
                const selectedUsers = Array.isArray(viewData.users) && viewData.users.length > 0
                ? viewData.users.join(',')
                : null;
              
              const [viewResult] = await db.execute(
                `UPDATE dealviews 
                 SET name = ?, share_with = ?, accessibility = ?, users = ?
                 WHERE id = ?`,
                [viewData.title, viewData.share_with, viewData.accessibility, selectedUsers, viewData.id]
              );

        const viewId = viewResult.insertId;

       console.log("viewResultsds "+viewResult);

       

        // Get the created view
        const [views] = await db.execute(
            'SELECT * FROM dealviews WHERE id = ?',
            [viewData.id]
        );

        return views[0];

    } catch (error) {
        console.error('Error in dealviews:', error);
        throw new Error(`Failed to create account view: ${error.message}`);
    }
}

// 📦 Deduplicate by field
function deduplicate(rows, field) {
    const seen = new Set();
    return rows.filter(row => {
        if (!row[field] || seen.has(row[field])) return false;
        seen.add(row[field]);
        return true;
    });
}

// 🔄 Field Processor
async function processFields(fields, choices, deal) {
    return await Promise.all(fields.map(async field => {
        const value = deal[field.field_name];
        const fieldObj = { ...field };

        fieldObj.choices = choices.filter(c => c.dealfield_id === field.id);
        fieldObj.field_value = value;

        // Custom fields
        if (field.custom_field === 1 && deal.custom_field) {
            const customFields = JSON.parse(deal.custom_field || '{}');
            // console.log({customFields});
            const fieldName = fieldObj.field_name;
            const match = customFields.find(c => Object.hasOwn(c, fieldName));
            fieldObj.field_value = match ? match[fieldName] : null
            // fieldObj.field_value = customFields.find(c => Object.keys(c)[0] === fieldObj.field_name) || null;
        }

        // Lookup handling
        if (field.field_type === 'lookup' && field.lookup_type && field.lookup_column) {
            try {
                const [lookupRows] = await db.execute(
                    `SELECT id, ${field.lookup_column} FROM ${field.lookup_type} WHERE active = 1`
                );
                fieldObj.choices = lookupRows.map(r => ({
                    id: r.id,
                    dealfield_id: field.id,
                    custom_option: r[field.lookup_column]
                }));
            } catch (err) {
                console.warn(`Failed lookup for ${field.field_name}:`, err);
                fieldObj.choices = [];
            }
        }

        return fieldObj;
    }));
}

// Bulk import deals
async function bulkImport(csvData, columnMappings, userId) {
    const connection = await db.getConnection();
    try {
        await connection.beginTransaction();
        console.log("bulkImport inits "+JSON.stringify(csvData));

        const results = {
            total: csvData.length,
            successful: 0,
            failed: 0,
            errors: [],
            errorCsvData: [],
            errorCsvFile: null
        };

        // Pre-fetch validation data
        const [users] = await connection.execute('SELECT id, record_id FROM users WHERE active = 1');
        const [accounts] = await connection.execute('SELECT id FROM accounts WHERE active = 1');
        const [contacts] = await connection.execute('SELECT id FROM contacts WHERE active = 1');
        
        // Create lookup maps
        const userMap = new Map();
        users.forEach(user => {
            if (user.record_id) {
                userMap.set(user.record_id.toString(), user.id);
            }
        });
        
        const accountIds = new Set(accounts.map(acc => acc.id.toString()));
        const contactIds = new Set(contacts.map(cont => cont.id.toString()));

        for (let i = 0; i < csvData.length; i++) {
            const row = csvData[i];
            const rowNumber = i + 2; // +2 because CSV has header and arrays are 0-indexed

            try {
                // Prepare deal data
                const dealData = {
                    owner_id: userId,
                    created_at: new Date().toISOString().slice(0, 19).replace('T', ' '),
                    updated_at: new Date().toISOString().slice(0, 19).replace('T', ' ')
                };

                // Separate standard fields and custom fields
                const standardFields = [];
                const customFields = [];

                // Process each mapped column
                for (const mapping of columnMappings) {
                    if (mapping.isMapped && mapping.apiField) {
                    const csvValue = row[mapping.csvColumn];
                        const field = mapping.apiField;

                        // Skip if no value provided
                        if (csvValue === undefined || csvValue === null || csvValue === '') {
                            continue;
                        }

                                                let processedValue = csvValue.toString().trim();

                        // Handle different field types based on field_name and field_type
                        switch (field.field_type.toLowerCase()) {
                            case 'text field':
                            case 'textarea':
                                processedValue = csvValue.toString().trim();
                                break;

                            case 'number':
                                const numValue = parseFloat(csvValue);
                                if (!isNaN(numValue)) {
                                    processedValue = numValue;
                                }
                                break;

                            case 'datepicker':
                                // Handle various date formats
                                if (csvValue) {
                                    const date = new Date(csvValue);
                                    if (!isNaN(date.getTime())) {
                                        processedValue = date.toISOString().split('T')[0];
                                    }
                                }
                                break;

                            case 'dropdown':
                            case 'multiselect':
                                // For dropdown/multiselect, query the choices table directly
                                if (field.field_name) {
                                    const [choices] = await connection.execute(
                                        'SELECT id, custom_option FROM dealchoices WHERE dealfield_id = ? AND custom_option = ?',
                                        [field.id, csvValue.toString().trim()]
                                    );
                                    
                                    if (choices.length > 0) {
                                        processedValue = choices[0].id;
                                    } else {
                                        throw new Error(`Invalid dropdown value: "${csvValue}" for field "${field.field_label}" - Choice not found`);
                                    }
                                }
                                break;

                            case 'lookup':
                                // Handle lookup fields based on field_name
                                if (field.field_name === 'owner_id' || field.field_name === 'created_by' || field.field_name === 'modified_by') {
                                    // For user lookup fields, get user ID from users table based on record_id
                                    const userRecordId = csvValue.toString().trim();
                                    const mappedUserId = userMap.get(userRecordId);
                                    if (!mappedUserId) {
                                        throw new Error(`Invalid ${field.field_name}: "${userRecordId}" - User not found`);
                                    }
                                    processedValue = mappedUserId;
                                } else if (field.field_name === 'sales_account_id' || field.field_name === 'account_id') {
                                    // For sales account lookup, check if the value looks like a record_id
                                    if (csvValue.toString().includes('zcrm_') || /^\d+$/.test(csvValue.toString())) {
                                        // This looks like a record_id, try to find by record_id first
                                        const [accountResult] = await connection.execute(
                                            'SELECT id FROM accounts WHERE record_id = ? AND active = 1',
                                            [csvValue.toString().trim()]
                                        );
                                        
                                        if (accountResult.length > 0) {
                                            processedValue = accountResult[0].id;
                                        } else {
                                            // If not found by record_id, try by name
                                            const [accountResult2] = await connection.execute(
                                                'SELECT id FROM accounts WHERE name = ? AND active = 1',
                                                [csvValue.toString().trim()]
                                            );
                                            
                                            if (accountResult2.length > 0) {
                                                processedValue = accountResult2[0].id;
                                            } else {
                                                throw new Error(`Invalid sales account: "${csvValue}" - Account not found`);
                                            }
                                        }
                                    } else {
                                        // This looks like a display name, query by name
                                        const [accountResult] = await connection.execute(
                                            'SELECT id FROM accounts WHERE name = ? AND active = 1',
                                            [csvValue.toString().trim()]
                                        );
                                        
                                        if (accountResult.length > 0) {
                                            processedValue = accountResult[0].id;
                                        } else {
                                            throw new Error(`Invalid sales account: "${csvValue}" - Account not found`);
                                        }
                                    }
                                } else if (field.field_name === 'contacts' || field.field_name === 'contact_id') {
                                    // For contact lookup, check if the value looks like a record_id
                                    if (csvValue.toString().includes('zcrm_') || /^\d+$/.test(csvValue.toString())) {
                                        // This looks like a record_id, try to find by record_id first
                                        const [contactResult] = await connection.execute(
                                            'SELECT id FROM contacts WHERE record_id = ? AND active = 1',
                                            [csvValue.toString().trim()]
                                        );
                                        
                                        if (contactResult.length > 0) {
                                            processedValue = contactResult[0].id;
                                        } else {
                                            // If not found by record_id, try by name
                                            const [contactResult2] = await connection.execute(
                                                'SELECT id FROM contacts WHERE name = ? AND active = 1',
                                                [csvValue.toString().trim()]
                                            );
                                            
                                            if (contactResult2.length > 0) {
                                                processedValue = contactResult2[0].id;
                                            } else {
                                                throw new Error(`Invalid contact: "${csvValue}" - Contact not found`);
                                            }
                                        }
                                    } else {
                                        // This looks like a display name, query by name
                                        const [contactResult] = await connection.execute(
                                            'SELECT id FROM contacts WHERE name = ? AND active = 1',
                                            [csvValue.toString().trim()]
                                        );
                                        
                                        if (contactResult.length > 0) {
                                            processedValue = contactResult[0].id;
                                        } else {
                                            throw new Error(`Invalid contact: "${csvValue}" - Contact not found`);
                                        }
                                    }
                                } else if (field.lookup_type && field.lookup_column) {
                                    // For other lookup fields, check if the value looks like a record_id
                                    if (csvValue.toString().includes('zcrm_') || /^\d+$/.test(csvValue.toString())) {
                                        // This looks like a record_id, try to find by record_id first
                                        const [lookupResult] = await connection.execute(
                                            `SELECT id FROM ${field.lookup_type} WHERE record_id = ? AND active = 1`,
                                            [csvValue.toString().trim()]
                                        );
                                        
                                        if (lookupResult.length > 0) {
                                            processedValue = lookupResult[0].id;
                                        } else {
                                            // If not found by record_id, try by the lookup_column
                                            const [lookupResult2] = await connection.execute(
                                                `SELECT id FROM ${field.lookup_type} WHERE ${field.lookup_column} = ? AND active = 1`,
                                                [csvValue.toString().trim()]
                                            );
                                            
                                            if (lookupResult2.length > 0) {
                                                processedValue = lookupResult2[0].id;
                                            } else {
                                                throw new Error(`Invalid ${field.field_label}: "${csvValue}" - Record not found in ${field.lookup_type}`);
                                            }
                                        }
                                    } else {
                                        // This looks like a display name, query by lookup_column
                                        const [lookupResult] = await connection.execute(
                                            `SELECT id FROM ${field.lookup_type} WHERE ${field.lookup_column} = ? AND active = 1`,
                                            [csvValue.toString().trim()]
                                        );
                                        
                                        if (lookupResult.length > 0) {
                                            processedValue = lookupResult[0].id;
                                        } else {
                                            throw new Error(`Invalid ${field.field_label}: "${csvValue}" - Record not found in ${field.lookup_type}`);
                                        }
                                    }
                                } else {
                                    throw new Error(`Lookup field "${field.field_label}" has no lookup configuration`);
                                }
                                break;

                            default:
                                processedValue = csvValue.toString().trim();
                        }

                        // Separate standard and custom fields
                        if (field.custom_field === 1) {
                            customFields.push({ [field.field_name]: processedValue });
                        } else {
                            dealData[field.field_name] = processedValue;
                        }
                    }
                }

                // Add custom fields as JSON string - format as array of objects
                if (customFields.length > 0) {
                    dealData.custom_field = JSON.stringify(customFields);
                }

                // Add active status
                dealData.active = 1;

                // Insert the deal
                const keys = Object.keys(dealData);
                const values = Object.values(dealData);
                const placeholders = keys.map(() => '?').join(', ');

                const [result] = await connection.execute(
                    `INSERT INTO deals (${keys.join(', ')}) VALUES (${placeholders})`,
                    values
                );

                results.successful++;

            } catch (error) {
                results.failed++;
                const errorRecord = {
                    row: rowNumber,
                    error: error.message,
                    data: row
                };
                results.errors.push(errorRecord);
                
                // Add error record to CSV data
                const errorCsvRow = {
                    ...row,
                    'Error Row': rowNumber,
                    'Error Message': error.message
                };
                results.errorCsvData.push(errorCsvRow);
            }
        }

        await connection.commit();

        // Generate error CSV file if there are errors
        if (results.errorCsvData.length > 0) {
            const fs = require('fs');
            const path = require('path');
            
            // Create filename with timestamp
            const timestamp = new Date().toISOString().replace(/[:.]/g, '-').slice(0, 19);
            const filename = `deals_bulk_import_errors_${timestamp}.csv`;
            const filepath = path.join(__dirname, '../public/uploads', filename);
            
            // Ensure uploads directory exists
            const uploadsDir = path.dirname(filepath);
            if (!fs.existsSync(uploadsDir)) {
                fs.mkdirSync(uploadsDir, { recursive: true });
            }
            
            // Get CSV headers from the original CSV data (first row) plus error columns
            const originalHeaders = Object.keys(csvData[0] || {});
            const errorHeaders = ['Error Row', 'Error Message'];
            const headers = [...originalHeaders, ...errorHeaders];
            
            // Create CSV content
            let csvContent = headers.join(',') + '\n';
            
            results.errorCsvData.forEach(row => {
                const values = headers.map(header => {
                    let value = '';
                    if (header === 'Error Row') {
                        value = row['Error Row'] || '';
                    } else if (header === 'Error Message') {
                        value = row['Error Message'] || '';
                    } else {
                        value = row[header] || '';
                    }
                    // Escape quotes and wrap in quotes if contains comma or quote
                    const escapedValue = value.toString().replace(/"/g, '""');
                    return `"${escapedValue}"`;
                });
                csvContent += values.join(',') + '\n';
            });
            
            // Write CSV file
            fs.writeFileSync(filepath, csvContent);
            
            // Set the file path for response
            results.errorCsvFile = `/uploads/${filename}`;
        }

        return results;

    } catch (error) {
        await connection.rollback();
        throw error;
    } finally {
        connection.release();
    }
}


module.exports = {
    create,
    findAll,
    findById,
    update,
    deleteRecord,
    findDealMeetings,
    findByDeals,
    addProducts,
    getdealstages,
    createdealstages,
    updatedealstages,
    getdealPipelines,
    getDealRelations,
    getDefaultPipelineAndStage,
    updateDealStage,
    // deletedealstages
    getAllFiltersList,
    createFilterView,
    deleteFilterView,
    getFieldsForFilter,
    getDealStageslist,
    getDealListWithFilters,
    updateFilterView,
    bulkImport
};