const db = require('../config/db');

async function create(fieldData) {
    const {
        field_label,
        field_name,
        field_type,
        required,
        quick_add,
        tool_tip,
        placeholder,
        choices,
        read_only,
        has_dependent,
        dealgroups_id,
        active,
        order,
        lookup_type,
        lookup_column,
        custom_field
    } = fieldData;

    // Start transaction
    const connection = await db.getConnection();
    try {
        await connection.beginTransaction();

        // Insert into dealfields table (without choices)
        const [result] = await connection.execute(
            `INSERT INTO dealfields (
                field_label, field_name, field_type, required, quick_add,
                tool_tip, placeholder, read_only, has_dependent,
                dealgroups_id, active, \`order\`, lookup_type, lookup_column,
                custom_field, created_at, updated_at
            ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW(), NOW())`,
            [
                field_label, field_name, field_type, required, quick_add,
                tool_tip, placeholder, read_only, has_dependent,
                dealgroups_id, active, order, lookup_type, lookup_column,
                custom_field
            ]
        );

        const dealFieldId = result.insertId;

        // If choices are provided, insert them into dealchoices table
        if (choices && Array.isArray(choices) && choices.length > 0) {
            for (const choice of choices) {
                await connection.execute(
                    `INSERT INTO dealchoices (
                        dealfield_id, custom_option, created_at, updated_at
                    ) VALUES (?, ?, NOW(), NOW())`,
                    [dealFieldId, choice]
                );
            }
        }

        await connection.commit();
        return dealFieldId;
    } catch (error) {
        await connection.rollback();
        throw error;
    } finally {
        connection.release();
    }
}

// Function to find all deal fields
async function findAll(userId) {
    const connection = await db.getConnection();
    try {
        // Get user's role and access scope
        const [userRole] = await connection.execute(
            `SELECT r.access_scope, r.id as role_id, u.territory_id 
             FROM users u 
             JOIN roles r ON u.roles_id = r.id 
             WHERE u.id = ?`,
            [userId]
        );

        if (!userRole || !userRole.length) {
            throw new Error('User role not found');
        }

        const accessScope = JSON.parse(userRole[0].access_scope);
        const territories = userRole[0].territory_id ? userRole[0].territory_id.split(',') : [];

        let [rows] = await db.execute('SELECT * FROM dealfields WHERE active=1 ORDER BY `order` ASC');
        const [choices] = await db.execute('SELECT * FROM dealchoices');
        const [pipeline_id] = await db.execute('SELECT id, pipeline_name FROM dealpipelines');
        const [stage_id] = await db.execute('SELECT id, deal_stage FROM dealstages');

        // Process each field
        const processedRows = await Promise.all(rows.map(async field => {
            // Handle regular choices
            field.choices = choices.filter(choice => choice.dealfield_id === field.id);
            field.field_value = null;

            // Handle lookup type fields
            if (field.field_type === 'lookup' && field.lookup_type && field.lookup_column) {
                try {
                    let query = `SELECT id, ${field.lookup_column} FROM ${field.lookup_type} WHERE active = 1`;
                    const params = [];

                    // Get view permission for the specific module
                    const modulePermission = field.lookup_type === 'contacts' ? accessScope?.contact?.view :
                                          field.lookup_type === 'accounts' ? accessScope?.sales_account?.view :
                                          field.lookup_type === 'deals' ? accessScope?.deal?.view : 'global';

                    switch (modulePermission) {
                        case 'owned':
                            query += ' AND owner_id = ?';
                            params.push(userId);
                            break;

                        case 'territory':
                            if (territories.length) {
                                const placeholders = territories.map(() => '?').join(',');
                                query += ` AND territory_id IN (${placeholders})`;
                                params.push(...territories);
                            } else {
                                field.choices = [];
                                return field;
                            }
                            break;

                        case 'restricted':
                            field.choices = [];
                            return field;

                        case 'global':
                        default:
                            // No additional filters for global access
                            break;
                    }

                    // Execute filtered query
                    const [lookupData] = await connection.execute(query, params);

                    // Format lookup data as choices
                    field.choices = lookupData.map(item => ({
                        id: item.id,
                        dealfield_id: field.id,
                        custom_option: item[field.lookup_column]
                    }));
                } catch (error) {
                    console.error(`Error fetching lookup data for field ${field.field_name}:`, error);
                    field.choices = [];
                }
            }

            if (field.field_name.toLowerCase() === 'deal_pipeline_id') {
                try {

                    // Format lookup data as choices
                    field.choices = pipeline_id.map(item => ({
                        id: item.id,
                        custom_option: item['pipeline_name']
                    }));
                } catch (error) {
                    console.error(`Error fetching lookup data for field ${field.field_name}:`, error);
                    field.choices = [];
                }
            }

            if (field.field_name.toLowerCase() === 'deal_stage_id') {
                try {

                    // Format lookup data as choices
                    field.choices = stage_id.map(item => ({
                        id: item.id,
                        custom_option: item['deal_stage']
                    }));
                } catch (error) {
                    console.error(`Error fetching lookup data for field ${field.field_name}:`, error);
                    field.choices = [];
                }
            }

            return field;
        }));

        return processedRows;
    } catch (error) {
        console.error('Error in findAll:', error);
        throw error;
    } finally {
        connection.release();
    }
}
async function findById(id) {
    const [rows] = await db.execute(`
        SELECT df.* 
        FROM dealfields df
        LEFT JOIN dealgroups dg ON df.dealgroups_id = dg.id
        WHERE df.id = ?
    `, [id]);
    return rows[0];
}

async function update(id, fieldData) {
    const {
        field_label,
        field_name,
        field_type,
        required,
        quick_add,
        tool_tip,
        placeholder,
        choices,
        read_only,
        has_dependent,
        dealgroups_id,
        active,
        order,
        lookup_type,
        lookup_column,
        custom_field
    } = fieldData;

    // Start transaction
    const connection = await db.getConnection();
    try {
        await connection.beginTransaction();

        // Update dealfields table (without choices)
        const [result] = await connection.execute(
            `UPDATE dealfields SET 
                field_label = ?, 
                field_name = ?, 
                field_type = ?, 
                required = ?,
                quick_add = ?, 
                tool_tip = ?, 
                placeholder = ?,
                read_only = ?, 
                has_dependent = ?, 
                dealgroups_id = ?, 
                active = ?,
                \`order\` = ?, 
                lookup_type = ?, 
                lookup_column = ?, 
                custom_field = ?,
                updated_at = NOW()
            WHERE id = ?`,
            [
                field_label, 
                field_name, 
                field_type, 
                required,
                quick_add, 
                tool_tip, 
                placeholder,
                read_only, 
                has_dependent, 
                dealgroups_id, 
                active,
                order, 
                lookup_type, 
                lookup_column, 
                custom_field,
                id
            ]
        );

        // Handle choices if field type is select or multiselect
        if (field_type === 'select' || field_type === 'multiselect') {
            // First, delete existing choices
            await connection.execute(
                'DELETE FROM dealchoices WHERE dealfield_id = ?',
                [id]
            );

            // Then insert new choices if provided
            if (choices && Array.isArray(choices) && choices.length > 0) {
                for (const choice of choices) {
                    await connection.execute(
                        `INSERT INTO dealchoices (
                            dealfield_id, custom_option, created_at, updated_at
                        ) VALUES (?, ?, NOW(), NOW())`,
                        [id, choice]
                    );
                }
            }
        }

        await connection.commit();
        return result.affectedRows > 0;
    } catch (error) {
        await connection.rollback();
        throw error;
    } finally {
        connection.release();
    }
}

async function remove(id) {
    const [result] = await db.execute(
        'UPDATE dealfields SET active = 0 WHERE id = ?',
        [id]
    );
    return result.affectedRows > 0;
}

// update the order
async function order(fields) {

    try {
        for (const field of fields) {
            await db.execute(
                `UPDATE dealfields 
                 SET \`order\` = ?, 
                     updated_at = NOW() 
                 WHERE id = ?`,
                [field.order, field.id]
            );
        }
        
        return {
            status: 200,
            success: true,
            message: 'Field order updated successfully'
        };
} catch (error) {
    console.error('Error updating field order:', error);
    throw new Error(`Failed to update field order: ${error.message}`);
}

}


module.exports = {
    create,
    findAll,
    findById,
    update,
    remove,
    order
};