const db = require('../config/db');
const fs = require('fs');
const path = require('path');

class File {
    static async upload(data) {
        try {
            // Validate required fields
            if (!data.belongsto || !data.belongsto_id) {
                throw new Error('belongsto and belongsto_id are required');
            }

            // Create upload directory if not exists
            const uploadDir = path.join(process.cwd(), 'public', 'uploads', data.belongsto);
            if (!fs.existsSync(uploadDir)) {
                fs.mkdirSync(uploadDir, { recursive: true });
            }

            // Prepare data with null fallbacks
            const fileData = {
                file_name: data.originalname || null,
                file_path: data.filename ? `/uploads/${data.belongsto}/${data.filename}` : null,
                file_type: data.mimetype || null,
                file_size: data.size || 0,
                belongsto: data.belongsto,
                belongsto_id: data.belongsto_id,
                created_by: data.userId || null
            };

            const [result] = await db.execute(
                `INSERT INTO fileimages 
                 (file_name, file_path, file_type, file_size, belongsto, belongsto_id, created_by)
                 VALUES (?, ?, ?, ?, ?, ?, ?)`,
                [
                    fileData.file_name,
                    fileData.file_path,
                    fileData.file_type,
                    fileData.file_size,
                    fileData.belongsto,
                    fileData.belongsto_id,
                    fileData.created_by
                ]
            );

            return {
                id: result.insertId,
                file_name: fileData.file_name,
                file_path: fileData.file_path
            };
        } catch (error) {
            throw new Error(`Failed to upload file: ${error.message}`);
        }
    }

    static async getFiles(belongsto, belongstoId) {
        try {
            const [files] = await db.execute(
                `SELECT f.*, u.name as uploaded_by 
                 FROM fileuploads f
                 LEFT JOIN users u ON f.owner = u.id
                 WHERE f.type = ? AND f.type_id = ?
                 ORDER BY f.created_at DESC`,
                [belongsto, belongstoId]
            );

            const baseUrl = process.env.BASE_URL || 'http://localhost:5000';

            return files.map(file => ({
                id: file.id,
                file_name: file.file_name,
                file_type: file.file_type,
                file_size: file.file_size,
                file_url: `${baseUrl}${file.file_path}`,
                uploaded_by: file.uploaded_by,
                created_at: file.created_at
            }));
        } catch (error) {
            throw new Error(`Failed to get files: ${error.message}`);
        }
    }

    static async delete(fileId) {
        try {
            const [files] = await db.execute(
                'SELECT * FROM fileimages WHERE id = ?',
                [fileId]
            );

            if (!files.length) {
                throw new Error('File not found');
            }

            const filePath = path.join(process.cwd(), 'public', files[0].file_path);
            if (fs.existsSync(filePath)) {
                fs.unlinkSync(filePath);
            }

            await db.execute('DELETE FROM fileimages WHERE id = ?', [fileId]);

            return true;
        } catch (error) {
            throw new Error(`Failed to delete file: ${error.message}`);
        }
    }
}

module.exports = File;