const db = require('../config/db');

async function getPriceSettings() {
    const [settings] = await db.execute(
        `SELECT * FROM pricesettings ORDER BY id LIMIT 1`
    );
    return settings[0];
}

async function getQuoteCycles() {
    const [cycles] = await db.execute(
        `SELECT * FROM quotecycles `
    );
    return cycles;
}

async function updatePriceSettings(data) {
    const connection = await db.getConnection();
    
    try {
        await connection.beginTransaction();
        
        const {
            pricing_type,
            tax_type,
            sales_tax,
            discount_type,
            discount,
            quotes  // New field to receive quote types
        } = data;

        // Update price settings
        const [result] = await connection.execute(
            `UPDATE pricesettings 
             SET 
                tax_type = ?,
                sales_tax = ?,
                discount_type = ?,
                discount = ?
             WHERE id = (SELECT id FROM (SELECT id FROM pricesettings LIMIT 1) as temp)`,
            [ tax_type, sales_tax, discount_type, discount]
        );

        // If quotes are provided, update quote cycles
        if (Array.isArray(quotes) && quotes.length > 0) {
            // Delete existing quote cycles
            await connection.execute('DELETE FROM quotecycles');

            // Insert new quote cycles
            for (const quoteType of quotes) {
                if (quoteType && quoteType.trim()) {
                    await connection.execute(
                        'INSERT INTO quotecycles (quote_type) VALUES (?)',
                        [quoteType.trim()]
                    );
                }
            }
        }

        await connection.commit();
        return true;
    } catch (error) {
        await connection.rollback();
        throw error;
    } finally {
        connection.release();
    }
}

async function updateQuoteCycles(quoteTypes) {
    // Start transaction
    const connection = await db.getConnection();
    try {
        await connection.beginTransaction();

        // Delete existing quote cycles
        await connection.execute('DELETE FROM quotecycles');

        // Insert new quote cycles
        for (const quoteType of quoteTypes) {
            if (quoteType && quoteType.trim()) {
                await connection.execute(
                    'INSERT INTO quotecycles (quote_type, active) VALUES (?, 1)',
                    [quoteType.trim()]
                );
            }
        }

        await connection.commit();
        return true;
    } catch (error) {
        await connection.rollback();
        throw error;
    } finally {
        connection.release();
    }
}

module.exports = {
    getPriceSettings,
    getQuoteCycles,
    updatePriceSettings,
    updateQuoteCycles
}; 