const db = require('../config/db');

async function create(fieldData) {
    try {
        // First, insert the product field
        const [fieldResult] = await db.execute(
            `INSERT INTO productfields (
                field_label, 
                field_name, 
                field_type, 
                required, 
                quick_add,
                tool_tip, 
                placeholder, 
                choices, 
                read_only, 
                has_dependent,
                active, 
                lookup_type, 
                lookup_column,
                custom_field, 
                created_at, 
                updated_at
            ) 
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW(), NOW())`,
            [
                fieldData.field_label,
                fieldData.field_name,
                fieldData.field_type,
                fieldData.required || 0,
                fieldData.quick_add || 0,
                fieldData.tool_tip || null,
                fieldData.placeholder || null,
                JSON.stringify(fieldData.choices || []),
                fieldData.read_only || 0,
                fieldData.has_dependent || 0,
                fieldData.active || 1,
                fieldData.lookup_type || null,
                fieldData.lookup_column || null,
                fieldData.custom_field || 0
            ]
        );

        const productFieldId = fieldResult.insertId;

        // If there are choices, insert them individually
        if (fieldData.choices && Array.isArray(fieldData.choices) && fieldData.choices.length > 0) {
            for (const choice of fieldData.choices) {
                await db.execute(
                    `INSERT INTO productfieldchoices (
                        productfield_id,
                        custom_option,
                        created_at,
                        updated_at
                    ) VALUES (?, ?, NOW(), NOW())`,
                    [
                        productFieldId,
                        choice
                    ]
                );
            }
        }

        return productFieldId;
    } catch (error) {
        console.error('Error creating product field:', error);
        throw error;
    }
}
async function findAll() {
    let [rows] = await db.execute('SELECT * FROM productfields WHERE active=1 ORDER BY id');
    const [choices] = await db.execute('SELECT * FROM productfieldchoices');

    // Process each field
    const processedRows = await Promise.all(rows.map(async field => {
        // Handle regular choices
        field.choices = choices.filter(choice => choice.productfield_id === field.id);
        field.field_value = null;

        // Handle lookup type fields
        if (field.field_type === 'lookup' && field.lookup_type && field.lookup_column) {
            try {
                // Get data from lookup table
                const [lookupData] = await db.execute(
                    `SELECT id, ${field.lookup_column} FROM ${field.lookup_type} WHERE active = 1`
                );

                // Format lookup data as choices
                field.choices = lookupData.map(item => ({
                    id: item.id,
                    dealfield_id: field.id,
                    custom_option: item[field.lookup_column]
                }));
            } catch (error) {
                console.error(`Error fetching lookup data for field ${field.field_name}:`, error);
                field.choices = [];
            }
        }

        return field;
    }));

    return processedRows;
}

// findall by group
async function findAllWithGroupName() {
    // Get all the required data
    let [fields] = await db.execute('SELECT * FROM productfields ORDER BY productgroups_id');
    const [choices] = await db.execute('SELECT * FROM productfieldchoices');
    const [groups] = await db.execute('SELECT * FROM productgroups WHERE active = 1 ORDER BY id');

    // Process each field
    const processedFields = await Promise.all(fields.map(async field => {
        // Handle regular choices
        field.choices = choices.filter(choice => choice.productfield_id === field.id);
        field.field_value = null;

        // Handle lookup type fields
        if (field.field_type === 'lookup' && field.lookup_type && field.lookup_column) {
            try {
                // Get data from lookup table
                const [lookupData] = await db.execute(
                    `SELECT id, ${field.lookup_column} FROM ${field.lookup_type} WHERE active = 1`
                );

                // Format lookup data as choices
                field.choices = lookupData.map(item => ({
                    id: item.id,
                    dealfield_id: field.id,
                    custom_option: item[field.lookup_column]
                }));
            } catch (error) {
                console.error(`Error fetching lookup data for field ${field.field_name}:`, error);
                field.choices = [];
            }
        }

        return field;
    }));

    // Group fields by their product groups
    const groupedFields = groups.map(group => {
        return {
            group_id: group.id,
            group_name: group.group_name,
            group_order: group.group_order,
            group_description: group.description || null,
            is_active: group.active,
            fields: processedFields.filter(field => field.productgroups_id === group.id)
        };
    });

    // Sort groups by group_order
    groupedFields.sort((a, b) => a.group_order - b.group_order);

    // Add ungrouped fields if any exist
    const ungroupedFields = processedFields.filter(field => !field.productgroups_id);
    if (ungroupedFields.length > 0) {
        groupedFields.push({
            group_id: null,
            group_name: "Ungrouped Fields",
            group_order: Number.MAX_SAFE_INTEGER,
            group_description: null,
            is_active: 1,
            fields: ungroupedFields
        });
    }

    return {
        total_groups: groupedFields.length,
        total_fields: processedFields.length,
        groups: groupedFields
    };
}


// findall by id
async function findById(id) {
    const [rows] = await db.execute(`
        SELECT df.*, dg.name as group_name 
        FROM productfields df
        LEFT JOIN productgroups dg ON df.productgroups_id = dg.id
        WHERE df.id = ?
    `, [id]);
    return rows[0];
}

async function update(id, fieldData) {
    const connection = await db.getConnection();

    try {
        await connection.beginTransaction();

        // Update the product field
        const [result] = await connection.execute(
            `UPDATE productfields SET 
                field_label = ?, 
                field_name = ?, 
                field_type = ?, 
                required = ?,
                quick_add = ?, 
                tool_tip = ?, 
                placeholder = ?, 
                read_only = ?, 
                has_dependent = ?, 
                productgroups_id = ?, 
                active = ?, 
                lookup_type = ?, 
                lookup_column = ?, 
                custom_field = ?,
                updated_at = NOW()
            WHERE id = ?`,
            [
                fieldData.field_label,
                fieldData.field_name,
                fieldData.field_type,
                fieldData.required || 0,
                fieldData.quick_add || 0,
                fieldData.tool_tip || null,
                fieldData.placeholder || null,
                fieldData.read_only || 0,
                fieldData.has_dependent || 0,
                fieldData.productgroups_id || null,
                fieldData.active || 1,
                fieldData.lookup_type || null,
                fieldData.lookup_column || null,
                fieldData.custom_field || 0,
                id
            ]
        );

        // If there are choices, update them
        if (fieldData.choices && Array.isArray(fieldData.choices)) {
            // First delete existing choices
            await connection.execute(
                'DELETE FROM productfieldchoices WHERE productfield_id = ?',
                [id]
            );

            // Then insert new choices
            for (const choice of fieldData.choices) {
                await connection.execute(
                    `INSERT INTO productfieldchoices (
                        productfield_id,
                        custom_option,
                        created_at,
                        updated_at
                    ) VALUES (?, ?, NOW(), NOW())`,
                    [
                        id,
                        choice
                    ]
                );
            }
        }

        await connection.commit();

        // Get updated field with choices
        const [updatedField] = await connection.execute(
            'SELECT * FROM productfields WHERE id = ?',
            [id]
        );

        const [updatedChoices] = await connection.execute(
            'SELECT * FROM productfieldchoices WHERE productfield_id = ?',
            [id]
        );

        return {
            ...updatedField[0],
            choices: updatedChoices,
            affectedRows: result.affectedRows
        };

    } catch (error) {
        await connection.rollback();
        throw error;
    } finally {
        connection.release();
    }
}

async function remove(id) {
    const [result] = await db.execute(
        'UPDATE productfields SET active = 0 WHERE id = ?',
        [id]
    );
    return result.affectedRows > 0;
}

async function addProductGroup(groupData) {
    const [result] = await db.execute(
        'INSERT INTO productgroups (group_name, active, created_at, updated_at) VALUES (?, ?, NOW(), NOW())',
        [groupData.group_name, groupData.active || 1]
    );
    return result.insertId;
}
// update product group
async function updateProductGroup(id, groupData) {
    const [result] = await db.execute(
        'UPDATE productgroups SET group_name = ?, active = ?, updated_at = NOW() WHERE id = ?',
        [groupData.group_name, groupData.active || 1, id]
    );
    return result.affectedRows > 0;
}

// delete product group
async function deleteProductGroup(id) {
    const [result] = await db.execute('DELETE FROM productgroups WHERE id = ?', [id]);
    return result.affectedRows > 0;
}

// get product field by group name
async function getGroupName() {
    const [result] = await db.execute('SELECT id, group_name FROM productgroups');
    // console.log("result",result);
    return result;
}

module.exports = {
    create,
    findAll,
    findById,
    update,
    remove,
    findAllWithGroupName,
    addProductGroup,
    updateProductGroup,
    deleteProductGroup,
    getGroupName
};