const db = require('../config/db');
const moment = require('moment');
const fs = require('fs');
const path = require('path');
const crypto = require('crypto');

async function create(Data) {
    try {
        const { product: allFields } = Data;
        
        // Filter out empty/null values first
        const validFields = Array.from(allFields).filter(field => 
            field.field_value !== null && 
            field.field_value !== undefined && 
            field.field_value !== ''
        );

        // Separate custom and standard fields
        const custom_fields = validFields
            .filter(field => field.custom_field == 1)
            .map(field => ({ [field.field_name]: field.field_value }));
            
        const standardFields = validFields
            .filter(field => field.custom_field == 0);

        // Only proceed if we have fields to insert
        if (standardFields.length === 0 && custom_fields.length === 0) {
            throw new Error('No valid fields provided');
        }

        // Prepare values for insertion
        const product = [
            ...standardFields.map(field => field.field_value)
        ];

        // Only add custom_fields if there are any
        if (custom_fields.length > 0) {
            product.push(JSON.stringify(custom_fields));
        }

        // Create column names string
        const keys = standardFields
            .map(field => field.field_name);
            
        // Add custom_field column if we have custom fields
        if (custom_fields.length > 0) {
            keys.push('custom_field');
        }

        const keyString = keys.join(', ');
        
        // Create placeholders for values
        const placeholders = '?' + ', ?'.repeat(product.length - 1);

        // Execute insert query
        const [result] = await db.execute(
            `INSERT INTO products (${keyString}) VALUES (${placeholders})`,
            product
        );
        
        // Return the ID of newly created product
        return result.insertId;

    } catch (error) {
        console.error('Error in product creation:', error);
        throw new Error(`Failed to create product: ${error.message}`);
    }
}
async function findById(id) {
    try {
        // Get product data with formatted dates
        const [rows] = await db.execute(
            'SELECT p.*, DATE_FORMAT(p.create_at, "%Y-%m-%d") as formatted_create_at, DATE_FORMAT(p.update_at, "%Y-%m-%d") as formatted_update_at FROM products p WHERE p.id = ? AND p.active = 1',
            [id]
        );
        
        if (!rows.length) return null;

        // Get all fields and choices
        const [fields] = await db.execute('SELECT * FROM productfields ORDER BY id');
        const [choices] = await db.execute('SELECT * FROM productfieldchoices');

        // Get available currencies
        const [currencies] = await db.execute(
            'SELECT * FROM currencies WHERE active = 1'
        );

        // Get product's currency values
        const [productCurrencies] = await db.execute(
            'SELECT * FROM productcurrencies WHERE product_id = ?',
            [id]
        );

        // Get price settings
        const [priceSettings] = await db.execute(
            'SELECT pricing_type FROM pricesettings LIMIT 1'
        );

        // Define pricing types based on settings
        let pricingtypes = [];
        if (!priceSettings.length || priceSettings[0].pricing_type === 'Both') {
            pricingtypes = [
                { pricing_type: "One-time Pricing" },
                { pricing_type: "Subscription Pricing" }
            ];
        } else {
            pricingtypes = [
                { pricing_type: priceSettings[0].pricing_type }
            ];
        }

        // Get files from fileimages table
        const [files] = await db.execute(
            `SELECT * FROM fileimages 
             WHERE belongsto = 'products' 
             AND belongsto_id = ?`,
            [id]
        );

        // Format product data
        const product = {
            id: parseInt(rows[0].id),
            category: rows[0].category,
            subcategory: rows[0].subcategory,
            product_code: rows[0].product_code,
            product_name: rows[0].product_name,
            display_name: rows[0].display_name,
            product_image: rows[0].product_image,
            item_description: rows[0].item_description,
            sku_number: rows[0].sku_number,
            owner: rows[0].owner?.toString(),
            valid_till: rows[0].valid_till,
            parent_product: rows[0].parent_product,
            gross_amount: rows[0].gross_amount,
            net_amount: rows[0].net_amount,
            created_by: rows[0].created_by?.toString(),
            updated_by: rows[0].updated_by?.toString(),
            create_at: rows[0].formatted_create_at,
            update_at: rows[0].formatted_update_at,
            base_currency_amount: rows[0].base_currency_amount,
            currency: rows[0].currency,
            pricing_type: rows[0].pricing_type,
            setup_fee: rows[0].setup_fee,
            subscription_cycle: rows[0].subscription_cycle,
            no_of_billing_cycle: rows[0].no_of_billing_cycle,
            available: rows[0].available,
            stockquantity: rows[0].stockquantity,
            tax: rows[0].tax,
            active: rows[0].active,
            custom_field: rows[0].custom_field,
            created_at: rows[0].created_at,
            updated_at: rows[0].updated_at,
            currencies: productCurrencies.map(pc => ({
                base_currency_amount: pc.base_currency_amount,
                currency: pc.currency
            })),
            files: files.map(file => ({
                id: file.id,
                file_name: file.file_name,
                file_path: file.file_path,
                file_type: file.file_type,
                file_size: file.file_size,
                created_at: file.created_at
            }))
        };

        // Process fields with their values and choices
        const products = fields.map(field => {
            field.choices = choices.filter(choice => choice.productfield_id === field.id);
            field.field_value = field.custom_field === 1 && rows[0].custom_field 
                ? JSON.parse(rows[0].custom_field).find(cf => cf[field.field_name])?.[field.field_name] || null
                : rows[0][field.field_name];
            return field;
        });

        return {
            id: id.toString(),
            product: product,
            products: products,
            currencies: currencies,
            pricingtypes: pricingtypes,
            product_image: rows[0].product_image || "",
            files: files.map(file => ({
                id: file.id,
                file_name: file.filename,
                file_path: file.file_url,
                file_size: file.file_size,
                created_at: file.created_at
            }))
        };

    } catch (error) {
        console.error('Error in findById:', error);
        throw new Error(`Failed to fetch product: ${error.message}`);
    }
}
async function findAll() {
    try {
        // Get all products
        const [rows] = await db.execute(
            'SELECT * FROM products WHERE active = 1 ORDER BY created_at DESC'
        );

        // Get all fields and choices
        const [fields] = await db.execute('SELECT * FROM productfields ORDER BY id');
        const [choices] = await db.execute('SELECT * FROM productfieldchoices');

        // Process each product
        return rows.map(product => {
            const processedFields = fields.map(field => {
                // Add choices if any
                field.choices = choices.filter(choice => choice.productfield_id === field.id);
                
                // Set standard field value
                field.field_value = product[field.field_name];

                // Handle custom fields
                if (field.custom_field === 1 && product.custom_field) {
                    const customFields = JSON.parse(product.custom_field);
                    field.field_value = customFields.find(cf => cf[field.field_name])?.[field.field_name] || null;
                }

                return field;
            });

            return {
                id: product.id.toString(),
                products: processedFields
            };
        });

    } catch (error) {
        console.error('Error in findAll:', error);
        throw new Error(`Failed to fetch products: ${error.message}`);
    }
}

async function update(id, Data) {
    try {
        const { product: allFields } = Data;
        
        // Filter out empty/null values first
        const validFields = Array.from(allFields).filter(field => 
            field.field_value !== null && 
            field.field_value !== undefined && 
            field.field_value !== ''
        );

        // Separate custom and standard fields
        const custom_fields = validFields
            .filter(field => field.custom_field == 1)
            .map(field => ({ [field.field_name]: field.field_value }));
            
        const standardFields = validFields
            .filter(field => field.custom_field == 0);

        // Only proceed if we have fields to update
        if (standardFields.length === 0 && custom_fields.length === 0) {
            throw new Error('No valid fields provided for update');
        }

        // Create update fields string
        const updateFields = standardFields
            .map(field => `${field.field_name} = ?`);

        // Add custom_field if we have custom fields
        if (custom_fields.length > 0) {
            updateFields.push('custom_field = ?');
        }

        const updateString = updateFields.join(', ');

        // Prepare values for update
        const updateValues = [
            ...standardFields.map(field => field.field_value)
        ];

        // Add custom fields value if any
        if (custom_fields.length > 0) {
            updateValues.push(JSON.stringify(custom_fields));
        }

        // Add ID for WHERE clause
        updateValues.push(id);

        // Execute update query
        const [result] = await db.execute(
            `UPDATE products SET ${updateString} WHERE id = ?`,
            updateValues
        );

        return result.affectedRows > 0;

    } catch (error) {
        console.error('Error in update:', error);
        throw new Error(`Failed to update product: ${error.message}`);
    }
}
async function deleteRecord(id) {
    try {
        const [result] = await db.execute(
            'UPDATE products SET active = 1 WHERE id = ?',
            [id]
        );
        return result.affectedRows > 0;
    } catch (error) {
        console.error('Error in deleteRecord:', error);
        throw new Error(`Failed to delete product: ${error.message}`);
    }
}
async function filterProducts(filters, userId) {
    try {
        let query = `SELECT p.*, u.name as owner_name 
                     FROM products p 
                     LEFT JOIN users u ON p.owner = u.id 
                     WHERE p.active = 1`;
        let params = [];


        // Field mapping for special fields
        const fieldMapping = {
            'owner': 'p.owner',
            'created_at': 'p.created_at',
            'updated_at': 'p.updated_at'
        };

        if (filters && filters.length > 0) {
            for (const filter of filters) {
                const { contains, field_name, value_id } = filter;
                
                // Get the correct field name with table alias if needed
                const mappedField = fieldMapping[field_name] || `p.${field_name}`;

                switch (contains) {
                    case 'contains':
                        if (Array.isArray(value_id) && value_id.length > 0) {
                            const placeholders = value_id.map(() => '?').join(',');
                            query += ` AND ${mappedField} IN (${placeholders})`;
                            params.push(...value_id);
                        }
                        break;
                    case 'does not contain':
                        if (Array.isArray(value_id) && value_id.length > 0) {
                            const placeholders = value_id.map(() => '?').join(',');
                            query += ` AND ${mappedField} NOT IN (${placeholders})`;
                            params.push(...value_id);
                        }
                        break;

                    case 'is not empty':
                        query += ` AND ${mappedField} IS NOT NULL AND ${mappedField} != ''`;
                        break;

                    case 'is empty':
                        query += ` AND (${mappedField} IS NULL OR ${mappedField} = '')`;
                        break;

                    case '< less than':
                        query += ` AND ${mappedField} < ?`;
                        params.push(value_id);
                        break;

                    case '> greater than':
                        query += ` AND ${mappedField} > ?`;
                        params.push(value_id);
                        break;

                    case '= equal to':
                        query += ` AND ${mappedField} = ?`;
                        params.push(value_id);
                        break;

                    case '!= not equal to':
                        query += ` AND ${mappedField} != ?`;
                        params.push(value_id);
                        break;

                    case 'between':
                        if (Array.isArray(value_id) && value_id.length === 2) {
                            query += ` AND ${mappedField} BETWEEN ? AND ?`;
                            params.push(value_id[0], value_id[1]);
                        }
                        break;

                    // Date based filters
                    case 'last 30 minutes':
                        query += ` AND ${mappedField} >= ?`;
                        params.push(moment().subtract(30, 'minutes').format('YYYY-MM-DD HH:mm:ss'));
                        break;

                    case 'last 60 minutes':
                        query += ` AND ${mappedField} >= ?`;
                        params.push(moment().subtract(60, 'minutes').format('YYYY-MM-DD HH:mm:ss'));
                        break;

                    case 'last 2 hours':
                        query += ` AND ${mappedField} >= ?`;
                        params.push(moment().subtract(2, 'hours').format('YYYY-MM-DD HH:mm:ss'));
                        break;

                    case 'last 24 hours':
                        query += ` AND ${mappedField} >= ?`;
                        params.push(moment().subtract(24, 'hours').format('YYYY-MM-DD HH:mm:ss'));
                        break;

                    case 'last 7 days':
                        query += ` AND ${mappedField} >= ?`;
                        params.push(moment().subtract(7, 'days').format('YYYY-MM-DD HH:mm:ss'));
                        break;

                    case 'last 30 days':
                        query += ` AND ${mappedField} >= ?`;
                        params.push(moment().subtract(30, 'days').format('YYYY-MM-DD HH:mm:ss'));
                        break;

                    case 'last 90 days':
                        query += ` AND ${mappedField} >= ?`;
                        params.push(moment().subtract(90, 'days').format('YYYY-MM-DD HH:mm:ss'));
                        break;

                    case 'last 12 months':
                        query += ` AND ${mappedField} >= ?`;
                        params.push(moment().subtract(12, 'months').format('YYYY-MM-DD HH:mm:ss'));
                        break;
                }
            }
        }

        // Add user filter if userId is provided
       

        query += ` ORDER BY p.created_at DESC`;

        console.log('Query:', query);
        console.log('Params:', params);

        let filtersQuery = 'SELECT * FROM productviews'
        //const [filtersType] = await db.execute(filtersQuery, params);

        let queryFilte = `SELECT pv.*, pf.*
            FROM productviews pv
            LEFT JOIN productfilters pf ON pv.id = pf.view_id`;

            const [filtersType] = await db.execute(`
                SELECT pv.*, pf.*
                FROM productviews pv
                LEFT JOIN productfilters pf ON pv.id = pf.view_id
              `);
            
              // Grouping logic
              const grouped = {};
            
              filtersType.forEach(row => {
                const viewId = row.id;
            
                if (!grouped[viewId]) {
                  grouped[viewId] = {
                    ...row,
                    fileds: [] // <-- intentionally keeping your typo to match your sample
                  };
                }
            
                if (row.view_id) {
                  const filter = {
                    // Extract only filter-related fields
                    id: row.pf_id, // assuming filter id
                    field_id: row.field_id,
                    contains: row.contains,
                    filter: row.filter,
                    // ... add more fields as needed from productfilters
                  };
            
                  grouped[viewId].fileds.push(filter);
                }
              });
            
              // Convert object to array
              const result = Object.values(grouped);

  //const [filtersType] = await db.execute(queryFilte, []);
        const [rows] = await db.execute(query, params);

        // Get all fields and choices
        const [fields] = await db.execute('SELECT * FROM productfields ORDER BY id');
        const [choices] = await db.execute('SELECT * FROM productfieldchoices');
        var map ={};
        var productsInfoArray=[];
        // Process each product
         rows.map(product => {
            // Create a copy of fields for each product to avoid sharing references
            const productFields = fields.map(field => ({
                ...field,
                choices: choices.filter(choice => choice.productfield_id === field.id),
                field_value: field.custom_field === 1 && product.custom_field 
                    ? JSON.parse(product.custom_field).find(cf => cf[field.field_name])?.[field.field_name] || null
                    : product[field.field_name]
            }));
            var details={
                id: product.id.toString(),
                products: productFields,
                owner_name: product.owner_name
            }
            productsInfoArray.push(details);
           
            });
            
            map = {
                productInfo:productsInfoArray,
                filtersType:result
            } 
            return map;
       

    } catch (error) {
        console.error('Error in filterProducts:', error);
        throw new Error(`Failed to filter products: ${error.message}`);
    }
}
async function updatePrice(productId, pricingData) {
    try {
        const { pricing_type, currencies, subscription_cycle, no_of_billing_cycle, setup_fee } = pricingData;

        // Validate required fields
        if (!productId) {
            throw new Error('Product ID is required');
        }

        if (!pricing_type) {
            throw new Error('Pricing type is required');
        }

        // Validate currencies array
        if (!Array.isArray(currencies) || currencies.length === 0) {
            throw new Error('At least one currency is required');
        }

        // Validate currency data
        for (const curr of currencies) {
            if (!curr.base_currency_amount) {
                throw new Error('Currency amount is required');
            }
            if (!curr.currency) {
                throw new Error('Currency code is required');
            }
        }

        // Validate subscription fields if pricing type is subscription
        if (pricing_type === 'Subscription Pricing') {
            if (!subscription_cycle) {
                throw new Error('Subscription cycle is required for subscription pricing');
            }
            if (!no_of_billing_cycle || no_of_billing_cycle <= 0) {
                throw new Error('Valid number of billing cycles is required for subscription pricing');
            }
        }

        //await db.execute('START TRANSACTION');

        try {
            // Update product with pricing type and subscription details
            const [result] = await db.execute(
                `UPDATE products 
                 SET pricing_type = ?,
                     base_currency_amount = ?,
                     currency = ?,
                     subscription_cycle = ?,
                     no_of_billing_cycle = ?,
                     updated_at = CURRENT_TIMESTAMP 
                 WHERE id = ? AND active = 1`,
                [
                    pricing_type,
                    currencies[0].base_currency_amount,
                    currencies[0].currency,
                    pricing_type === 'Subscription Pricing' ? subscription_cycle : null,
                    pricing_type === 'Subscription Pricing' ? no_of_billing_cycle : null,
                    productId
                ]
            );

            if (result.affectedRows === 0) {
                throw new Error('Product not found or inactive');
            }

            // Delete existing currency mappings
            await db.execute(
                'DELETE FROM productcurrencies WHERE product_id = ?',
                [productId]
            );

            // Insert new currency mappings with setup fee
            for (const curr of currencies) {
                await db.execute(
                    `INSERT INTO productcurrencies 
                     (product_id, base_currency_amount, currency, setup_fee) 
                     VALUES (?, ?, ?, ?)`,
                    [
                        productId,
                        curr.base_currency_amount,
                        curr.currency,
                        pricing_type === 'Subscription Pricing' ? (setup_fee || null) : null
                    ]
                );
            }

            //await db.execute('COMMIT');

            return {
                success: true,
                productId: productId,
                pricing_type: pricing_type,
                currencies: currencies.map(curr => ({
                    ...curr,
                    setup_fee: pricing_type === 'Subscription Pricing' ? (setup_fee || null) : null
                })),
                ...(pricing_type === 'Subscription Pricing' && {
                    subscription_cycle,
                    no_of_billing_cycle
                })
            };

        } catch (error) {
            //await db.execute('ROLLBACK');
            throw error;
        }

    } catch (error) {
        console.error('Error in updatePrice:', error);
        throw new Error(`Failed to update product price: ${error.message}`);
    }
}
async function getProductFields(userId) {
    try {
        // Validate userId
        if (!userId) {
            throw new Error('User ID is required');
        }

        // Get all product fields
        const [fields] = await db.execute(
            'SELECT * FROM productfields WHERE active = 1 ORDER BY id'
        );

        if (!fields || fields.length === 0) {
            throw new Error('No product fields found');
        }

        // Get users for lookup fields
        const [users] = await db.execute(
            'SELECT id, name FROM users WHERE active = 1'
        );

        // Get all field choices
        const [choices] = await db.execute(
            'SELECT * FROM productfieldchoices'
        );

        // Process each field
        const processedFields = fields.map(field => {
            let values = [];
            let contains_list = [];

            // Set contains_list based on field type
            switch (field.field_type.toLowerCase()) {
                case 'dropdown':
                    contains_list = [
                        { name: "contains" },
                        { name: "does not contain" },
                        { name: "is empty" },
                        { name: "is not empty" }
                    ];
                    // Add dropdown values from choices with id and custom_option
                    values = choices
                        .filter(choice => choice.productfield_id === field.id)
                        .map(choice => ({
                            id: choice.id,
                            value: choice.value,
                            value: choice.custom_option || 0
                        }));
                    break;

                case 'number':
                    contains_list = [
                        { name: "< less than" },
                        { name: "> greater than" },
                        { name: "= equal to" },
                        { name: "!= not equal to" },
                        { name: "between" },
                        { name: "is not empty" },
                        { name: "is empty" }
                    ];
                    break;

                case 'datepicker':
                    contains_list = [
                        { name: "today" },
                        { name: "yesterday" },
                        { name: "last 7 days" },
                        { name: "last 30 days" },
                        { name: "this month" },
                        { name: "last month" },
                        { name: "is empty" },
                        { name: "is not empty" }
                    ];
                    break;

                case 'lookup':
                    contains_list = [
                        { name: "contains" },
                        { name: "does not contain" },
                        { name: "is empty" },
                        { name: "is not empty" }
                    ];
                    // Add lookup values
                    values = [
                        { id: null, value: "unassigned" },
                        ...users.map(user => ({
                            id: user.id,
                            value: user.id === userId ? "Me" : user.name
                        }))
                    ];
                    break;

                default:
                    contains_list = [
                        { name: "contains" },
                        { name: "does not contain" },
                        { name: "is empty" },
                        { name: "is not empty" }
                    ];
            }

            return {
                id: field.id,
                custom_field: field.custom_field || 0,
                field_label: field.field_label,
                field_name: field.field_name,
                field_type: field.field_type,
                values: values,
                contains_list: contains_list
            };
        });

        return processedFields;

    } catch (error) {
        console.error('Error in getProductFields:', error);
        throw new Error(`Failed to fetch product fields: ${error.message}`);
    }
}
async function createProductView(viewData) {
    try {
                // Insert product view
        const [viewResult] = await db.execute(
            `INSERT INTO productviews 
            (name, share_with, accessibility, users) 
            VALUES (?, ?, ?, ?)`,
            [viewData.title, viewData.share_with, viewData.accessibility, viewData.selected_users.join(',') || null]
        );

        const viewId = viewResult.insertId;

        // Insert filters if any
        if (viewData.filter && Array.isArray(viewData.filter)) {
            for (const filter of viewData.filter) {
                if (filter.field_label && filter.field_label.trim() !== '') {
                    await db.execute(
                        `INSERT INTO productfilters 
                        (view_id, name, field_id, contains, filter, active) 
                        VALUES (?, ?, ?, ?, ?, ?)`,
                        [
                            viewId,
                            filter.field_label,
                            filter.field_name,
                            filter.contains,
                            JSON.stringify(filter.value_id),
                            1
                        ]
                    );
                }
            }
        }

       

        // Get the created view
        const [views] = await db.execute(
            'SELECT * FROM productviews WHERE id = ?',
            [viewId]
        );

        return views[0];

    } catch (error) {
        console.error('Error in createProductView:', error);
        throw new Error(`Failed to create product view: ${error.message}`);
    }
}
async function deletePrice(productId) {
    try {
        // Validate product ID
        if (!productId) {
            throw new Error('Product ID is required');
        }

        // Update product to remove pricing related fields
        const [result] = await db.execute(
            `UPDATE products 
             SET base_currency_amount = NULL,
                 currency = NULL,
                 pricing_type = NULL,
                 setup_fee = NULL,
                 subscription_cycle = NULL,
                 no_of_billing_cycle = NULL,
                 updated_at = CURRENT_TIMESTAMP
             WHERE id = ? AND active = 1`,
            [productId]
        );

        if (result.affectedRows === 0) {
            throw new Error('Product not found or inactive');
        }

        return true;

    } catch (error) {
        console.error('Error in deletePrice:', error);
        throw new Error(`Failed to remove product price: ${error.message}`);
    }
}
async function updateProductView(viewData) {
    try {
        const { view_id, title, share_with, accessibility } = viewData;

        // Check if view exists
        const [existingView] = await db.execute(
            'SELECT * FROM productviews WHERE id = ?',
            [view_id]
        );

        if (!existingView.length) {
            throw new Error('Product view not found');
        }

        // Update view
        const [result] = await db.execute(
            `UPDATE productviews 
             SET name = ?,
                 share_with = ?,
                 accessibility = ?,
                 updated_at = CURRENT_TIMESTAMP
             WHERE id = ?`,
            [title, share_with, accessibility, view_id]
        );

        if (result.affectedRows === 0) {
            throw new Error('Failed to update product view');
        }

        // Get updated view
        const [updatedView] = await db.execute(
            'SELECT * FROM productviews WHERE id = ?',
            [view_id]
        );

        return updatedView[0];

    } catch (error) {
        console.error('Error in updateProductView:', error);
        throw new Error(`Failed to update product view: ${error.message}`);
    }
}

async function deleteProductView(viewId) {
    try {
        //await db.execute('START TRANSACTION');

        // Delete related filters first
        await db.execute(
            'DELETE FROM productfilters WHERE view_id = ?',
            [viewId]
        );

        // Delete the view
        const [result] = await db.execute(
            'DELETE FROM productviews WHERE id = ?',
            [viewId]
        );

        //await db.execute('COMMIT');

        return result.affectedRows > 0;

    } catch (error) {
       // await db.execute('ROLLBACK');
        console.error('Error in deleteProductView:', error);
        throw new Error(`Failed to delete product view: ${error.message}`);
    }
}
async function getCategories() {
    try {
        // Get category field from productfields
        const [categoryField] = await db.execute(
            `SELECT id, field_name 
             FROM productfields 
             WHERE field_name = 'category' 
             AND active = 1`
        );

        if (!categoryField.length) {
            throw new Error('Category field not found');
        }

        // Get category choices from productfieldchoices
        const [categoryChoices] = await db.execute(
            `SELECT id, custom_option as category_name 
             FROM productfieldchoices 
             WHERE productfield_id = ?`,
            [categoryField[0].id]
        );

        if (!categoryChoices.length) {
            return [];
        }

        // Get product currencies for all products
        const [allProductCurrencies] = await db.execute(
            `SELECT pc.* 
             FROM productcurrencies pc
             `
        );

        // Process each category
        const processedCategories = await Promise.all(categoryChoices.map(async (category) => {
            // Get products for this category
            const [products] = await db.execute(
                `SELECT p.*, u.name as owner_name,
                 DATE_FORMAT(p.create_at, "%Y-%m-%d") as formatted_create_at,
                 DATE_FORMAT(p.update_at, "%Y-%m-%d") as formatted_update_at,
                 (SELECT COUNT(*) FROM products 
                  WHERE category = ? AND active = 1) as product_count  
                 FROM products p 
                 LEFT JOIN users u ON p.owner = u.id 
                 WHERE p.category = ? AND p.active = 1 
                 ORDER BY p.created_at DESC`,
                [category.id, category.id]
            );

            return {
                id: category.id,
                category: category.category_name,
                category_name: category.category_name,
                product_count: products[0]?.product_count || 0,
                products: products.map(product => ({
                    id: product.id.toString(),
                    product_name: product.product_name,
                    owner_name: product.owner_name,
                    base_currency_amount: product.base_currency_amount,
                    currency: product.currency,
                    pricing_type: product.pricing_type,
                    created_at: product.created_at,
                    product_image: product.product_image || "",
                    productcurrencies: allProductCurrencies
                        .filter(pc => pc.product_id === product.id)
                        .map(currency => ({
                            id: currency.id,
                            currency_name: currency.currency,
                            base_currency_amount: currency.base_currency_amount,
                            setup_fee: currency.setup_fee
                        }))
                }))
            };
        }));

        return processedCategories;

    } catch (error) {
        console.error('Error in getCategories:', error);
        throw new Error(`Failed to fetch categories: ${error.message}`);
    }
}
async function addDealProducts(dealId, products, dealValue,userId) {
    try {
        //await db.execute('START TRANSACTION');

        try {
            // Validate deal exists
            const [deal] = await db.execute(
                'SELECT * FROM deals WHERE id = ? AND active = 0',
                [dealId]
            );

            if (!deal.length) {
                throw new Error('Deal not found');
            }

            let totalPrice = 0;
            let totalItems = 0;

            // Process each product
            for (const product of products) {
                const totalAmount = parseFloat(product.unitprice || 0) * parseInt(product.quantity || 0);

                if (product.id) {
                    // Update existing deal product
                    await db.execute(
                        `UPDATE dealproducts 
                         SET product_id = ?,
                             unitprice = ?,
                             quantity = ?,
                             discount_type = ?,
                             discount = ?,
                             totalprice = ?,
                             setupfree = ?,
                             billingcycle = ?,
                             no_of_billing_cycle = ?,
                             updated_at = CURRENT_TIMESTAMP
                         WHERE id = ?`,
                        [
                            product.product_id || null,
                            product.unitprice || 0,
                            product.quantity || 0,
                            product.discount_type || null,
                            product.discount || 0,
                            totalAmount,
                            product.setupfree || null,
                            product.billingcycle || null,
                            product.no_of_billing_cycle || null,
                            product.id
                        ]
                    );
                } else {
                    // Insert new deal product
                    await db.execute(
                        `INSERT INTO dealproducts 
                         (deal_id, product_id, unitprice, quantity, discount_type, 
                          discount, totalprice, setupfree, billingcycle, 
                          no_of_billing_cycle, userid) 
                         VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`,
                        [
                            dealId,
                            product.product_id || null,
                            product.unitprice || 0,
                            product.quantity || 0,
                            product.discount_type || null,
                            product.discount || 0,
                            totalAmount,
                            product.setupfree || null,
                            product.billingcycle || null,
                            product.no_of_billing_cycle || null,
                            userId
                        ]
                    );

                    // If deal has quotation, add product to quote
                    if (deal[0].quotation_id) {
                        await db.execute(
                            `INSERT INTO quoteproducts 
                             (quotation_id, product_id, unitprice, quantity, 
                              discount_type, discount, totalprice, setupfree, 
                              billingcycle, no_of_billing_cycle, userid) 
                             VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`,
                            [
                                deal[0].quotation_id,
                                product.product_id || null,
                                product.unitprice || 0,
                                product.quantity || 0,
                                product.discount_type || null,
                                product.discount || 0,
                                totalAmount,
                                product.setupfree || null,
                                product.billingcycle || null,
                                product.no_of_billing_cycle || null,
                                userId
                            ]
                        );
                    }
                }

                totalPrice += totalAmount;
                totalItems++;
            }

            // Update deal amount
            await db.execute(
                'UPDATE deals SET amount = ? WHERE id = ?',
                [dealValue || 0, dealId]
            );

            // Update contact deal stats
            await updateContactDealStats(dealId);

            //await db.execute('COMMIT');
            return true;

        } catch (error) {
            //await db.execute('ROLLBACK');
            throw error;
        }

    } catch (error) {
        console.error('Error in addDealProducts:', error);
        throw new Error(`Failed to add deal products: ${error.message}`);
    }
}

// Helper function to update contact deal stats
async function updateContactDealStats(dealId) {
    const [contactDeals] = await db.execute(
        `SELECT 
            COUNT(*) as deal_count,
            SUM(CASE WHEN deal_stage IN ('Won', 'Lost') THEN 1 ELSE 0 END) as open_deal_count,
            SUM(CASE WHEN deal_stage IN ('Won', 'Lost') THEN amount ELSE 0 END) as open_deal_value
         FROM deals 
         WHERE contacts = (SELECT contacts FROM deals WHERE id = ?)`,
        [dealId]
    );

    if (contactDeals.length) {
        await db.execute(
            `UPDATE contacts 
             SET dealcount = ?,
                 open_dealvalue = ?,
                 open_dealcount = ?
             WHERE id = (SELECT contacts FROM deals WHERE id = ?)`,
            [
                contactDeals[0].deal_count || 0,
                contactDeals[0].open_deal_value || 0,
                contactDeals[0].open_deal_count || 0,
                dealId
            ]
        );
    }
}
async function getDealProducts(dealId) {
    try {
        // Validate and get deal
        const [deal] = await db.execute(
            'SELECT * FROM deals WHERE id = ? AND active = 1',
            [dealId]
        );

        if (!deal.length) {
            throw new Error('Deal not found');
        }

        // Get quotations for the deal
        const [quotations] = await db.execute(
            'SELECT id, quotation_name FROM quotations WHERE deal_id = ? AND active = 1',
            [dealId]
        );

        // Check auto sync status
        const autoSync = deal[0].quotation_id ? true : false;
        const syncQuoteId = deal[0].quotation_id || '';

        // Get deal products with sync status
        const [dealProducts] = await db.execute(
            `SELECT 
                dp.*, 
                p.product_name,
                p.pricing_type,
                CASE 
                    WHEN qp.id IS NOT NULL THEN true 
                    ELSE false 
                END as is_synced,
                COALESCE(q.id, '') as synced_quote_id,
                COALESCE(q.quotation_name, '') as synced_quote_name
             FROM dealproducts dp
             LEFT JOIN products p ON dp.product_id = p.id 
             LEFT JOIN quoteproducts qp ON dp.product_id = qp.product_id 
                AND qp.quotations_id = ?
             LEFT JOIN quotations q ON qp.quotations_id = q.id
             WHERE dp.deal_id = ?`,
            [deal[0].quotation_id, dealId]
        );

        let totalPrice = 0;

        // Format deal products
        const formattedProducts = dealProducts.map(product => {
            totalPrice += parseFloat(product.totalprice || 0);
            return {
                id: product.id,
                deal_id: product.deal_id,
                product_id: product.product_id,
                product_name: product.product_name,
                unitprice: product.unitprice,
                quantity: product.quantity,
                discount_type: product.discount_type,
                discount: product.discount,
                totalprice: product.totalprice,
                setupfee: product.setupfree,
                billingcycle: product.billingcycle,
                pricing_type: product.pricing_type,
                is_synced: product.is_synced,
                synced_quote_id: product.synced_quote_id,
                synced_quote_name: product.synced_quote_name
            };
        });

        // Get price settings and default currency
        const [priceSettings] = await db.execute(
            'SELECT sales_tax FROM pricesettings LIMIT 1'
        );

        const [currency] = await db.execute(
            'SELECT `decimal` FROM currencies WHERE is_default = 1 AND active = 1 LIMIT 1'
        );

        // Calculate tax and total amounts
        const tax = priceSettings[0]?.sales_tax || 0;
        const taxAmount = (totalPrice * tax) / 100;
        const totalAmount = totalPrice + taxAmount;
        const decimals = currency[0]?.decimal || 2;

        // Format deal value
        const dealValue = {
            subtotal: totalPrice.toFixed(decimals),
            salestax: tax,
            salestaxamount: taxAmount.toFixed(decimals),
            dealvalue: totalAmount.toFixed(decimals),
            autosync: autoSync,
            syncquoteid: syncQuoteId,
            quotations: quotations
        };

        return {
            dealproducts: formattedProducts,
            dealvalue: dealValue
        };

    } catch (error) {
        console.error('Error in getDealProducts:', error);
        throw new Error(`Failed to get deal products: ${error.message}`);
    }
}
async function syncQuote(dealId, quoteId, userId) {
    try {
        //await db.execute('START TRANSACTION');

        try {
            // Get quote products
            const [quoteProducts] = await db.execute(
                'SELECT * FROM quoteproducts WHERE quotations_id = ?',
                [quoteId]
            );

            // Add each quote product to deal
            for (const product of quoteProducts) {
                await db.execute(
                    `INSERT INTO dealproducts 
                     (deal_id, product_id, unitprice, quantity, discount_type,
                      discount, totalprice, setupfree, billingcycle,
                      no_of_billing_cycle, userid) 
                     VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`,
                    [
                        dealId,
                        product.product_id,
                        product.unitprice,
                        product.quantity,
                        product.discount_type,
                        product.discount,
                        product.totalprice,
                        product.setupfree,
                        product.billingcycle,
                        product.no_of_billing_cycle,
                        userId
                    ]
                );
            }

            // Update deal with quote ID
            await db.execute(
                'UPDATE deals SET quotation_id = ? WHERE id = ?',
                [quoteId, dealId]
            );

            // Mark quote as synced
            await db.execute(
                "UPDATE quotations SET synced = 'synced' WHERE id = ?",
                [quoteId]
            );

            //await db.execute('COMMIT');
            return true;

        } catch (error) {
            //await db.execute('ROLLBACK');
            throw error;
        }

    } catch (error) {
        console.error('Error in syncQuote:', error);
        throw new Error(`Failed to sync quote: ${error.message}`);
    }
}
async function removeSync(dealId) {
    try {
        //await db.execute('START TRANSACTION');

        try {
            // Update deal to remove quote ID
            const [result] = await db.execute(
                'UPDATE deals SET quotation_id = NULL WHERE id = ?',
                [dealId]
            );

            if (result.affectedRows === 0) {
                throw new Error('Deal not found');
            }

            // Delete all deal products
            await db.execute(
                'DELETE FROM dealproducts WHERE deal_id = ?',
                [dealId]
            );

            //await db.execute('COMMIT');
            return true;

        } catch (error) {
           // await db.execute('ROLLBACK');
            throw error;
        }

    } catch (error) {
        console.error('Error in removeSync:', error);
        throw new Error(`Failed to remove sync: ${error.message}`);
    }
}
async function removeDealProduct(dealId, productId) {
    try {
        //await db.execute('START TRANSACTION');

        try {
            // Check if deal and product exist
            const [dealProduct] = await db.execute(
                'SELECT * FROM dealproducts WHERE id = ? AND deal_id = ?',
                [productId, dealId]
            );

            if (!dealProduct.length) {
                throw new Error('Deal product not found');
            }

            // Delete the product from deal
            await db.execute(
                'DELETE FROM dealproducts WHERE id = ? AND deal_id = ?',
                [productId, dealId]
            );

            // Get the deal's quotation ID
            const [deal] = await db.execute(
                'SELECT quotation_id FROM deals WHERE id = ?',
                [dealId]
            );

            // If deal has quotation, remove product from quote too
            if (deal[0]?.quotation_id) {
                await db.execute(
                    'DELETE FROM quoteproducts WHERE quotation_id = ? AND product_id = ?',
                    [deal[0].quotation_id, dealProduct[0].product_id]
                );
            }

            // Recalculate deal total
            const [remainingProducts] = await db.execute(
                'SELECT SUM(totalprice) as total FROM dealproducts WHERE deal_id = ?',
                [dealId]
            );

            // Update deal amount
            await db.execute(
                'UPDATE deals SET amount = ? WHERE id = ?',
                [remainingProducts[0]?.total || 0, dealId]
            );

            // Update contact deal stats
            await updateContactDealStats(dealId);

           // await db.execute('COMMIT');
            return true;

        } catch (error) {
           // await db.execute('ROLLBACK');
            throw error;
        }

    } catch (error) {
        console.error('Error in removeDealProduct:', error);
        throw new Error(`Failed to remove deal product: ${error.message}`);
    }
}
async function uploadFile(file, data, userId) {
    try {
        // Validate file
        if (!file) {
            throw new Error('File is required');
        }

        // Generate unique filename using MD5
        const originalName = file.originalname;
        const fileType = path.extname(originalName).substring(1);
        const filename = crypto.createHash('md5').update(originalName).digest('hex');
        const newFileName = `${filename}.${fileType}`;

        // Create upload directory if it doesn't exist
        const uploadDir = path.join(process.cwd(), 'public', 'fileupload');
        if (!fs.existsSync(uploadDir)) {
            fs.mkdirSync(uploadDir, { recursive: true });
        }

        // Move file to destination
        const filePath = path.join(uploadDir, newFileName);
        fs.copyFileSync(file.path, filePath);

        // Insert file record in database
        const [result] = await db.execute(
            `INSERT INTO fileuploads 
             (file_name, display_name, file_size, share_with_team, 
              type, type_id, owner, upload_at) 
             VALUES (?, ?, ?, ?, ?, ?, ?, CURRENT_TIMESTAMP)`,
            [
                newFileName,
                originalName,
                file.size,
                data.share_with_team || 0,
                'products',
                data.type_id,
                userId
            ]
        );

        // Add record in fileimages table for product association
        if (result.insertId) {
            await db.execute(
                `INSERT INTO fileimages 
                 (filename, file_url, file_size, 
                  belongsto, belongsto_id, user_id) 
                 VALUES (?, ?, ?, ?, ?, ?)`,
                [
                    originalName,
                    `/fileupload/${newFileName}`,
                    file.size,
                    'products',
                    data.type_id,
                    userId
                ]
            );
        }

        return {
            status: true,
            file_id: result.insertId,
            file_name: newFileName,
            display_name: originalName
        };

    } catch (error) {
        console.error('Error in uploadFile:', error);
        throw new Error(`Failed to upload file: ${error.message}`);
    }
}
async function deleteFile(fileId) {
    try {
        //await db.execute('START TRANSACTION');

        try {
            // Get file info first
            const [fileInfo] = await db.execute(
                'SELECT * FROM fileimages WHERE id = ?',
                [fileId]
            );

            if (!fileInfo.length) {
                throw new Error('File not found');
            }

            // Delete from fileimages table - Fixed SQL syntax
            const [result] = await db.execute(
                'UPDATE fileimages SET active = 1 WHERE id = ?',
                [fileId]
            );

            if (result.affectedRows === 0) {
                throw new Error('File delete failed');
            }

            // Delete physical file if it exists
            if (fileInfo[0].file_url) {  // Changed from file_path to file_url
                const filePath = path.join(__dirname, '..', 'public', fileInfo[0].file_url);
                try {
                    if (fs.existsSync(filePath)) {
                        await fs.promises.unlink(filePath);
                    }
                } catch (unlinkError) {
                    console.error('Error deleting physical file:', unlinkError);
                    // Continue even if physical file deletion fails
                }
            }

            //await db.execute('COMMIT');
            return true;

        } catch (error) {
            //await db.execute('ROLLBACK');
            throw error;
        }

    } catch (error) {
        console.error('Error in deleteFile:', error);
        throw new Error(`Failed to delete file: ${error.message}`);
    }
}
async function getLookupData() {
    try {
        // Get active users
        const [users] = await db.execute(
            'SELECT id, name FROM users WHERE active = 1 ORDER BY name'
        );

        // Get territories
        const [territories] = await db.execute(
            'SELECT id, territory FROM territories WHERE active = 1 ORDER BY territory'
        );

        return {
            users: users.map(user => ({
                id: user.id,
                name: user.name
            })),
            territories: territories.map(territory => ({
                id: territory.id,
                name: territory.territory
            }))
        };

    } catch (error) {
        console.error('Error in getLookupData:', error);
        throw new Error(`Failed to fetch lookup data: ${error.message}`);
    }
}


async function getAllFiltersList() {
    try {
        console.log("=== Starting getAllFiltersList Model Function ===");
        console.log("Executing database query for productviews");
        
        // Get the views with their filters
        const [views] = await db.execute(`
            SELECT cv.*, cf.id as filter_id, cf.name as filter_name, 
                   cf.field_id, cf.contains, cf.filter as filter_value
            FROM productviews cv
            LEFT JOIN productfilters cf ON cv.id = cf.view_id
                    `
        );
        
        // Transform the data to match findAll format
        const transformedViews = views.reduce((acc, row) => {
            if (!acc[row.id]) {
                acc[row.id] = {
                    id: row.id,
                    name: row.name,
                    share_with: row.share_with,
                    accessibility: row.accessibility,
                    users: row.users ? row.users.split(',') : [],
                    owner_name: row.owner_name,
                    created_at: formatDateTime(row.created_at),
                    updated_at: formatDateTime(row.updated_at),
                    filters: []
                };
            }
            
            if (row.filter_id) {
                acc[row.id].filters.push({
                    id: row.filter_id,
                    name: row.filter_name,
                    field_id: row.field_id,
                    contains: row.contains,
                    filter: row.filter_value ? JSON.parse(row.filter_value) : [],
                });
            }
            
            return acc;
        }, {});

        const result = Object.values(transformedViews);
        
        console.log("Number of views found:", result.length);
        return result;
        
    } catch (error) {
        console.error("=== Error in getAllFiltersList Model Function ===");
        console.error("Error details:", error);
        throw new Error(`Failed to fetch product filters: ${error.message}`);
    }
}
function formatDateTime(datetime) {
    if (!datetime) return null;
    return new Date(datetime).toISOString();
}


async function updateFilterView(viewData) {
    console.log("updateFilterView inits "+JSON.stringify(viewData));
    try {
                // Insert product view
                const selectedUsers = Array.isArray(viewData.users) && viewData.users.length > 0
                ? viewData.users.join(',')
                : null;
              
              const [viewResult] = await db.execute(
                `UPDATE productviews 
                 SET name = ?, share_with = ?, accessibility = ?, users = ?
                 WHERE id = ?`,
                [viewData.title, viewData.share_with, viewData.accessibility, selectedUsers, viewData.id]
              );

        // Get the created view
        const [views] = await db.execute(
            'SELECT * FROM productviews WHERE id = ?',
            [viewData.id]
        );

        return views[0];

    } catch (error) {
        console.error('Error in contactviews:', error);
        throw new Error(`Failed to create account view: ${error.message}`);
    }
}


async function deleteFilterView(viewId) {
    try {
        //await db.execute('START TRANSACTION');

        // Delete related filters first
        await db.execute(
            'DELETE FROM productfilters WHERE view_id = ?',
            [viewId]
        );

        // Delete the view
        const [result] = await db.execute(
            'DELETE FROM productviews WHERE id = ?',
            [viewId]
        );

        //await db.execute('COMMIT');

        return result.affectedRows > 0;

    } catch (error) {
       // await db.execute('ROLLBACK');
        console.error('Error in productviews:', error);
        throw new Error(`Failed to delete product view: ${error.message}`);
    }
}
module.exports = {
    create,
    findById,
    findAll,
    update,
    deleteRecord,
    filterProducts,
    updatePrice,
    getProductFields,
    createProductView,
    deletePrice,
    updateProductView,
    deleteProductView,
    getCategories,
    addDealProducts,
    getDealProducts,
    syncQuote,
    removeSync,
    removeDealProduct,
    uploadFile,
    deleteFile,
    getLookupData,
    getAllFiltersList,
    updateFilterView,
    deleteFilterView
};