const db = require('../config/db');
const PDFDocument = require('pdfkit');
const fs = require('fs');
const path = require('path');
const moment = require('moment');
class Quotation {
    static async create(fields, userId) {
        try {
            // Get last quotation number
            const [lastQuotation] = await db.execute(
                'SELECT quotation_no FROM quotations ORDER BY id DESC LIMIT 1'
            );

            // Convert to number and increment
            let quotationNo = 1;
            if (lastQuotation.length) {
                // Ensure numeric conversion
                const lastNumber = Number(lastQuotation[0].quotation_no);
console.log("last Number"+lastNumber);
                if (!isNaN(lastNumber)) {
                    quotationNo = lastNumber + 1;
                }
            }
console.log("last Number"+quotationNo);
            // Process fields
            const standardFields = {};
            const customFields = [];

            fields.forEach(field => {
                if (field.field_value !== null && field.field_value !== undefined) {
                    if (field.custom_field === 0) {
                        standardFields[field.field_name] = field.field_value;
                    } else {
                        customFields.push({
                            [field.field_name]: field.field_value
                        });
                    }
                }
            });

            // Insert quotation with numeric quotation_no
            const [result] = await db.execute(
                `INSERT INTO quotations 
                 (quotation_no, quotation_date, quotation_name, quotation_type,
                  quotation_template, deal_id, account_id, contact_id, 
                  total_price, custom_field, users_id, create_at, active)
                 VALUES (?, CURRENT_DATE, ?, ?, ?, ?, ?, ?, ?, ?, ?, CURRENT_TIMESTAMP, 1)`,
                [
                    quotationNo, // Using the numeric value
                    standardFields.quotation_name,
                    standardFields.quotation_type || null,
                    standardFields.quotation_template || null,
                    standardFields.deal_id || null,
                    standardFields.account_id || null,
                    standardFields.contact_id || null,
                    standardFields.total_price || 0,
                    customFields.length > 0 ? JSON.stringify(customFields) : null,
                    userId
                ]
            );

            return { id: result.insertId, quotation_no: quotationNo };

        } catch (error) {
            throw new Error(`Failed to create quotation: ${error.message}`);
        }
    }

    static async getAll(filters = {}) {
        try {
            let query = `
                SELECT q.*, 
                       a.name as account_name,
                       c.first_name as contact_name,
                       d.name as deal_name,
                       u.name as owner_name,
                       DATE_FORMAT(q.create_at, '%Y-%m-%d') as formatted_create_at
                FROM quotations q
                LEFT JOIN accounts a ON q.account_id = a.id
                LEFT JOIN contacts c ON q.contact_id = c.id
                LEFT JOIN deals d ON q.deal_id = d.id
                LEFT JOIN users u ON q.users_id = u.id
                WHERE q.active = 1
            `;

            const params = [];

            // Add deal_id filter
            if (filters.deal_id) {
                query += ` AND q.deal_id = ?`;
                params.push(filters.deal_id);
            }

            // Add search filter
            if (filters.search) {
                query += ` AND (q.quotation_name LIKE ? OR a.name LIKE ?)`;
                params.push(`%${filters.search}%`, `%${filters.search}%`);
            }

            query += ` ORDER BY q.create_at DESC`;

            const [quotations] = await db.execute(query, params);
            return quotations;

        } catch (error) {
            throw new Error(`Failed to fetch quotations: ${error.message}`);
        }
    }

    static async getById(id) {
        try {
            // Get base URL from environment variable or use default
            const baseUrl = process.env.BASE_URL || 'http://localhost:5000';
            // Get quotation details
            const [quotations] = await db.execute(
                `SELECT q.*, 
                        a.name as account_name,
                        c.first_name as contact_name,
                        d.name as deal_name,
                        u.name as owner_name,
                        DATE_FORMAT(q.create_at, '%Y-%m-%d') as formatted_create_at
                 FROM quotations q
                 LEFT JOIN accounts a ON q.account_id = a.id
                 LEFT JOIN contacts c ON q.contact_id = c.id
                 LEFT JOIN deals d ON q.deal_id = d.id
                 LEFT JOIN users u ON q.users_id = u.id
                 WHERE q.id = ? AND q.active = 1`,
                [id]
            );

            if (!quotations.length) {
                throw new Error('Quotation not found');
            }

            const quotation = quotations[0];

            // Check if PDF exists, if not generate it
            if ((!quotation.pdf_path)||(quotation.pdf_path)) {
                const standardFields = {
                    quotation_no:quotation.quotation_no,
                    quotation_name: quotation.quotation_name,
                    quotation_type: quotation.quotation_type,
                    quotation_template: quotation.quotation_template,
                    deal_id: quotation.deal_id,
                    account_id: quotation.account_id,
                    contact_id: quotation.contact_id,
                    total_price: quotation.total_price
                };

                const customFields = quotation.custom_field ? JSON.parse(quotation.custom_field) : [];

                // Generate PDF
                await this.generateQuotationPDF(id, standardFields, customFields);

                // Get updated PDF path
                const [updatedQuote] = await db.execute(
                    'SELECT pdf_path FROM quotations WHERE id = ?',
                    [id]
                );
                quotation.pdf_path = updatedQuote[0]?.pdf_path;
            }

            // Add PDF URLs
            if (quotation.pdf_path) {
                quotation.pdf_url = quotation.pdf_path;
                quotation.pdf_full_url = `${baseUrl}${quotation.pdf_path}`;
            }

            // Get fields and process values
            const [fields] = await db.execute('SELECT * FROM quotationfields WHERE active = 1');
            const customFields = quotation.custom_field ? JSON.parse(quotation.custom_field) : [];

            // Get quotation stages
            const [stages] = await db.execute(
                `SELECT id, stage 
                 FROM quotationstages 
                 WHERE active = 1`
            );

            // Process fields and add quotation stages
            quotation.fields = fields.map(field => ({
                ...field,
                field_value: field.custom_field === 0 ? 
                    quotation[field.field_name] : 
                    customFields.find(cf => cf[field.field_name])?.[field.field_name] || null
            }));

            quotation.stages = stages.map(stage => ({
                id: stage.id,
                stage_name: stage.stage,
                is_current: stage.stage === quotation.quotation_stage
            }));

            return quotation;

        } catch (error) {
            throw new Error(`Failed to fetch quotation: ${error.message}`);
        }
    }

    static async update(id, fields, userId) {
        try {
            //await db.execute('START TRANSACTION');

            try {
                const standardFields = {};
                const customFields = [];

                fields.forEach(field => {
                    if (field.field_value !== null && field.field_value !== undefined) {
                        if (field.custom_field === 0) {
                            standardFields[field.field_name] = field.field_value;
                        } else {
                            customFields.push({
                                [field.field_name]: field.field_value
                            });
                        }
                    }
                });

                const [result] = await db.execute(
                    `UPDATE quotations 
                     SET quotation_name = ?,
                         quotation_type = ?,
                         quotation_template = ?,
                         deal_id = ?,
                         account_id = ?,
                         contact_id = ?,
                         total_price = ?,
                         custom_field = ?,
                         users_id = ?,
                         update_at = CURRENT_TIMESTAMP
                     WHERE id = ? AND active = 1`,
                    [
                        standardFields.quotation_name,
                        standardFields.quotation_type || null,
                        standardFields.quotation_template || null,
                        standardFields.deal_id || null,
                        standardFields.account_id || null,
                        standardFields.contact_id || null,
                        standardFields.total_price || 0,
                        customFields.length > 0 ? JSON.stringify(customFields) : null,
                        userId,
                        id
                    ]
                );

                if (result.affectedRows === 0) {
                    throw new Error('Quotation not found');
                }

                //await db.execute('COMMIT');
                return true;

            } catch (error) {
                //await db.execute('ROLLBACK');
                throw error;
            }
        } catch (error) {
            throw new Error(`Failed to update quotation: ${error.message}`);
        }
    }

static async delete(id, userId) {
        try {
           // await db.execute('START TRANSACTION');

            try {
                // Update quotation status to inactive
                const [result] = await db.execute(
                    `UPDATE quotations 
                     SET active = 0,
                         deleted_by = ?,
                         deleted_at = CURRENT_TIMESTAMP
                     WHERE id = ? AND active = 1`,
                    [userId, id]
                );

                if (result.affectedRows === 0) {
                    throw new Error('Quotation not found or already deleted');
                }

                //await db.execute('COMMIT');
                return true;

            } catch (error) {
                //await db.execute('ROLLBACK');
                throw error;
            }

        } catch (error) {
            throw new Error(`Failed to delete quotation: ${error.message}`);
        }
    }
    static async manageQuoteProducts(quoteId, products, userId) {
        try {
            //await db.execute('START TRANSACTION');

            try {
                // Validate quote exists
                const [quote] = await db.execute(
                    'SELECT * FROM quotations WHERE id = ? AND active = 1',
                    [quoteId]
                );

                if (!quote.length) {
                    throw new Error('Quote not found');
                }

                let totalPrice = 0;
                let totalItems = 0;

                // Process each product
                for (const product of products) {
                    const totalAmountProduct = product.unitprice * product.quantity;

                    if (product.id) {
                        // Update existing product
                        await db.execute(
                            `UPDATE quoteproducts SET 
                             product_id = ?,
                             unitprice = ?,
                             quantity = ?,
                             discount_type = ?,
                             discount = ?,
                             totalprice = ?,
                             setupfree = ?,
                             billingcycle = ?,
                             no_of_billing_cycle = ?,
                             updated_at = CURRENT_TIMESTAMP
                             WHERE id = ? AND quotations_id = ?`,
                            [
                                product.product_id,
                                product.unitprice,
                                product.quantity,
                                product.discount_type,
                                product.discount,
                                totalAmountProduct,
                                product.setupfee || null,
                                product.billingcycle || null,
                                product.no_of_billing_cycle || null,
                                product.id,
                                quoteId
                            ]
                        );
                    } else {
                        // Add new product
                        await db.execute(
                            `INSERT INTO quoteproducts 
                             (quotations_id, product_id, unitprice, quantity,
                              discount_type, discount, totalprice, setupfree,
                              billingcycle, no_of_billing_cycle, userid, created_at)
                             VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, CURRENT_TIMESTAMP)`,
                            [
                                quoteId,
                                product.product_id,
                                product.unitprice,
                                product.quantity,
                                product.discount_type,
                                product.discount,
                                totalAmountProduct,
                                product.setupfee || null,
                                product.billingcycle || null,
                                product.no_of_billing_cycle || null,
                                userId
                            ]
                        );
                    }

                    totalPrice += totalAmountProduct;
                    totalItems++;
                }

                // Get tax settings
                const [priceSettings] = await db.execute(
                    'SELECT sales_tax FROM pricesettings LIMIT 1'
                );

                const [defaultCurrency] = await db.execute(
                    'SELECT * FROM currencies WHERE is_default = 1 AND active = 1 LIMIT 1'
                );

                const tax = priceSettings[0]?.sales_tax || 0;
                const taxAmount = (totalPrice * tax) / 100;
                const totalAmount = totalPrice + taxAmount;

                // Update quote totals
                await db.execute(
                    `UPDATE quotations SET 
                     total_items = ?,
                     total_amount = ?,
                     gst = ?,
                     gst_amount = ?,
                     total_price = ?,
                     currency = ?,
                     updated_at = CURRENT_TIMESTAMP
                     WHERE id = ?`,
                    [
                        totalItems,
                        totalPrice.toFixed(defaultCurrency[0].decimal),
                        tax,
                        taxAmount.toFixed(defaultCurrency[0].decimal),
                        totalAmount.toFixed(defaultCurrency[0].decimal),
                        defaultCurrency[0].currency_code,
                        quoteId
                    ]
                );

                //await db.execute('COMMIT');
                return { totalItems, totalAmount };

            } catch (error) {
                //await db.execute('ROLLBACK');
                throw error;
            }
        } catch (error) {
            throw new Error(`Failed to manage quote products: ${error.message}`);
        }
    }
    static async deleteQuoteProduct(quoteProductId, quoteId, userId) {
        try {
            //await db.execute('START TRANSACTION');

            try {
                // Delete quote product
                const [result] = await db.execute(
                    'DELETE FROM quoteproducts WHERE id = ? AND quotations_id = ?',
                    [quoteProductId, quoteId]
                );

                if (result.affectedRows === 0) {
                    throw new Error('Quote product not found');
                }

                // Recalculate quote totals
                const [products] = await db.execute(
                    'SELECT SUM(totalprice) as total_price, COUNT(*) as total_items FROM quoteproducts WHERE quotations_id = ?',
                    [quoteId]
                );

                const totalPrice = products[0].total_price || 0;
                const totalItems = products[0].total_items || 0;

                // Get tax settings
                const [priceSettings] = await db.execute(
                    'SELECT sales_tax FROM pricesettings LIMIT 1'
                );

                const [defaultCurrency] = await db.execute(
                    'SELECT * FROM currencies WHERE is_default = 1 AND active = 1 LIMIT 1'
                );

                const tax = priceSettings[0]?.sales_tax || 0;
                const taxAmount = (totalPrice * tax) / 100;
                const totalAmount = totalPrice + taxAmount;

                // Update quote totals
                await db.execute(
                    `UPDATE quotations SET 
                     total_items = ?,
                     total_amount = ?,
                     gst = ?,
                     gst_amount = ?,
                     total_price = ?,
                     currency = ?,
                     updated_at = CURRENT_TIMESTAMP,
                     updated_by = ?
                     WHERE id = ?`,
                    [
                        totalItems,
                        totalPrice.toFixed(defaultCurrency[0].decimal),
                        tax,
                        taxAmount.toFixed(defaultCurrency[0].decimal),
                        totalAmount.toFixed(defaultCurrency[0].decimal),
                        defaultCurrency[0].currency_code,
                        userId,
                        quoteId
                    ]
                );

                //await db.execute('COMMIT');
                return { totalItems, totalAmount };

            } catch (error) {
                //await db.execute('ROLLBACK');
                throw error;
            }

        } catch (error) {
            throw new Error(`Failed to delete quote product: ${error.message}`);
        }
    }
    static async updateStage(quoteId, stageData) {
        try {
            //await db.execute('START TRANSACTION');

            try {
                // Check if quote exists
                const [quote] = await db.execute(
                    'SELECT * FROM quotations WHERE id = ? AND active = 1',
                    [quoteId]
                );

                if (!quote.length) {
                    throw new Error('Quote not found');
                }

                // Update quote stage
                const [result] = await db.execute(
                    `UPDATE quotations 
                     SET quotation_stage = ?,
                         updated_at = CURRENT_TIMESTAMP,
                         stage_updated_at = CURRENT_TIMESTAMP
                     WHERE id = ? AND active = 1`,
                    [stageData.quotestage, quoteId]
                );

                if (result.affectedRows === 0) {
                    throw new Error('Quote stage update failed');
                }

                //await db.execute('COMMIT');
                return true;

            } catch (error) {
                //await db.execute('ROLLBACK');
                throw error;
            }

        } catch (error) {
            throw new Error(`Failed to update quote stage: ${error.message}`);
        }
    }
    static async generateAndSharePDF(quotationId) {
        try {
            // Get quotation details
            const [quotations] = await db.execute(
                `SELECT q.*, 
                        a.name as account_name,
                        c.first_name as contact_name,
                        d.name as deal_name,
                        u.name as owner_name,
                        DATE_FORMAT(q.create_at, '%Y-%m-%d') as formatted_create_at
                 FROM quotations q
                 LEFT JOIN accounts a ON q.account_id = a.id
                 LEFT JOIN contacts c ON q.contact_id = c.id
                 LEFT JOIN deals d ON q.deal_id = d.id
                 LEFT JOIN users u ON q.users_id = u.id
                 WHERE q.id = ? AND q.active = 1`,
                [quotationId]
            );

            if (!quotations.length) {
                throw new Error('Quotation not found');
            }

            const quotation = quotations[0];

            // Check if PDF already exists and is accessible
            if (quotation.pdf_path) {
                const pdfPath = path.join(process.cwd(), 'public', quotation.pdf_path);
                if (fs.existsSync(pdfPath)) {
                    return { pdf_url: quotation.pdf_path };
                }
            }

            // Process fields
            const standardFields = {
                quotation_name: quotation.quotation_name,
                quotation_type: quotation.quotation_type,
                quotation_template: quotation.quotation_template,
                deal_id: quotation.deal_id,
                account_id: quotation.account_id,
                contact_id: quotation.contact_id,
                total_price: quotation.total_price
            };

            const customFields = quotation.custom_field ? JSON.parse(quotation.custom_field) : [];

            // Generate PDF using class method
            await Quotation.generateQuotationPDF(quotationId, standardFields, customFields);

            // Get updated PDF path
            const [updatedQuotation] = await db.execute(
                'SELECT pdf_path FROM quotations WHERE id = ?',
                [quotationId]
            );

            if (!updatedQuotation.length || !updatedQuotation[0].pdf_path) {
                throw new Error('Failed to generate PDF');
            }

            return { pdf_url: updatedQuotation[0].pdf_path };

        } catch (error) {
            throw new Error(`Failed to generate and share PDF: ${error.message}`);
        }
    }
     static async generateQuotationPDF(quotationId, standardFields, customFields) {
        try {
console.log("quote id"+quotationId);
            // Create PDF document with better margins
            const doc = new PDFDocument({
                size: 'A4',
                margins: {
                    top: 50,
                    bottom: 50,
                    left: 50,
                    right: 50
                }
            });

            // Create quotations directory if it doesn't exist
            const quotationsDir = path.join(process.cwd(), 'public', 'quotations');
            if (!fs.existsSync(quotationsDir)) {
                fs.mkdirSync(quotationsDir, { recursive: true });
            }

            // Generate unique filename
            const filename = `quotation_${quotationId}_${moment().format('YYYYMMDD_HHmmss')}.pdf`;
            const filepath = path.join(quotationsDir, filename);

            // Pipe PDF to file
            doc.pipe(fs.createWriteStream(filepath));

            // Add company logo
            const logoPath = path.join(process.cwd(), 'public', 'images', 'logo.png');
            if (fs.existsSync(logoPath)) {
                doc.image(logoPath, 50, 45, { width: 150 })
                   .moveDown(2);
            }

            // Add quotation header with proper spacing
            doc.fontSize(20)
               .text('QUOTATION', { align: 'center' })
               .moveDown(2);

            // Add quotation details with right alignment
            doc.fontSize(12)
               .text(`Quotation No: ${standardFields.quotation_no || 'N/A'}`, { align: 'right' })
               .text(`Date: ${moment().format('DD/MM/YYYY')}`, { align: 'right' })
               .moveDown(2);

            // Create two-column layout for company and client details
            const startY = doc.y;

            // Left column - Bill To
            if (standardFields.account_id) {
                const [account] = await db.execute(
                    'SELECT * FROM accounts WHERE id = ?',
                    [standardFields.account_id]
                );
                if (account.length) {
                    doc.fontSize(14)
                       .text('Bill To:', 50, startY, { underline: true })
                       .fontSize(12)
                       .text(account[0].name, 50)
                       .text(account[0].address || '')
                       .text(account[0].city || '')
                       .text(account[0].country || '');
                }
            }

            // Right column - Contact Details
            if (standardFields.contact_id) {
                const [contact] = await db.execute(
                    'SELECT * FROM contacts WHERE id = ?',
                    [standardFields.contact_id]
                );
                if (contact.length) {
                    doc.fontSize(14)
                       .text('Contact Person:', 300, startY, { underline: true })
                       .fontSize(12)
                       .text(`${contact[0].first_name} ${contact[0].last_name}`, 300)
                       .text(contact[0].email || '')
                       .text(contact[0].phone || '');
                }
            }

            // Move down after the two columns
            doc.moveDown(3);

            // Add items table with proper alignment
            doc.fontSize(14)
               .text('Quotation Items:', { underline: true })
               .moveDown();

            // Get quotation items
            const [items] = await db.execute(
                `SELECT qp.*, p.product_name 
                 FROM quoteproducts qp 
                 LEFT JOIN products p ON qp.product_id = p.id 
                 WHERE qp.quotations_id = ?`,
                [quotationId]
            );

            // Define table layout
            const tableTop = doc.y + 20;
            const tableHeaders = {
                item: { x: 50, width: 150 },
                quantity: { x: 200, width: 80 },
                price: { x: 280, width: 100 },
                discount: { x: 380, width: 80 },
                total: { x: 460, width: 100 }
            };

            // Add table headers
            doc.fontSize(10)
               .text('Item', tableHeaders.item.x, tableTop)
               .text('Quantity', tableHeaders.quantity.x, tableTop)
               .text('Unit Price', tableHeaders.price.x, tableTop)
               .text('Discount', tableHeaders.discount.x, tableTop)
               .text('Total', tableHeaders.total.x, tableTop);

            // Add horizontal line
            doc.moveTo(50, tableTop + 15)
               .lineTo(560, tableTop + 15)
               .stroke();

            // Add items
            let y = tableTop + 30;
            items.forEach(item => {
                console.log("item name"+item.product_name);
                const unitPrice = parseFloat(item.unitprice || 0);
                const quantity = parseInt(item.quantity || 0);
                const discount = parseFloat(item.discount || 0);
                const total = (unitPrice * quantity) - discount;

                doc.fontSize(10)
                   .text(item.product_name || 'N/A', tableHeaders.item.x, y)
                   .text(quantity.toString(), tableHeaders.quantity.x, y)
                   .text(unitPrice.toFixed(2), tableHeaders.price.x, y)
                   .text(discount.toFixed(2), tableHeaders.discount.x, y)
                   .text(total.toFixed(2), tableHeaders.total.x, y);

                y += 20;
            });

            // Add bottom line
            doc.moveTo(50, y)
               .lineTo(560, y)
               .stroke();

            // Add total section with right alignment
            y += 20;
            const totalPrice = parseFloat(standardFields.total_price || 0);
            doc.fontSize(12)
               .text('Total Amount:', tableHeaders.discount.x, y)
               .text(totalPrice.toFixed(2), tableHeaders.total.x, y);

            // Add terms and conditions at the bottom
            doc.moveDown(4)
               .fontSize(14)
               .text('Terms and Conditions:', 50, doc.y, { underline: true })
               .moveDown()
               .fontSize(10);

            // Define terms array
            const terms = [
                'This quotation is valid for 30 days from the date of issue.',
                'Payment terms: 50% advance payment and balance before delivery.',
                'Prices are exclusive of all applicable taxes.',
                'Delivery timeline will be confirmed after receipt of purchase order.',
                'Installation and training charges are included unless specified otherwise.'
            ];

            // Add terms with proper alignment and bullet points
            terms.forEach((term, index) => {
                doc.text(`${index + 1}. ${term}`, {
                    width: 500,
                    align: 'left',
                    indent: 20,
                    paragraphGap: 5,
                    continued: false
                });
            });

            // Add footer
            doc.moveDown(2)
               .fontSize(8)
               .text('For any queries, please contact our sales team.', {
                   align: 'center',
                   color: 'grey'
               });

            // Finalize PDF
            doc.end();

            // Update quotation with PDF path
            await db.execute(
                'UPDATE quotations SET pdf_path = ? WHERE id = ?',
                [`/quotations/${filename}`, quotationId]
            );

        } catch (error) {
            console.error('Error generating PDF:', error);
            throw new Error(`Failed to generate quotation PDF: ${error.message}`);
        }
    }
    static async getQuoteProducts(quotationId) {
        try {
            // Get quotation products with product details
            const [products] = await db.execute(
                `SELECT qp.*,
                        p.product_name,
                        p.product_code,
                        p.pricing_type
                 FROM quoteproducts qp
                 LEFT JOIN products p ON qp.product_id = p.id
                 WHERE qp.quotations_id = ?
                 ORDER BY qp.created_at`,
                [quotationId]
            );

            // Calculate totals
            let subtotal = 0;
            const formattedProducts = products.map(product => {
                const total = (product.unitprice * product.quantity) - (product.discount || 0);
                subtotal += total;
                
                return {
                    id: product.id,
                    product_id: product.product_id,
                    product_name: product.product_name,
                    product_code: product.product_code,
                    description: product.description,
                    unitprice: product.unitprice,
                    quantity: product.quantity,
                    discount_type: product.discount_type,
                    discount: product.discount || 0,
                    totalprice: total,
                    setupfee: product.setupfree,
                    billingcycle: product.billingcycle,
                    no_of_billing_cycle: product.no_of_billing_cycle,
                    pricing_type: product.pricing_type
                };
            });

            // Get tax settings
            const [priceSettings] = await db.execute(
                'SELECT sales_tax FROM pricesettings LIMIT 1'
            );

            const tax = priceSettings[0]?.sales_tax || 0;
            const taxAmount = (subtotal * tax) / 100;
            const total = subtotal + taxAmount;

            return {
                products: formattedProducts,
                summary: {
                    subtotal: subtotal.toFixed(2),
                    tax_rate: tax,
                    tax_amount: taxAmount.toFixed(2),
                    total: total.toFixed(2)
                }
            };

        } catch (error) {
            throw new Error(`Failed to get quotation products: ${error.message}`);
        }
    }

    

}
module.exports = Quotation;