const db = require('../config/db');

async function create(data) {
    const { field_label, field_name, field_type, custom_field } = data;
    const [result] = await db.execute(
        'INSERT INTO quotationfields (field_label, field_name, field_type, custom_field) VALUES (?, ?, ?, ?)',
        [field_label, field_name, field_type, custom_field || 0]
    );
    return result.insertId;
}

async function findAll(userId) {
    const connection = await db.getConnection();
    try {
        // Get user's role and access scope
        const [userRole] = await connection.execute(
            `SELECT r.access_scope, r.id as role_id, u.territory_id 
             FROM users u 
             JOIN roles r ON u.roles_id = r.id 
             WHERE u.id = ?`,
            [userId]
        );

        if (!userRole || !userRole.length) {
            throw new Error('User role not found');
        }

        const accessScope = JSON.parse(userRole[0].access_scope);
        const territories = userRole[0].territory_id ? userRole[0].territory_id.split(',') : [];

        let [rows] = await db.execute(`SELECT * FROM quotationfields WHERE active = 1 ORDER BY \`order\``);
        const [choices] = await db.execute('SELECT * FROM quotationfieldchoices');

        // Process each field
        const processedRows = await Promise.all(rows.map(async field => {
            // Handle regular choices
            field.choices = choices.filter(choice => choice.quotationfield_id === field.id);
            field.field_value = null;

            // Handle lookup type fields
            if (field.field_type === 'lookup' && field.lookup_type && field.lookup_column) {
                try {
                    let query = `SELECT id, ${field.lookup_column} FROM ${field.lookup_type} WHERE active = 1`;
                    const params = [];

                    // Get view permission for the specific module
                    const modulePermission = field.lookup_type === 'contacts' ? accessScope?.contact?.view :
                                           field.lookup_type === 'accounts' ? accessScope?.sales_account?.view :
                                           field.lookup_type === 'deals' ? accessScope?.deal?.view : 'global';

                    switch (modulePermission) {
                        case 'owned':
                            query += ' AND owner_id = ?';
                            params.push(userId);
                            break;

                        case 'territory':
                            if (territories.length) {
                                const placeholders = territories.map(() => '?').join(',');
                                query += ` AND territory_id IN (${placeholders})`;
                                params.push(...territories);
                            } else {
                                field.choices = [];
                                return field;
                            }
                            break;

                        case 'restricted':
                            field.choices = [];
                            return field;

                        case 'global':
                            // No additional filters needed
                            break;
                    }

                    // Execute filtered query
                    const [lookupData] = await connection.execute(query, params);

                    // Format lookup data as choices
                    field.choices = lookupData.map(item => ({
                        id: item.id,
                        quotationfield_id: field.id,
                        value: item[field.lookup_column],
                        custom_option: 0
                    }));
                } catch (error) {
                    console.error(`Error fetching lookup data for field ${field.field_name}:`, error);
                    field.choices = [];
                }
            }

            return field;
        }));

        return processedRows;

    } catch (error) {
        console.error('Error in findAll:', error);
        throw error;
    } finally {
        connection.release();
    }
}

async function findById(id) {
    let [rows] = await db.execute('SELECT * FROM quotationfields WHERE id = ? AND active = 1', [id]);
    const [choices] = await db.execute('SELECT * FROM quotationfieldchoices');

    if (rows.length === 0) {
        return null;
    }

    const field = rows[0];
    field.choices = choices.filter(choice => choice.quotationfield_id === field.id);
    field.field_value = null;

    if (field.field_type === 'lookup' && field.lookup_type && field.lookup_column) {
        try {
            const [lookupData] = await db.execute(
                `SELECT id, ${field.lookup_column} FROM ${field.lookup_type} WHERE active = 1`
            );

            field.choices = lookupData.map(item => ({
                id: item.id,
                quotationfield_id: field.id,
                value: item[field.lookup_column],
                custom_option: 0
            }));
        } catch (error) {
            console.error(`Error fetching lookup data for field ${field.field_name}:`, error);
            field.choices = [];
        }
    }

    return field;
}

async function update(id, data) {
    const { field_label, field_name, field_type, custom_field } = data;
    const [result] = await db.execute(
        'UPDATE quotationfields SET field_label = ?, field_name = ?, field_type = ?, custom_field = ? WHERE id = ?',
        [field_label, field_name, field_type, custom_field || 0, id]
    );
    return result.affectedRows > 0;
}

async function deleteRecord(id) {
    const [result] = await db.execute('UPDATE quotationfields SET active = 0 WHERE id = ?', [id]);
    return result.affectedRows > 0;
}
async function addQuotation(fields, userId) {
    try {
        //await db.execute('START TRANSACTION');

        try {
            // Get last quotation number
            const [lastQuotation] = await db.execute(
                'SELECT quotation_no FROM quotations ORDER BY quotation_no DESC LIMIT 1'
            );

            // Generate new quotation number
            let quotationNo = 1;
        if (lastQuotation.length) {
            // Ensure numeric conversion
            const lastNumber = Number(lastQuotation[0].quotation_no);
            if (!isNaN(lastNumber)) {
                quotationNo = lastNumber + 1;
            }
        }

            // Process standard and custom fields
            const standardFields = {};
            const customFields = [];

            fields.forEach(field => {
                if (field.field_value !== null && field.field_value !== undefined) {
                    if (field.custom_field === 0) {
                        standardFields[field.field_name] = field.field_value;
                    } else {
                        customFields.push({
                            [field.field_name]: field.field_value
                        });
                    }
                }
            });

            // Insert quotation with standard fields
            const [result] = await db.execute(
                `INSERT INTO quotations 
                 (quotation_no, quotation_date, quotation_name, quotation_type,
                  quotation_template, deal_id, account_id, contact_id, 
                  total_price, custom_field, users_id, create_at, active)
                 VALUES (?, CURRENT_DATE, ?, ?, ?, ?, ?, ?, ?, ?, ?, CURRENT_TIMESTAMP, 1)`,
                [
                    quotationNo,
                    standardFields.quotation_name,
                    standardFields.quotation_type || null,
                    standardFields.quotation_template || null,
                    standardFields.deal_id || null,
                    standardFields.account_id || null,
                    standardFields.contact_id || null,
                    standardFields.total_price || 0,
                    customFields.length > 0 ? JSON.stringify(customFields) : null,
                    userId
                ]
            );

            //await db.execute('COMMIT');

            return {
                id: result.insertId,
                quotation_no: quotationNo
            };

        } catch (error) {
            //await db.execute('ROLLBACK');
            throw error;
        }

    } catch (error) {
        console.error('Error in addQuotation:', error);
        throw new Error(`Failed to create quotation: ${error.message}`);
    }
}

async function order(fields) {
    try {
        for (const field of fields) {
            await db.execute(
                `UPDATE quotationfields
                    SET \`order\` = ?, 
                        updated_at = NOW() 
                    WHERE id = ?`,
                [field.order, field.id]
            );
        }

        return {
            status: 200,
            success: true,
            message: 'Field order updated successfully'
        };
    } catch (error) {
        console.error('Error updating field order:', error);
        throw new Error(`Failed to update field order: ${error.message}`);
    }
}

module.exports = {
    create,
    findAll,
    findById,
    update,
    deleteRecord,
    addQuotation,
    order
};