const db = require('../config/db');

// owner list
async function getOwnerList() {
    const [rows] = await db.execute('SELECT id, name FROM users');
    return rows;
}


// deals report
async function getDealsReport(owner_id = null, start_date = null, end_date = null) {
    try {
        let query = `
            SELECT 
                d.*,
                u.name as owner_name,
                s.deal_stage as stage_name,
                p.pipeline_name as pipeline_name,
                c.first_name as contact_name,
                c.sales_accounts as contact_accounts
            FROM deals d
            LEFT JOIN users u ON d.owner_id = u.id
            LEFT JOIN dealstages s ON d.deal_stage_id = s.id
            LEFT JOIN dealpipelines p ON d.deal_pipeline_id = p.id
            LEFT JOIN contacts c ON d.contacts = c.id`;

        const params = [];

        // Add owner filter if provided
        if (owner_id) {
            query += ' AND d.owner_id = ?';
            params.push(owner_id);
        }

        // Add date range filter if provided
        if (start_date && end_date) {
            query += ' AND DATE(d.created_at) BETWEEN ? AND ?';
            params.push(start_date, end_date);
        }

        // Add ordering
        query += ' ORDER BY d.created_at DESC';

        const [deals] = await db.execute(query, params);

        // Process the deals data
        return deals.map(deal => {
            // Parse contact accounts if available
            let contactAccounts = [];
            if (deal.contact_accounts) {
                try {
                    contactAccounts = JSON.parse(deal.contact_accounts);
                } catch (e) {
                    console.error('Error parsing contact accounts:', e);
                    contactAccounts = [];
                }
            }

            // Parse attendees if available
            let attendees = [];
            if (deal.attendees) {
                try {
                    attendees = deal.attendees.split(',').map(id => parseInt(id));
                } catch (e) {
                    console.error('Error parsing attendees:', e);
                    attendees = [];
                }
            }

            return {
                id: deal.id,
                name: deal.name,
                amount: deal.amount,
                owner: {
                    id: deal.owner_id,
                    name: deal.owner_name
                },
                contact: {
                    id: deal.contacts,
                    name: deal.contact_name,
                    accounts: contactAccounts
                },
                stage: {
                    id: deal.deal_stage_id,
                    name: deal.stage_name
                },
                pipeline: {
                    id: deal.deal_pipeline_id,
                    name: deal.pipeline_name
                },
                status: deal.status,
                expected_close_date: deal.expected_close_date,
                attendees: attendees,
                created_at: deal.created_at,
                updated_at: deal.updated_at
            };
        });
    } catch (error) {
        console.error('Error in getDealsReport:', error);
        throw new Error('Error fetching deals report');
    }
} 

// conveyance report
async function getConveyanceReport(owner_id = null, start_date = null, end_date = null) {
    try {
        let query = `
            SELECT 
                sad.*,
                u.name as owner_name,
                cu.name as updater_name
            FROM salesactivitydatas sad
            LEFT JOIN users u ON sad.owner_id = u.id
            LEFT JOIN users cu ON sad.updater_id = cu.id
            WHERE sad.salesactivities_id IN (3, 10)
            AND sad.status = 'Closed'`;

        const params = [];

        // Add owner filter if provided
        if (owner_id) {
            query += ' AND sad.owner_id = ?';
            params.push(owner_id);
        }

        // Add date range filter if provided
        if (start_date && end_date) {
            query += ' AND DATE(sad.start_date) BETWEEN ? AND ?';
            params.push(start_date, end_date);
        }

        // Add ordering
        query += ' ORDER BY sad.start_date DESC, u.name ASC';

        const [activities] = await db.execute(query, params);

        // Group activities by date
        const groupedActivities = activities.reduce((acc, activity) => {
            const date = new Date(activity.start_date);
            const formattedDate = date.toLocaleDateString('en-GB', {
                day: '2-digit',
                month: '2-digit',
                year: 'numeric'
            });

            if (!acc[formattedDate]) {
                acc[formattedDate] = {
                    conveyance_date: formattedDate,
                    distance: "0.00",
                    charge: 0,
                    activities: []
                };
            }

            acc[formattedDate].activities.push({
                id: activity.id,
                title: activity.title,
                check_in_time: activity.checkin_time ? new Date(activity.checkin_time).toLocaleTimeString('en-US', {
                    hour: '2-digit',
                    minute: '2-digit',
                    hour12: true
                }) : null,
                check_in_location: activity.checkin_location,
                check_out_time: activity.checkout_time ? new Date(activity.checkout_time).toLocaleTimeString('en-US', {
                    hour: '2-digit',
                    minute: '2-digit',
                    hour12: true
                }) : null,
                check_out_location: activity.checkout_location,
                status: activity.status,
                time_spent: activity.timespent_display,
                owner_name: activity.owner_name,
                updater_name: activity.updater_name,
                distance: activity.distance || 0,
                claim_amount: activity.claim_amount || 0,
                transport_mode: activity.transport_mode
            });

            // Sum up distance and charge
            if (activity.distance) {
                acc[formattedDate].distance = (parseFloat(acc[formattedDate].distance) + parseFloat(activity.distance)).toFixed(2);
            }
            if (activity.claim_amount) {
                acc[formattedDate].charge += parseFloat(activity.claim_amount);
            }

            return acc;
        }, {});

        // Calculate totals
        const totals = {
            total_distance: Object.values(groupedActivities).reduce((sum, day) => sum + parseFloat(day.distance), 0).toFixed(2),
            total_charge: Object.values(groupedActivities).reduce((sum, day) => sum + day.charge, 0).toFixed(2)
        };

        return {
            conveyance_data: Object.values(groupedActivities).sort((a, b) => {
                const dateA = a.conveyance_date.split('/').reverse().join('-');
                const dateB = b.conveyance_date.split('/').reverse().join('-');
                return new Date(dateA) - new Date(dateB);
            }),
            totals
        };
    } catch (error) {
        console.error('Error in getConveyanceReport:', error);
        throw new Error('Error fetching conveyance report');
    }
}

// claim report
async function getClaimReport(owner_id = null, start_date = null, end_date = null) {
    try {
        let query = `
            SELECT 
                sad.*,
                u.name as owner_name,
                cu.name as updater_name
            FROM salesactivitydatas sad
            LEFT JOIN users u ON sad.owner_id = u.id
            LEFT JOIN users cu ON sad.updater_id = cu.id
            WHERE sad.salesactivities_id IN (3, 10)
            AND sad.claim_status = 'Completed'`;

        const params = [];

        // Add owner filter if provided
        if (owner_id) {
            query += ' AND sad.owner_id = ?';
            params.push(owner_id);
        }

        // Add date range filter if provided
        if (start_date && end_date) {
            query += ' AND DATE(sad.start_date) BETWEEN ? AND ?';
            params.push(start_date, end_date);
        }

        // Add ordering
        query += ' ORDER BY sad.start_date DESC, u.name ASC';

        const [claims] = await db.execute(query, params);

        // Process the claims data
        const processedClaims = claims.map(claim => ({
            id: claim.id,
            title: claim.title,
            date: new Date(claim.start_date).toLocaleDateString('en-GB', {
                day: '2-digit',
                month: '2-digit',
                year: 'numeric'
            }),
            owner: {
                id: claim.owner_id,
                name: claim.owner_name
            },
            updater: {
                id: claim.updater_id,
                name: claim.updater_name
            },
            check_in: {
                time: claim.checkin_time ? new Date(claim.checkin_time).toLocaleTimeString('en-US', {
                    hour: '2-digit',
                    minute: '2-digit',
                    hour12: true
                }) : null,
                location: claim.checkin_location
            },
            check_out: {
                time: claim.checkout_time ? new Date(claim.checkout_time).toLocaleTimeString('en-US', {
                    hour: '2-digit',
                    minute: '2-digit',
                    hour12: true
                }) : null,
                location: claim.checkout_location
            },
            status: claim.status,
            time_spent: claim.timespent_display,
            distance: claim.distance || 0,
            claim_amount: claim.claim_amount || 0,
            transport_mode: claim.transport_mode,
            claim_status: claim.claim_status,
            claim_image: claim.claim_image
        }));

        // Calculate totals
        const totals = {
            total_claims: processedClaims.length,
            total_amount: processedClaims.reduce((sum, claim) => sum + (parseFloat(claim.claim_amount) || 0), 0).toFixed(2),
            total_distance: processedClaims.reduce((sum, claim) => sum + (parseFloat(claim.distance) || 0), 0).toFixed(2)
        };

        return {
            claims: processedClaims,
            totals
        };
    } catch (error) {
        console.error('Error in getClaimReport:', error);
        throw new Error('Error fetching claim report');
    }
}

// attendance report
async function getAttendanceReport(owner_id = null, start_date = null, end_date = null) {
    try {
        let query = `
            SELECT 
                sad.*,
                u.name as owner_name,
                cu.name as updater_name
            FROM salesactivitydatas sad
            LEFT JOIN users u ON sad.owner_id = u.id
            LEFT JOIN users cu ON sad.updater_id = cu.id
            WHERE sad.salesactivities_id = 10`;

        const params = [];

        // Add owner filter if provided
        if (owner_id) {
            query += ' AND sad.owner_id = ?';
            params.push(owner_id);
        }

        // Add date range filter if provided
        if (start_date && end_date) {
            query += ' AND DATE(sad.start_date) BETWEEN ? AND ?';
            params.push(start_date, end_date);
        }

        // Add ordering
        query += ' ORDER BY sad.start_date DESC, u.name ASC';

        const [attendances] = await db.execute(query, params);

        // Group attendances by date
        const groupedAttendances = attendances.reduce((acc, attendance) => {
            const date = new Date(attendance.start_date);
            const formattedDate = date.toLocaleDateString('en-GB', {
                day: '2-digit',
                month: '2-digit',
                year: 'numeric'
            });

            if (!acc[formattedDate]) {
                acc[formattedDate] = {
                    date: formattedDate,
                    attendances: []
                };
            }

            acc[formattedDate].attendances.push({
                id: attendance.id,
                title: attendance.title,
                owner: {
                    id: attendance.owner_id,
                    name: attendance.owner_name
                },
                updater: {
                    id: attendance.updater_id,
                    name: attendance.updater_name
                },
                check_in: {
                    time: attendance.checkin_time ? new Date(attendance.checkin_time).toLocaleTimeString('en-US', {
                        hour: '2-digit',
                        minute: '2-digit',
                        hour12: true
                    }) : null,
                    location: attendance.checkin_location
                },
                check_out: {
                    time: attendance.checkout_time ? new Date(attendance.checkout_time).toLocaleTimeString('en-US', {
                        hour: '2-digit',
                        minute: '2-digit',
                        hour12: true
                    }) : null,
                    location: attendance.checkout_location
                },
                status: attendance.status,
                time_spent: attendance.timespent_display,
                work_summary: attendance.work_summary
            });

            return acc;
        }, {});

        // Calculate totals
        const totals = {
            total_days: Object.keys(groupedAttendances).length,
            total_attendances: attendances.length,
            present_count: attendances.filter(a => a.status === 'completed').length,
            absent_count: attendances.filter(a => a.status === 'absent').length
        };

        return {
            attendance_data: Object.values(groupedAttendances).sort((a, b) => {
                const dateA = a.date.split('/').reverse().join('-');
                const dateB = b.date.split('/').reverse().join('-');
                return new Date(dateA) - new Date(dateB);
            }),
            totals
        };
    } catch (error) {
        console.error('Error in getAttendanceReport:', error);
        throw new Error('Error fetching attendance report');
    }
}

module.exports = {
    getDealsReport,
    getConveyanceReport,
    getClaimReport,
    getOwnerList,
    getAttendanceReport
}
