const db = require('../config/db');
const { CONTAINS_LIST } = require('../config/utils');
const dayjs = require('dayjs');


async function create(Data) {
    console.log(Data);
    try {


        const { sales_account: allFields } = Data;
        if (!allFields || !Array.isArray(allFields)) {
            throw new Error('Invalid input data format');
        }

        // Get unique fields from accountfields table
        const [uniqueFields] = await db.execute(
            'SELECT field_name FROM accountsfields WHERE unique_field = "on"'
        );

        // Check for uniqueness constraints
        for (const field of allFields) {
            if (uniqueFields.some(uf => uf.field_name === field.field_name) && field.field_value) {
                // Check if value already exists
                const [existing] = await db.execute(
                    `SELECT id FROM accounts WHERE ${field.field_name} = ? AND active = 1`,
                    [field.field_value]
                );
                console.log("existing",existing);
                if (existing.length > 0) {
                    throw new Error(`Account with ${field.field_name} "${field.field_value}" already exists`);
                }
            }
        }

        // Continue with existing logic
        const custom_fields = allFields
            .filter(field => 
                field.custom_field === 1 && 
                field.field_value !== null && 
                field.field_value !== undefined && 
                field.field_value !== ''
            )
            .map(field => ({ 
                [field.field_name]: field.field_value 
            }));

        const standardFields = allFields
            .filter(field => 
                field.custom_field === 0 && 
                field.field_value !== null && 
                field.field_value !== undefined && 
                field.field_value !== ''
            );

        const fieldNames = standardFields
            .map(field => field.field_name)
            .concat(['custom_field', 'active']);

        const values = [
            ...standardFields.map(field => field.field_value),
            custom_fields.length > 0 ? JSON.stringify(custom_fields) : null,
            1
        ];

        const placeholders = fieldNames.map(() => '?').join(', ');

        const [result] = await db.execute(
            `INSERT INTO accounts (${fieldNames.join(', ')}) VALUES (${placeholders})`,
            values
        );


        return result.insertId;

    } catch (error) {
        console.error('Error creating account:', error);
        throw new Error(`Failed to create account: ${error.message}`);
    } 
}
async function findAll(filter = {}, user_id) {
    const [rows] = await db.execute('SELECT * FROM accounts WHERE active=1 AND owner_id = ? ORDER BY created_at DESC', [user_id]);

    // console.log(rows);
    return rows;
}

async function findById(id) {
    try {
        // Get account with notes and related info
        const query = `
            SELECT 
                ac.*,
                n.id as note_id, n.notes, n.voicenote, n.belongsto_id, n.belongsto,
                n.users_id, n.note_time, n.active as note_active, n.is_deleted,
                n.created_at as note_created_at, n.updated_at as note_updated_at,
                u.name as user_name
            FROM accounts ac
            LEFT JOIN notes n ON n.belongsto_id = ac.id AND n.belongsto = 'accounts' AND n.is_deleted = 0
            LEFT JOIN users u ON u.id = n.users_id
            WHERE ac.id = ?`;

        const [accountRows] = await db.execute(query, [id]);
        if (accountRows.length === 0) return null;
        
        // Get account fields and choices
        const [fields] = await db.execute('SELECT * FROM accountsfields WHERE active=1 ORDER BY `order` ASC');
        const [choices] = await db.execute('SELECT * FROM accountsfieldchoices');
        
        // Format fields with their choices and values
        const accountFields = await Promise.all(fields.map(async field => {
            // Handle regular choices
            field.choices = choices.filter(choice => choice.accountsfield_id === field.id).map(choice => ({
                id: choice.id,
                accountfield_id: choice.accountsfield_id,
                custom_option: choice.custom_option,
                created_at: choice.created_at,
                updated_at: choice.updated_at
            }));

            // Set standard field value
            field.field_value = accountRows[0][field.field_name];
            
            // Set custom field value if applicable
            if (field.custom_field === 1 && accountRows[0].custom_field) {
                const customFields = JSON.parse(accountRows[0].custom_field);
                field.field_value = customFields.find(cf => cf[field.field_name])?.[field.field_name] || null;
            }

            // Handle lookup type fields
            if (field.field_type === 'lookup' && field.lookup_type && field.lookup_column) {
                try {
                    const [lookupData] = await db.execute(
                        `SELECT id, ${field.lookup_column} FROM ${field.lookup_type} WHERE active = 1`
                    );

                    // Format lookup data as choices
                    field.choices = lookupData.map(item => ({
                        id: item.id,
                        accountfield_id: field.id,
                        custom_option: item[field.lookup_column],
                        created_at: new Date().toISOString(),
                        updated_at: new Date().toISOString()
                    }));

                } catch (error) {
                    console.error(`Error fetching lookup data for field ${field.field_name}:`, error);
                    field.choices = [];
                }
            }

            return {
                id: field.id,
                custom_field: field.custom_field,
                field_label: field.field_label,
                field_name: field.field_name,
                field_type: field.field_type,
                required: field.required,
                quick_add: field.quick_add,
                tool_tip: field.tool_tip,
                placeholder: field.placeholder,
                choices: field.choices,
                read_only: field.read_only,
                has_dependent: field.has_dependent,
                accountgroups_id: field.accountgroups_id,
                active: field.active,
                order: field.order,
                lookup_type: field.lookup_type,
                lookup_column: field.lookup_column,
                created_at: field.created_at,
                updated_at: field.updated_at,
                field_value: field.field_value
            };
        }));

        // Get tasks
        const [tasks] = await db.execute(
            'SELECT * FROM tasks WHERE active=1 and related_type = "accounts" AND related_to = ?',
            [id]
        );

        // Get meetings
        const [meetings] = await db.execute(
            'SELECT * FROM salesactivitydatas WHERE active=1 and targetable_type = "accounts" AND targetable_id = ? AND salesactivities_id = 3',
            [id]
        );

        // Get contacts
        const [contacts] = await db.execute(
            'SELECT c.* FROM contacts c JOIN accounts ac ON c.sales_accounts = ac.id WHERE ac.id = ?',
            [id]
        );

        // Get deals summary
        const [deals] = await db.execute(
            `SELECT COUNT(*) as count, 
             SUM(amount) as total_value,
             SUM(CASE WHEN deal_stage_id = 1 THEN 1 ELSE 0 END) as open_count,
             SUM(CASE WHEN deal_stage_id = 1 THEN amount ELSE 0 END) as open_value
             FROM deals WHERE sales_account_id = ?`,
            [id]
        );

        // Process notes from the joined query
        const notes = [];
        accountRows.forEach(row => {
            if (row.note_id && !row.is_deleted) {
                if (!notes.some(note => note.id === row.note_id)) {
                    notes.push({
                        id: row.note_id,
                        text: row.notes,
                        voice_note: row.voicenote
                    });
                }
            }
        });

        // Sort notes by note_time in descending order (newest first)
        notes.sort((a, b) => new Date(b.note_time) - new Date(a.note_time));

        return {
            account: {
                id: id.toString(),
                accounts: accountFields,
                tasks: tasks,
                meetings: meetings,
                sales_account: null,
                notes: notes,
                notes_count: notes.length,
                contacts: contacts,
                deals_summary: {
                    count: deals[0].count || 0,
                    total_value: deals[0].total_value || 0,
                    open_count: deals[0].open_count || 0,
                    open_value: deals[0].open_value || 0
                }
            }
        };
    } catch (error) {
        console.error('Error in findById:', error);
        throw new Error(`Failed to fetch account: ${error.message}`);
    }
}
// Helper function to get activity icons
function getActivityIcon(type) {
    switch(type) {
        case 'Task':
            return 'sales/to-do-list.png';
        case 'Meeting':
            return 'sales/footprint.png';
        case 'Attendance':
            return 'sales/calendaricon.png';
        default:
            return '';
    }
}
async function update(id, Data) {
    try {
        // Check if Data and sales_account exists
        if (!Data || !Data.sales_account) {
            throw new Error('Invalid update data format');
        }

        const allFields = Array.isArray(Data.sales_account) ? 
            Data.sales_account : 
            [];

            const uniqueFields = Array.from(allFields).filter(field => field.unique_field == 1 || field.unique_field == 'on');
            for (const field of allFields) {
                if (uniqueFields.some(uf => uf.field_name === field.field_name) && field.field_value) {
                    console.log("field.field_name",field.field_name, "field.field_value",field.field_value  );
                    // Check if value already exists
                    const [existing] = await db.execute(
                        `SELECT * FROM accounts WHERE ${field.field_name} = ? AND active = 1`,
                        [field.field_value]
                    );
                    // console.log("existing",existing);
                    console.log(typeof id);
                    console.log(typeof existing[0].id)
                    // if ( Array.isArray(existing) && existing.length > 0 && String(existing[0].id).trim() !== String(id).trim())
                    if (existing.length > 0 && Number(existing[0].id) !== Number(id)) {
                        console.log(`Account with ${field.field_name} "${field.field_value}" already exists`);
                        return {
                            status: 409,
                            success: false,
                            message: `Account with ${field.field_name} "${field.field_value}" already exists`,
                        };
                    }
                }
            }

        // Separate custom and standard fields
        const custom_fields = allFields
            .filter(field => field && field.custom_field === 1)
            .map(field => ({ [field.field_name]: field.field_value }));

        const standardFields = allFields
            .filter(field => field && field.custom_field === 0);

        // Prepare update fields and values
        const updateFields = standardFields
            .map(field => `${field.field_name} = ?`)
            .concat(['custom_field = ?'])
            .join(', ');

        const updateValues = [
            ...standardFields.map(field => field.field_value),
            custom_fields.length > 0 ? JSON.stringify(custom_fields) : null,
            id
        ];

        // Execute update query
        const [result] = await db.execute(
            `UPDATE accounts SET ${updateFields} WHERE id = ?`,
            updateValues
        );

        // Return true if update was successful
        return result.affectedRows > 0;

    } catch (error) {
        console.error('Error updating account:', error);
        throw new Error(`Failed to update account: ${error.message}`);
    }
}
async function deleteRecord (id) {
    const [result] = await db.execute('UPDATE accounts SET active = 0 WHERE id = ?', [id]);
    return result.affectedRows > 0;
}
async function findAccountMeetings(accountId) {
    try {
        const [meetings] = await db.execute(
            `SELECT 
                sad.*,
                u.name as owner_name,
               COALESCE(sao.outcome, '') as outcome_name,
                sa.activity_name,
                sa.activity_icon
             FROM salesactivitydatas sad
             LEFT JOIN users u ON sad.owner_id = u.id
             LEFT JOIN salesactivityoutcomes sao ON sad.outcome = sao.id
             LEFT JOIN salesactivities sa ON sad.salesactivities_id = sa.id
             WHERE sad.targetable_type = 'accounts'
             AND sad.targetable_id = ?
             AND sad.salesactivities_id = 3
             AND sad.active = 1
             ORDER BY sad.start_date DESC, sad.start_time DESC`,
            [accountId]
        );

        return meetings.map(meeting => ({
            id: meeting.id,
            title: meeting.title,
            description: meeting.description,
            activity_name: meeting.activity_name,
            activity_icon: meeting.activity_icon,
            activity_title: meeting.activity_title,
            start_date: meeting.start_date,
            end_date: meeting.end_date,
            start_time: formatTime(meeting.start_time),
            end_time: formatTime(meeting.end_time),
            duration: meeting.duration,
            outcome: meeting.outcome_name,
            status: meeting.status,
            priority: meeting.priority,
            location: meeting.location,
            owner_id: meeting.owner_id,
            owner_name: meeting.owner_name,
            owner_email: meeting.owner_email,
            owner_mobile: meeting.owner_mobile,
            salesactivities_id: meeting.salesactivities_id,
            targetable_type: meeting.targetable_type,
            targetable_id: meeting.targetable_id,
            transport_mode: meeting.transport_mode,
            distance: meeting.distance,
            claim_amount: meeting.claim_amount,
            checkin_location: meeting.checkin_location,
            checkin_datetime: formatDateTime(meeting.checkin_datetime),
            checkout_location: meeting.checkout_location,
            checkout_datetime: formatDateTime(meeting.checkout_datetime),
            custom_field: meeting.custom_field ? JSON.parse(meeting.custom_field) : null,
            allow_reschedule: meeting.allow_reschedule,
            allow_complete: meeting.allow_complete,
            active: meeting.active,
            created_at: formatDateTime(meeting.created_at),
            updated_at: formatDateTime(meeting.updated_at)
        }));

    } catch (error) {
        console.error('Error in findAccountMeetings:', error);
        throw new Error(`Failed to fetch account meetings: ${error.message}`);
    }
}

async function findByTasks(dealid) {
    try {
        console.log("id", dealid)
        const [rows] = await db.execute(
            `SELECT t.*, 
                u.name as owner_name,
                COALESCE(t.task_type, 'General') as task_type_name
             FROM tasks t
             LEFT JOIN users u ON t.user_id = u.id
             WHERE t.active = 1
             AND (t.related_type = "accounts" OR t.related_type = "Accounts")
             AND t.related_to = ?
             ORDER BY t.due_date DESC, t.due_time DESC`,
            [dealid]
        );

        return rows.map(task => ({
                id: task.id,
                title: task.title,
                description: task.description,
                task_type: task.task_type_name,
                related_type: task.related_type,
                related_to: task.related_to,
                mobile_number: task.mobile_number,
                due_date: formatDate(task.due_date) ? formatDate(task.due_date).replaceAll('/', '-').split('-').reverse().join('-') : null,
                due_time: formatTime(task.due_time),
                mark_as_complete: task.mark_as_complete,
                completed_date: formatDate(task.completed_date) ? formatDate(task.completed_date).replaceAll('/', '-').split('-').reverse().join('-') : null,
                completed_time: formatTime(task.completed_time),
                collaboratoes: task.collaboratoes,
                owner: task.owner,
                user_id: task.user_id,
                owner_name: task.owner_name || '',
                outcome: task.outcome,
                active: task.active,
                snooze_date: formatDate(task.snooze_date) ? formatDate(task.snooze_date).replaceAll('/', '-').split('-').reverse().join('-') : null,
                snooze_time: formatTime(task.snooze_time),
                created_at: formatDateTime(task.created_at),
                updated_at: formatDateTime(task.updated_at),
                status: task.mark_as_complete ? 'Completed' : 'Pending'
            }))
        ;
    } catch (error) {
        console.error('Detailed error:', error);
        throw new Error(`Failed to fetch account tasks: ${error.message}`);
    }
}

// formet the date 
function formatDate(date) {
    if (!date) return null;
    return new Date(date).toLocaleDateString('en-GB', {
        day: '2-digit',
        month: '2-digit',
        year: 'numeric'
    });
}

function formatTime(time) {
    if (!time) return null;
    return new Date(`1970-01-01T${time}`).toLocaleTimeString('en-US', {
        hour: '2-digit',
        minute: '2-digit',
        hour12: true
    });
}
function formatDateTime(datetime) {
    if (!datetime) return null;
    return new Date(datetime).toISOString();
}

// get account relations
async function getAccountRelations(accountId) {
    try {
        // Get related contacts
        const contactsQuery = `
            SELECT 
                c.id,
                CONCAT(c.first_name, ' ', c.last_name) as name,
                c.emails,
                c.mobile_number as phone,
                c.created_at
            FROM contacts c
            WHERE c.sales_accounts = ? AND c.active = 1
            ORDER BY c.created_at DESC`;

        // Get related deals
        const dealsQuery = `
            SELECT 
                d.id,
                d.name,
                d.amount,
                d.probability,
                d.deal_stage_id,
                ds.deal_stage as stage_name,
                d.created_at,
                CONCAT(c.first_name, ' ', c.last_name) as contact_name
            FROM deals d
            LEFT JOIN dealstages ds ON d.deal_stage_id = ds.id
            LEFT JOIN contacts c ON d.contacts = c.id
            WHERE d.sales_account_id = ? AND d.active = 1
            ORDER BY d.created_at DESC`;

        // Execute both queries
        const [contacts] = await db.execute(contactsQuery, [accountId]);
        const [deals] = await db.execute(dealsQuery, [accountId]);

        // Get deals summary
        const [dealsSummary] = await db.execute(
            `SELECT 
                COUNT(*) as total_deals,
                SUM(amount) as total_value,
                SUM(CASE WHEN deal_stage_id NOT IN (SELECT id FROM dealstages WHERE deal_stage IN ('Won', 'Lost')) 
                    THEN 1 ELSE 0 END) as open_deals,
                SUM(CASE WHEN deal_stage_id NOT IN (SELECT id FROM dealstages WHERE deal_stage IN ('Won', 'Lost')) 
                    THEN amount ELSE 0 END) as open_value,
                SUM(CASE WHEN deal_stage_id IN (SELECT id FROM dealstages WHERE deal_stage = 'Won') 
                    THEN 1 ELSE 0 END) as won_deals,
                SUM(CASE WHEN deal_stage_id IN (SELECT id FROM dealstages WHERE deal_stage = 'Won') 
                    THEN amount ELSE 0 END) as won_value
            FROM deals 
            WHERE sales_account_id = ? AND active = 1`,
            [accountId]
        );

        return {
            contacts: {
                total_contacts: contacts.length,
                contacts: contacts.map(contact => ({
                    id: contact.id,
                    name: contact.name,
                    email: contact.emails,
                    phone: contact.phone,
                    created_at: formatDateTime(contact.created_at)
                }))
            },
            deals: {
                summary: {
                    total_deals: dealsSummary[0].total_deals || 0,
                    total_value: dealsSummary[0].total_value || 0,
                    open_deals: dealsSummary[0].open_deals || 0,
                    open_value: dealsSummary[0].open_value || 0,
                    won_deals: dealsSummary[0].won_deals || 0,
                    won_value: dealsSummary[0].won_value || 0
                },
                deals: deals.map(deal => ({
                    id: deal.id,
                    name: deal.name,
                    amount: deal.amount,
                    probability: deal.probability,
                    expected_revenue: deal.expected_revenue,
                    close_date: formatDate(deal.close_date),
                    stage: {
                        id: deal.deal_stage_id,
                        name: deal.stage_name
                    },
                    contact_name: deal.contact_name,
                    created_at: formatDateTime(deal.created_at)
                }))
            }
        };

    } catch (error) {
        console.error('Error in getAccountRelations:', error);
        throw new Error(`Failed to fetch account relations: ${error.message}`);
    }
}

async function createAccountFilterView(viewData) {
    try {
                // Insert product view
        const [viewResult] = await db.execute(
            `INSERT INTO accountviews 
            (name, share_with, accessibility, users) 
            VALUES (?, ?, ?, ?)`,
            [viewData.title, viewData.share_with, viewData.accessibility, viewData.selected_users.join(',') || null]
        );

        const viewId = viewResult.insertId;

        // Insert filters if any
        if (viewData.filter && Array.isArray(viewData.filter)) {
            for (const filter of viewData.filter) {
                if (filter.field_label && filter.field_label.trim() !== '') {
                    await db.execute(
                        `INSERT INTO accountfilters 
                        (view_id, name, field_id, contains, filter, active) 
                        VALUES (?, ?, ?, ?, ?, ?)`,
                        [
                            viewId,
                            filter.field_label,
                            filter.field_name,
                            filter.contains,
                            JSON.stringify(filter.value_id),
                            1
                        ]
                    );
                }
            }
        }

       

        // Get the created view
        const [views] = await db.execute(
            'SELECT * FROM accountviews WHERE id = ?',
            [viewId]
        );

        return views[0];

    } catch (error) {
        console.error('Error in createAccountView:', error);
        throw new Error(`Failed to create account view: ${error.message}`);
    }
}

async function deleteAccountFilterView(viewId) {
    try {
        //await db.execute('START TRANSACTION');

        // Delete related filters first
        await db.execute(
            'DELETE FROM accountfilters WHERE view_id = ?',
            [viewId]
        );

        // Delete the view
        const [result] = await db.execute(
            'DELETE FROM accountviews WHERE id = ?',
            [viewId]
        );

        //await db.execute('COMMIT');

        return result.affectedRows > 0;

    } catch (error) {
       // await db.execute('ROLLBACK');
        console.error('Error in accountviews:', error);
        throw new Error(`Failed to delete product view: ${error.message}`);
    }
}
async function getAllFiltersList() {
    try {
        console.log("=== Starting getAllFiltersList Model Function ===");
        console.log("Executing database query for accountviews");
        // Get the views with their filters
        const [views] = await db.execute(`
            SELECT cv.*, cf.id as filter_id, cf.name as filter_name, 
                   cf.field_id, cf.contains, cf.filter as filter_value
                            FROM accountviews cv
            LEFT JOIN accountfilters cf ON cv.id = cf.view_id
            `
        );
        
        // Transform the data to match findAll format
        const transformedViews = views.reduce((acc, row) => {
            if (!acc[row.id]) {
                acc[row.id] = {
                    id: row.id,
                    name: row.name,
                    share_with: row.share_with,
                    accessibility: row.accessibility,
                    users: row.users ? row.users.split(',') : [],
                    owner_name: row.owner_name,
                    created_at: formatDateTime(row.created_at),
                    updated_at: formatDateTime(row.updated_at),
                    filters: []
                };
            }
            
            if (row.filter_id) {
                acc[row.id].filters.push({
                    id: row.filter_id,
                    name: row.filter_name,
                    field_id: row.field_id,
                    contains: row.contains,
                    filter: row.filter_value ? JSON.parse(row.filter_value) : [],
                });
            }
            
            return acc;
        }, {});

        // console.log(transformedViews, "transformedViews");
        const result = Object.values(transformedViews);
        // console.log("Number of views found:", result.length);
        return result;
        
    } catch (error) {
        console.error("=== Error in getAllFiltersList Model Function ===");
        console.error("Error details:", error);
        throw new Error(`Failed to fetch account filters: ${error.message}`);
    }
}
async function getFieldsForFilter(userId) {
    try {
        // Validate userId
        if (!userId) {
            throw new Error('User ID is required');
        }

        // Get all product fields
        const [fields] = await db.execute(
            'SELECT * FROM accountsfields WHERE active = 1 ORDER BY id'
        );

        if (!fields || fields.length === 0) {
            throw new Error('No product fields found');
        }

        // Get users for lookup fields
        const [users] = await db.execute(
            'SELECT id, name FROM users WHERE active = 1'
        );

        // Get all field choices
        const [choices] = await db.execute(
            'SELECT * FROM accountsfieldchoices'
        );

        // Process each field
        const processedFields = fields.map(field => {
            let values = [];
            let contains_list = [];

            // Set contains_list based on field type
            switch (field.field_type.toLowerCase()) {
                case 'dropdown':
                    contains_list = CONTAINS_LIST.DROPDOWN;
                    // Add dropdown values from choices with id and custom_option
                    values = choices
                        .filter(choice => choice.productfield_id === field.id)
                        .map(choice => ({
                            id: choice.id,
                            value: choice.value,
                            value: choice.custom_option || 0
                        }));
                    break;

                case 'number':
                    contains_list =CONTAINS_LIST.NUMBERS;
                    break;

                case 'datepicker':
                    contains_list = CONTAINS_LIST.DATE;
                    break;

                case 'lookup':
                    contains_list = CONTAINS_LIST.LOOKUP;
                    // Add lookup values
                    values = [
                        { id: null, value: "unassigned" },
                        ...users.map(user => ({
                            id: user.id,
                            value: user.id === userId ? "Me" : user.name
                        }))
                    ];
                    break;

                default:
                    contains_list = CONTAINS_LIST.DEFAULT;
            }

            return {
                id: field.id,
                custom_field: field.custom_field || 0,
                field_label: field.field_label,
                field_name: field.field_name,
                field_type: field.field_type,
                values: values,
                contains_list: contains_list
            };
        });

        return processedFields;

    } catch (error) {
        console.error('Error in getAccountFields:', error);
        throw new Error(`Failed to fetch saccount fields: ${error.message}`);
    }
}


async function getAccountListWithFilters(filters, userId, role_id) {
    const connection = await db.getConnection();
    try {
        // Step 1: Get user role and access scope
        const [userRows] = await connection.execute(
            `SELECT r.access_scope, r.id as role_id, u.territory_id 
             FROM users u 
             JOIN roles r ON u.roles_id = r.id 
             WHERE u.id = ?`,
            [userId]
        );

        if (!userRows.length) throw new Error('User role not found');
        const { access_scope, territory_id } = userRows[0];
        const { view } = JSON.parse(access_scope)?.sales_account || {};

        let baseQuery = `SELECT p.*, u.name as owner_name 
                         FROM accounts p 
                         LEFT JOIN users u ON p.owner_id = u.id 
                         WHERE p.active = 1`;

        let params = [];

        // Step 2: Apply permission-based filters
        switch (view) {
            case 'owned':
                baseQuery += ' AND p.owner_id = ?';
                params.push(userId);
                break;

            case 'territory':
                if (!territory_id) return [];
                if (role_id === 2) {
                    const territories = territory_id.split(',');
                    const placeholders = territories.map(() => '?').join(',');
                    console.log('Territory IDs:', territories);
                    console.log('Placeholders:', placeholders);
                    baseQuery += ` AND (p.territory_id IN (${placeholders}) OR p.owner_id = ?)`;
                    params.push(...territories);
                    params.push(userId);
                    break;
                } else {
                    const territories = territory_id.split(',');
                    const placeholders = territories.map(() => '?').join(',');
                    console.log('Territory IDs:', territories);
                    console.log('Placeholders:', placeholders);
                    baseQuery += ` AND p.territory_id IN (${placeholders})`;
                    params.push(...territories);
                    break;
                }

            case 'global':
            case 'restricted':
            default:
                if (view === 'restricted') return [];
        }

        // Step 3: Map filter fields
        const fieldMapping = {
            owner: 'p.owner_id',
            created_at: 'p.created_at',
            updated_at: 'p.updated_at',
            deal_stage: 'p.deal_stage_id',
            pipeline: 'p.deal_pipeline_id'
        };

        // Step 4: Apply filters
        if (filters?.length > 0) {
            for (const filter of filters) {
                const { contains, field_name, value_id } = filter;
                const mappedField = fieldMapping[field_name] || `p.${field_name}`;
                baseQuery = applyFilterToQuery({ baseQuery, contains, mappedField, value_id, params });
            }
        }

        baseQuery += ` ORDER BY p.created_at DESC LIMIT 1000`; // Prevent overload

        // Step 5: Execute final query
        const [accounts] = await connection.execute(baseQuery, params);

        return accounts;

    } catch (error) {
        console.error('Error in getAccountListWithFilters:', error);
        throw error;
    } finally {
        connection.release(); // Always release connection
    }
}

// Helper function to avoid repeating condition logic
function applyFilterToQuery({ baseQuery, contains, mappedField, value_id, params }) {
    switch (contains) {
        case 'contains':
            if (Array.isArray(value_id) && value_id.length) {
                const placeholders = value_id.map(() => '?').join(',');
                baseQuery += ` AND ${mappedField} IN (${placeholders})`;
                params.push(...value_id);
            }
            break;

        case 'does not contain':
            if (Array.isArray(value_id) && value_id.length) {
                const placeholders = value_id.map(() => '?').join(',');
                baseQuery += ` AND ${mappedField} NOT IN (${placeholders})`;
                params.push(...value_id);
            }
            break;

        case 'is not empty':
            baseQuery += ` AND ${mappedField} IS NOT NULL AND ${mappedField} != ''`;
            break;

        case 'is empty':
            baseQuery += ` AND (${mappedField} IS NULL OR ${mappedField} = '')`;
            break;

        case '< less than':
            baseQuery += ` AND ${mappedField} < ?`;
            params.push(value_id);
            break;

        case '> greater than':
            baseQuery += ` AND ${mappedField} > ?`;
            params.push(value_id);
            break;

        case '= equal to':
            baseQuery += ` AND ${mappedField} = ?`;
            params.push(value_id);
            break;

        case '!= not equal to':
            baseQuery += ` AND ${mappedField} != ?`;
            params.push(value_id);
            break;

        case 'between':
            if (Array.isArray(value_id) && value_id.length === 2) {
                baseQuery += ` AND ${mappedField} BETWEEN ? AND ?`;
                params.push(value_id[0], value_id[1]);
            }
            break;

        case 'last 30 minutes':
            baseQuery += ` AND ${mappedField} >= ?`;
            params.push(dayjs().subtract(30, 'minute').format('YYYY-MM-DD HH:mm:ss'));
            break;

        case 'last 60 minutes':
            baseQuery += ` AND ${mappedField} >= ?`;
            params.push(dayjs().subtract(60, 'minute').format('YYYY-MM-DD HH:mm:ss'));
            break;

        case 'last 2 hours':
            baseQuery += ` AND ${mappedField} >= ?`;
            params.push(dayjs().subtract(2, 'hour').format('YYYY-MM-DD HH:mm:ss'));
            break;

        case 'last 24 hours':
            baseQuery += ` AND ${mappedField} >= ?`;
            params.push(dayjs().subtract(24, 'hour').format('YYYY-MM-DD HH:mm:ss'));
            break;

        case 'last 7 days':
            baseQuery += ` AND ${mappedField} >= ?`;
            params.push(dayjs().subtract(7, 'day').format('YYYY-MM-DD HH:mm:ss'));
            break;

        case 'last 30 days':
            baseQuery += ` AND ${mappedField} >= ?`;
            params.push(dayjs().subtract(30, 'day').format('YYYY-MM-DD HH:mm:ss'));
            break;

        case 'last 90 days':
            baseQuery += ` AND ${mappedField} >= ?`;
            params.push(dayjs().subtract(90, 'day').format('YYYY-MM-DD HH:mm:ss'));
            break;

        case 'last 12 months':
            baseQuery += ` AND ${mappedField} >= ?`;
            params.push(dayjs().subtract(12, 'month').format('YYYY-MM-DD HH:mm:ss'));
            break;
    }
    return baseQuery;
}



async function updateFilterView(viewData) {
    console.log("updateFilterView inits "+JSON.stringify(viewData));
    try {
                // Insert product view
                const selectedUsers = Array.isArray(viewData.users) && viewData.users.length > 0
                ? viewData.users.join(',')
                : null;
              
              const [viewResult] = await db.execute(
                `UPDATE accountviews 
                 SET name = ?, share_with = ?, accessibility = ?, users = ?
                 WHERE id = ?`,
                [viewData.title, viewData.share_with, viewData.accessibility, selectedUsers, viewData.id]
              );

        // Get the created view
        const [views] = await db.execute(
            'SELECT * FROM accountviews WHERE id = ?',
            [viewData.id]
        );

        return views[0];

    } catch (error) {
        console.error('Error in accountviews:', error);
        throw new Error(`Failed to create account view: ${error.message}`);
    }
}

// Bulk import sales accounts
async function bulkImport(csvData, columnMappings, userId) {
    const connection = await db.getConnection();
    try {
        await connection.beginTransaction();
        console.log("bulkImport inits "+JSON.stringify(csvData));

        const results = {
            total: csvData.length,
            successful: 0,
            failed: 0,
            errors: [],
            errorCsvData: [],
            errorCsvFile: null
        };

        // Pre-fetch validation data
        const [users] = await connection.execute('SELECT id, record_id FROM users WHERE active = 1 AND record_id IS NOT NULL');
        const [accounts] = await connection.execute('SELECT id FROM accounts WHERE active = 1');
        
        // Create lookup maps
        const userMap = new Map();
        users.forEach(user => {
            if (user.record_id) {
                userMap.set(user.record_id.toString(), user.id);
            }
        });
        
        const accountIds = new Set(accounts.map(acc => acc.id.toString()));

        for (let i = 0; i < csvData.length; i++) {
            const row = csvData[i];
            const rowNumber = i + 2; // +2 because CSV has header and arrays are 0-indexed

            try {
                // Prepare account data
                const accountData = {
                    owner_id: userId,
                    created_at: new Date().toISOString().slice(0, 19).replace('T', ' '),
                    updated_at: new Date().toISOString().slice(0, 19).replace('T', ' ')
                };

                // Separate standard fields and custom fields
                const standardFields = [];
                const customFields = [];

                // Process each mapped column
                for (const mapping of columnMappings) {
                    if (mapping.isMapped && mapping.apiField) {
                        const csvValue = row[mapping.csvColumn];
                        const field = mapping.apiField;

                        // Skip if no value provided
                        if (csvValue === undefined || csvValue === null || csvValue === '') {
                            continue;
                        }

                        let processedValue = csvValue.toString().trim();

                        // Handle different field types based on field_name and field_type
                        switch (field.field_type.toLowerCase()) {
                            case 'text field':
                            case 'textarea':
                                processedValue = csvValue.toString().trim();
                                break;

                            case 'number':
                                const numValue = parseFloat(csvValue);
                                if (!isNaN(numValue)) {
                                    processedValue = numValue;
                                }
                                break;

                            case 'datepicker':
                                // Handle various date formats
                                if (csvValue) {
                                    const date = new Date(csvValue);
                                    if (!isNaN(date.getTime())) {
                                        processedValue = date.toISOString().split('T')[0];
                                    }
                                }
                                break;

                            case 'dropdown':
                            case 'multiselect':
                                // For dropdown/multiselect, query the choices table directly
                                if (field.field_name) {
                                    const [choices] = await connection.execute(
                                        'SELECT id, custom_option FROM accountsfieldchoices WHERE accountsfield_id = ? AND custom_option = ?',
                                        [field.id, csvValue.toString().trim()]
                                    );
                                    
                                    if (choices.length > 0) {
                                        processedValue = choices[0].id;
                                    } else {
                                        throw new Error(`Invalid dropdown value: "${csvValue}" for field "${field.field_label}" - Choice not found`);
                                    }
                                }
                                break;

                            case 'lookup':
                                // Handle lookup fields based on field_name
                                if (field.field_name === 'owner_id' || field.field_name === 'created_by' || field.field_name === 'modified_by') {
                                    // For user lookup fields, get user ID from users table based on record_id
                                    const userRecordId = csvValue.toString().trim();
                                    const mappedUserId = userMap.get(userRecordId);
                                    if (!mappedUserId) {
                                        throw new Error(`Invalid ${field.field_name}: "${userRecordId}" - User not found`);
                                    }
                                    processedValue = mappedUserId;
                                } else if (field.field_name === 'parent_account' || field.field_name === 'parent_account_id') {
                                    // For parent account lookup, check if the value looks like a record_id
                                    if (csvValue.toString().includes('zcrm_') || /^\d+$/.test(csvValue.toString())) {
                                        // This looks like a record_id, try to find by record_id first
                                        const [accountResult] = await connection.execute(
                                            'SELECT id FROM accounts WHERE record_id = ? AND active = 1',
                                            [csvValue.toString().trim()]
                                        );
                                        
                                        if (accountResult.length > 0) {
                                            processedValue = accountResult[0].id;
                                        } else {
                                            // If not found by record_id, try by name
                                            const [accountResult2] = await connection.execute(
                                                'SELECT id FROM accounts WHERE name = ? AND active = 1',
                                                [csvValue.toString().trim()]
                                            );
                                            
                                            if (accountResult2.length > 0) {
                                                processedValue = accountResult2[0].id;
                                            } else {
                                                throw new Error(`Invalid parent account: "${csvValue}" - Account not found`);
                                            }
                                        }
                                    } else {
                                        // This looks like a display name, query by name
                                        const [accountResult] = await connection.execute(
                                            'SELECT id FROM accounts WHERE name = ? AND active = 1',
                                            [csvValue.toString().trim()]
                                        );
                                        
                                        if (accountResult.length > 0) {
                                            processedValue = accountResult[0].id;
                                        } else {
                                            throw new Error(`Invalid parent account: "${csvValue}" - Account not found`);
                                        }
                                    }
                                } else if (field.lookup_type && field.lookup_column) {
                                    // For other lookup fields, check if the value looks like a record_id
                                    if (csvValue.toString().includes('zcrm_') || /^\d+$/.test(csvValue.toString())) {
                                        // This looks like a record_id, try to find by record_id first
                                        const [lookupResult] = await connection.execute(
                                            `SELECT id FROM ${field.lookup_type} WHERE record_id = ? AND active = 1`,
                                            [csvValue.toString().trim()]
                                        );
                                        
                                        if (lookupResult.length > 0) {
                                            processedValue = lookupResult[0].id;
                                        } else {
                                            // If not found by record_id, try by the lookup_column
                                            const [lookupResult2] = await connection.execute(
                                                `SELECT id FROM ${field.lookup_type} WHERE ${field.lookup_column} = ? AND active = 1`,
                                                [csvValue.toString().trim()]
                                            );
                                            
                                            if (lookupResult2.length > 0) {
                                                processedValue = lookupResult2[0].id;
                                            } else {
                                                throw new Error(`Invalid ${field.field_label}: "${csvValue}" - Record not found in ${field.lookup_type}`);
                                            }
                                        }
                                    } else {
                                        // This looks like a display name, query by lookup_column
                                        const [lookupResult] = await connection.execute(
                                            `SELECT id FROM ${field.lookup_type} WHERE ${field.lookup_column} = ? AND active = 1`,
                                            [csvValue.toString().trim()]
                                        );
                                        
                                        if (lookupResult.length > 0) {
                                            processedValue = lookupResult[0].id;
                                        } else {
                                            throw new Error(`Invalid ${field.field_label}: "${csvValue}" - Record not found in ${field.lookup_type}`);
                                        }
                                    }
                                } else {
                                    throw new Error(`Lookup field "${field.field_label}" has no lookup configuration`);
                                }
                                break;

                            default:
                                processedValue = csvValue.toString().trim();
                        }

                        // Separate standard and custom fields
                        if (field.custom_field === 1) {
                            customFields.push({ [field.field_name]: processedValue });
                        } else {
                            accountData[field.field_name] = processedValue;
                        }
                    }
                }

                // Add custom fields as JSON string - format as array of objects
                if (customFields.length > 0) {
                    accountData.custom_field = JSON.stringify(customFields);
                }

                // Add active status
                accountData.active = 1;

                // Insert the account
                const keys = Object.keys(accountData);
                const values = Object.values(accountData);
                const placeholders = keys.map(() => '?').join(', ');

                const [result] = await connection.execute(
                    `INSERT INTO accounts (${keys.join(', ')}) VALUES (${placeholders})`,
                    values
                );

                results.successful++;

            } catch (error) {
                results.failed++;
                const errorRecord = {
                    row: rowNumber,
                    error: error.message,
                    data: row
                };
                results.errors.push(errorRecord);
                
                // Add error record to CSV data
                const errorCsvRow = {
                    ...row,
                    'Error Row': rowNumber,
                    'Error Message': error.message
                };
                results.errorCsvData.push(errorCsvRow);
            }
        }

        await connection.commit();

        // Generate error CSV file if there are errors
        if (results.errorCsvData.length > 0) {
            const fs = require('fs');
            const path = require('path');
            
            // Create filename with timestamp
            const timestamp = new Date().toISOString().replace(/[:.]/g, '-').slice(0, 19);
            const filename = `accounts_bulk_import_errors_${timestamp}.csv`;
            const filepath = path.join(__dirname, '../public/uploads', filename);
            
            // Ensure uploads directory exists
            const uploadsDir = path.dirname(filepath);
            if (!fs.existsSync(uploadsDir)) {
                fs.mkdirSync(uploadsDir, { recursive: true });
            }
            
            // Get CSV headers from the original CSV data (first row) plus error columns
            const originalHeaders = Object.keys(csvData[0] || {});
            const errorHeaders = ['Error Row', 'Error Message'];
            const headers = [...originalHeaders, ...errorHeaders];
            
            // Create CSV content
            let csvContent = headers.join(',') + '\n';
            
            results.errorCsvData.forEach(row => {
                const values = headers.map(header => {
                    let value = '';
                    if (header === 'Error Row') {
                        value = row['Error Row'] || '';
                    } else if (header === 'Error Message') {
                        value = row['Error Message'] || '';
                    } else {
                        value = row[header] || '';
                    }
                    // Escape quotes and wrap in quotes if contains comma or quote
                    const escapedValue = value.toString().replace(/"/g, '""');
                    return `"${escapedValue}"`;
                });
                csvContent += values.join(',') + '\n';
            });
            
            // Write CSV file
            fs.writeFileSync(filepath, csvContent);
            
            // Set the file path for response
            results.errorCsvFile = `/uploads/${filename}`;
        }

        return results;

    } catch (error) {
        await connection.rollback();
        throw error;
    } finally {
        connection.release();
    }
}


module.exports = {
    create,
    findAll,
    findById,
    update,
    deleteRecord,
    findAccountMeetings,
    findByTasks,
    getAccountRelations,
    getAllFiltersList,
    createAccountFilterView,
    deleteAccountFilterView,
    getFieldsForFilter,
    getAccountListWithFilters,
    updateFilterView,
    bulkImport
};