const db = require('../config/db');
const NotificationModel = require('../models/NotificationModel');
const { Client } = require('@googlemaps/google-maps-services-js');
const googleMapsClient = new Client({});
const { google } = require('googleapis');
const calendar = google.calendar('v3');
// Add this helper function at the top of the file
async function createGoogleCalendarEvent(auth, eventData) {
    try {
        const calendar = google.calendar({ version: 'v3', auth });
        return await calendar.events.insert({
            calendarId: 'primary',
            resource: {
                summary: eventData.title,
                description: eventData.description,
                start: {
                    dateTime: eventData.start,
                    timeZone: 'Asia/Kolkata',
                },
                end: {
                    dateTime: eventData.end,
                    timeZone: 'Asia/Kolkata',
                },
                attendees: eventData.attendees,
                reminders: {
                    useDefault: true
                }
            }
        });
    } catch (error) {
        console.error('Error creating calendar event:', error);
        throw error;
    }
}
async function createSalesActivity(activityData) {
    const connection = await db.getConnection();
    try {
        await connection.beginTransaction();

        // Validate required fields
        if (!activityData.activity_name) {
            throw new Error('Activity name is required');
        }

        // Insert into salesactivities table
        const [result] = await connection.execute(
            `INSERT INTO salesactivities (
                activity_id, 
                activity_name, 
                activity_icon, 
                allow_checkin_checkout, 
                show_calendar, 
                mark_completed, 
                allow_edit, 
                active, 
                custom_activity, 
                created_on,
                created_at
            ) 
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, NOW(), NOW())`,
            [
                activity_id = null,
                activityData.activity_name,
                "/sales/" + activityData.activity_icon || null,
                activityData.allow_checkin_checkout || 0,
                activityData.show_calendar || 0,
                activityData.mark_completed || 0,
                activityData.allow_edit || 1,
                activityData.active || 1,
                activityData.custom_activity || 0
            ]
        );

        const salesActivityId = result.insertId;

        // Insert outcomes if provided
        if (activityData.outcomes && Array.isArray(activityData.outcomes)) {
            for (const outcome of activityData.outcomes) {
                await connection.execute(
                    `INSERT INTO salesactivityoutcomes (
                        salesactivities_id,
                        outcome,
                        created_at,
                        updated_at
                    ) VALUES (?, ?, NOW(), NOW())`,
                    [
                        salesActivityId,
                        outcome
                    ]
                );
            }
        }

        await connection.commit();

        return {
            id: salesActivityId,
            ...result,
            outcomes_count: activityData.outcomes ? activityData.outcomes.length : 0
        };

    } catch (error) {
        await connection.rollback();
        throw error;
    } finally {
        connection.release();
    }
}

async function getAllSalesActivityData(id) {
    const [result] = await db.execute(
        'SELECT * FROM salesactivitydatas WHERE owner_id = ? AND salesactivities_id = 3 AND active = 1 ORDER BY id DESC',
        [id]
    );

    // if(result.length == 0){
    //     return {
    //         status: 200,
    //         message: 'No sales activities found',
    //     };
    // }

    return result;
 }

// get all meetings
async function getAllMeetings(start_date = null, end_date = null, status = null, role_id = null, user_id = null, user = null) {
    try {

        console.log(start_date, end_date, status, role_id, user_id, user);
        let query = `
            SELECT 
                sad.*,
                owner.name AS owner_name,
                creator.name AS creator_name,
                updater.name AS updater_name,
                CASE 
                    WHEN sad.targetable_type = 'contacts' THEN CONCAT(c.first_name, ' ', c.last_name)
                    WHEN sad.targetable_type = 'accounts' THEN ac.name
                    WHEN sad.targetable_type = 'deals' THEN d.name
                    ELSE NULL
                END as target_name,
                CASE 
                    WHEN sad.targetable_type = 'contacts' THEN c.emails
                    WHEN sad.targetable_type = 'accounts' THEN ac.website
                    WHEN sad.targetable_type = 'deals' THEN d.amount
                    ELSE NULL
                END as target_details
            FROM salesactivitydatas sad
            LEFT JOIN users AS owner ON sad.owner_id = owner.id
            LEFT JOIN users AS creator ON sad.creater_id = creator.id
            LEFT JOIN users AS updater ON sad.updater_id = updater.id
            LEFT JOIN contacts c ON sad.targetable_type = 'contacts' AND sad.targetable_id = c.id
            LEFT JOIN accounts ac ON sad.targetable_type = 'accounts' AND sad.targetable_id = ac.id
            LEFT JOIN deals d ON sad.targetable_type = 'deals' AND sad.targetable_id = d.id
            WHERE sad.salesactivities_id = 3 AND sad.active = 1`;

        const params = [];

        // Add role-based filtering
        if (role_id === 2 && user_id) {
            // For role_id = 2 (Sales Executive), filter by territory
            query += ` AND owner.territory_id = (SELECT territory_id FROM users WHERE id = ?)`;
            params.push(user_id);
        }

        // For role_id = 1 (Admin), no additional filtering needed - shows all meetings
        console.log(start_date, end_date);
        // Add date filtering if dates are provided
        if (start_date && end_date) {
            query += ` AND sad.start_date BETWEEN ? AND ?`;
            params.push(start_date, end_date);
        }

        console.log(user);

        
        // Add status filtering if status is provided
        if (status) {
            query += ` AND sad.status = ?`;
            params.push(status);
        }

        if (user) {
            query += ` AND sad.owner_id = ?`;
            params.push(user);
        }


        query += ` ORDER BY sad.created_at DESC`;

        const [meetings] = await db.execute(query, params);
        console.log(meetings);

        // Process each meeting to get attendee details
        for (let meeting of meetings) {
            // Get attendee details
            if (meeting.attendees) {
                const attendeeIds = meeting.attendees.split(',').map(id => id.trim());
                if (attendeeIds.length > 0) {
                    const placeholders = attendeeIds.map(() => '?').join(',');
                    const [attendeeRows] = await db.execute(`
                        SELECT id, name, email, mobile
                        FROM users
                        WHERE id IN (${placeholders})`,
                        attendeeIds
                    );
                    meeting.attendee_details = attendeeRows;
                } else {
                    meeting.attendee_details = [];
                }
            } else {
                meeting.attendee_details = [];
            }

            // Structure target information
            meeting.target = {
                type: meeting.targetable_type,
                id: meeting.targetable_id,
                name: meeting.target_name,
            };

            // Remove the raw fields
            delete meeting.targetable_type;
            delete meeting.targetable_id;
            delete meeting.target_name;
            delete meeting.target_details;
        }

        console.log(meetings[0]);

        if (meetings.length > 0) {

            if (meetings[0].start_date) {
                meetings[0].start_date = formatDate(meetings[0].start_date).split('T')[0];
                console.log(meetings[0].start_date);
            }
            if (meetings[0].end_date) {
                meetings[0].end_date = formatDate(meetings[0].end_date).split('T')[0];
            }
        }

        return meetings;
    } catch (error) {
        console.error('Error in getAllMeetings:', error);
        throw new Error('Error fetching meetings');
    }
}




async function getSalesActivities() {
    const [rows] = await db.execute('SELECT * FROM salesactivities WHERE active = 1');
    return rows;
}

async function createSalesActivityData(Data) {
    let connection = null;
    try {
        // Get connection from pool
        connection = await db.getConnection();
        await connection.beginTransaction();

        // Convert time format if needed
        if (Data.start_time) {
            const timeParts = Data.start_time.match(/(\d+):(\d+)\s*(AM|PM)/i);
            if (timeParts) {
                let hours = parseInt(timeParts[1]);
                const minutes = timeParts[2];
                const meridiem = timeParts[3].toUpperCase();

                if (meridiem === 'PM' && hours < 12) hours += 12;
                if (meridiem === 'AM' && hours === 12) hours = 0;

                Data.start_time = `${hours.toString().padStart(2, '0')}:${minutes}:00`;
            }
        }
        if (Data.end_time) {
            const timeParts = Data.end_time.match(/(\d+):(\d+)\s*(AM|PM)/i);
            if (timeParts) {
                let hours = parseInt(timeParts[1]);
                const minutes = timeParts[2];
                const meridiem = timeParts[3].toUpperCase();

                if (meridiem === 'PM' && hours < 12) hours += 12;
                if (meridiem === 'AM' && hours === 12) hours = 0;

                Data.end_time = `${hours.toString().padStart(2, '0')}:${minutes}:00`;
            }
        }

        // Clean data
        Object.keys(Data).forEach(key => {
            if (Data[key] === undefined || Data[key] === '' || Data[key] === null) {
                delete Data[key];
            }
        });

        // Set default status
        Data.status = Data.status || 'Pending';
        Data.created_at = new Date();

        // Insert activity
        const keys = Object.keys(Data);
        const values = Object.values(Data);
        const placeholders = '?' + ', ?'.repeat(values.length - 1);
        // console.log(keys, 'and', values);
        
        const [result] = await connection.execute(
            `INSERT INTO salesactivitydatas (${keys}) VALUES (${placeholders})`,
            values
        );

        const activityId = result.insertId;
        
         // Create Google Calendar events
        if (Data.owner_id) {
            // Get owner's Google Calendar credentials
            const [ownerCreds] = await connection.execute(
                'SELECT google_calendar_token FROM users WHERE id = ?',
                [Data.owner_id]
            );

            if (ownerCreds[0]?.google_calendar_token) {
                const auth = new google.auth.OAuth2(
                    process.env.GOOGLE_CLIENT_ID,
                    process.env.GOOGLE_CLIENT_SECRET
                );
                // console.log("auth"+ownerCreds[0]?.google_calendar_token);
                auth.setCredentials(ownerCreds[0].google_calendar_token);

                // Create event data
                const eventData = {
                    title: Data.activity_title,
                    description: Data.description || '',
                    start: `${Data.start_date}T${Data.start_time}`,
                    end: `${Data.end_date}T${Data.end_time}`,
                    attendees: []
                };

                // Add attendees if present
                if (Data.attendees) {
                    const attendeeIds = Array.isArray(Data.attendees) ? 
                        Data.attendees : 
                        Data.attendees.split(',').map(id => id.trim());

                    if (attendeeIds.length > 0) {
                        const [attendeeEmails] = await connection.execute(
                            `SELECT email FROM users WHERE id IN (${attendeeIds.map(() => '?').join(',')})`,
                            attendeeIds
                        );

                        eventData.attendees = attendeeEmails.map(user => ({
                            email: user.email,
                            responseStatus: 'needsAction'
                        }));
                    }
                }

                // Create calendar event
                try {
                    const calendarEvent = await createGoogleCalendarEvent(auth, eventData);
                    // console.log("event"+calendarEvent);
                    // Store calendar event ID
                    await connection.execute(
                        'UPDATE salesactivitydatas SET calendar_event_id = ? WHERE id = ?',
                        [calendarEvent.data.id, activityId]
                    );
                } catch (calendarError) {
                    console.error('Error creating calendar event:', calendarError);
                    // Continue with the transaction even if calendar event creation fails
                }
            }
        }
        // Get activity type name
        const [activityTypes] = await connection.execute(
            'SELECT activity_name FROM salesactivities WHERE id = ?',
            [Data.salesactivities_id]
        );

        const activityName = activityTypes.length > 0 ? activityTypes[0].activity_name : 'Activity';
       
        // Send notifications
        if (Data.owner_id) {
            try {
                // Create trigger datetime by combining date and time
                const triggerDateTime = new Date(`${Data.start_date}T${Data.start_time}`);
                
                const notificationData = {
                    user_id: Data.owner_id,
                    message: `New ${activityName}: ${Data.activity_title}`,
                    trigger_date: Data.start_date,
                    trigger_time: triggerDateTime.toISOString().slice(0, 19).replace('T', ' '), // Format as 'YYYY-MM-DD HH:mm:ss'
                    targetable_type: 'sales_activity',
                    targetable_id: activityId,
                    read_status: 0
                };
                await NotificationModel.create(notificationData);
                // console.log('Notification created for owner');

                // Handle attendees notifications
                if (Data.attendees) {
                    // Convert attendees to array if it's a string
                    let attendees = Data.attendees;
                    if (typeof attendees === 'string') {
                        // If it's a comma-separated string, split it
                        if (attendees.includes(',')) {
                            attendees = attendees.split(',').map(id => id.trim());
                        } else {
                            // If it's a single ID, convert to array
                            attendees = [attendees];
                        }
                    }

                    // Ensure attendees is an array
                    if (Array.isArray(attendees)) {
                        for (const attendeeId of attendees) {
                            if (attendeeId !== Data.owner_id) {
                                const attendeeNotification = {
                                    user_id: attendeeId,
                                    message: `You've been added to ${activityName}: ${Data.activity_title}`,
                                    trigger_date: Data.start_date,
                                    trigger_time: triggerDateTime.toISOString().slice(0, 19).replace('T', ' '), // Format as 'YYYY-MM-DD HH:mm:ss'
                                    targetable_type: 'sales_activity',
                                    targetable_id: activityId,
                                    read_status: 0
                                };
                                await NotificationModel.create(attendeeNotification);
                            }
                        }
                    }
                    // console.log('Notifications created for attendees');
                }
            } catch (notificationError) {
                console.error('Error creating notifications:', notificationError);
                // Continue with the transaction even if notification creation fails
            }
        }

        // Block owner's Google Calendar if salesactivities_id is 3
        if (Data.salesactivities_id === 3) {
            try {
                // Assuming you have a function to get the owner's access token
                const ownerAccessToken = await getOwnerAccessToken(Data.owner_id);
                if (ownerAccessToken) {
                    const calendar = google.calendar({ version: 'v3', auth: ownerAccessToken });
                    const event = {
                        summary: Data.activity_title,
                        start: {
                            dateTime: new Date(`${Data.start_date}T${Data.start_time}`).toISOString(),
                            timeZone: 'UTC',
                        },
                        end: {
                            dateTime: new Date(`${Data.end_date}T${Data.end_time}`).toISOString(),
                            timeZone: 'UTC',
                        },
                    };
                    await calendar.events.insert({
                        calendarId: 'primary',
                        resource: event,
                    });
                    // console.log('Calendar event created for owner');
                }
            } catch (calendarError) {
                console.error('Error creating calendar event:', calendarError);
                // Continue with the transaction even if calendar event creation fails
            }
        }

        await connection.commit();
        return activityId;

    } catch (error) {
        if (connection) {
            await connection.rollback();
        }
        console.error('Error creating sales activity:', error);
        throw new Error(`Failed to create sales activity: ${error.message}`);
    } finally {
        if (connection) {
            connection.release();
        }
    }
}

// async function getAllSalesActivityData(id) {
//    const [result] = await db.execute(
//        'SELECT * FROM salesactivitydatas WHERE owner_id = ? AND salesactivities_id = 3 AND active = 1 ORDER BY id DESC',
//        [id]
//    );
//    return result;
// }

async function getSalesActivityData(id, userId) {
    // First get the sales activity data
    const [activityData] = await db.execute(
        'SELECT salesactivitydatas.*, ' +
        'owner.name as owner_name, ' +
        'creator.name as creator_name, ' +
        'updater.name as updater_name ' +
        'FROM salesactivitydatas ' +
        'LEFT JOIN users as owner ON salesactivitydatas.owner_id = owner.id ' +
        'LEFT JOIN users as creator ON salesactivitydatas.creater_id = creator.id ' +
        'LEFT JOIN users as updater ON salesactivitydatas.updater_id = updater.id ' +
        'WHERE salesactivitydatas.id = ?',
        [id]
    );
    // console.log("activityData",activityData);
    // const [outcome] = await db.execute('SELECT * FROM salesactivityoutcomes WHERE salesactivities_id = 3')

    if (!activityData[0]) {
        return null;
    }

    const activity = activityData[0];

    // console.log(activity);

    // Get attendees details
    if (activity.attendees) {
        const attendeeIds = activity.attendees.split(',').map(id => parseInt(id.trim()));
        if (attendeeIds.length > 0) {
            const placeholders = attendeeIds.map(() => '?').join(',');
            const [attendees] = await db.execute(
                `SELECT id, name FROM users WHERE id IN (${placeholders})`,
                attendeeIds
            );
            activity.attendees_details = attendees;
        }
    }

    // check meeting is inprogress or not
    const [processRows] = await db.execute(
    `SELECT * FROM salesactivitydatas WHERE owner_id = ? AND status = 'inprogress' AND salesactivities_id = 3 AND active = 1`,
    [userId]
    );

    activity.process = !(processRows.length > 0);

    if (activity.start_date) {
        activity.start_date = formatDate(activity.start_date).replaceAll('/','-').split('-').reverse().join('-')
    }
    if (activity.end_date) {
        activity.end_date = formatDate(activity.end_date).replaceAll('/','-').split('-').reverse().join('-')
    }

    


    // Get target details based on targetable_type
    if (activity.targetable_id && activity.targetable_type) {
        let targetQuery;
        switch (activity.targetable_type.toLowerCase()) {
            case 'contacts':
                targetQuery = 'SELECT id, first_name, last_name, emails FROM contacts WHERE id = ?';
                break;
            case 'accounts':
                targetQuery = 'SELECT id, name, website FROM accounts WHERE id = ?';
                break;
            case 'deals':
                targetQuery = 'SELECT id, name, amount FROM deals WHERE id = ?';
                break;
            default:
                targetQuery = null;
        }

        if (targetQuery) {
            const [targetData] = await db.execute(targetQuery, [activity.targetable_id]);
            if(activity.targetable_type.toLowerCase() === 'contacts'){
                targetData.forEach(contact => {
                    contact.name = contact.first_name + ' ' + contact.last_name;
                });
            }
            activity.target_details = targetData[0] || null;
        }
    }
    // outcomes
    const [outcomes] = await db.execute(
        'SELECT id, outcome FROM salesactivityoutcomes WHERE salesactivities_id = 3 ORDER BY id',
    )
    activity.outcomes = outcomes;
    // activity.outcomes = outcome;
    // console.log(activity);
    return activity;
}
// update the salesactivity data meeting status
async function updateMeetingStatus(id, data) {
    const outcome = data.outcome;
    const notes = data.notes;
    const next_plan = data.next_plan;
    const [result] = await db.execute(
        'UPDATE salesactivitydatas SET outcome = ? notes = ? next_plan = ? WHERE id =?',
        [outcome, notes, next_plan, id]
    );
    return result;
}

// update sales activity
async function updateSalesActivityData(Data, id) {

    try {
        // console.log("all data",Data)
        // Remove any time component from dates
        Data.start_date = Data.start_date.split('T')[0];
        Data.end_date = Data.end_date.split('T')[0];
        // Remove undefined or empty values
        Object.keys(Data).forEach(key => {
            if (Data[key] === undefined || Data[key] === '' || Data[key] === null) {
                delete Data[key];
            }
        });
        
        // Prepare update fields and values
        const updateFields = Object.keys(Data)
        .map(key => `${key} = ?`)
        .join(', ');
        // updateFields += ', updated_at = NOW()';
        
        const updateValues = Object.values(Data);
        // console.log("updated values",updateValues);
        const [result] = await db.execute(`UPDATE salesactivitydatas SET ${updateFields} WHERE id = ?`,[...updateValues, id]);
        // console.log( "results", result);
        return result;
    } catch (error) {
        console.error('Error updating sales activity data:', error);
        return {
            status: 400,
            success: false,
            message: 'Error updating sales activity data',
            error: error.message
        };
    }
}




async function updateSalesActivity(id, data) {
    const connection = await db.getConnection();
    try {
        await connection.beginTransaction();

        // Validate required fields
        if (!id) {
            throw new Error('Activity ID is required for update');
        }

        // Update salesactivities table
        const [result] = await connection.execute(
            `UPDATE salesactivities 
             SET activity_id = ?, 
                 activity_name = ?, 
                 activity_icon = ?, 
                 allow_checkin_checkout = ?, 
                 show_calendar = ?, 
                 mark_completed = ?, 
                 allow_edit = ?, 
                 active = ?, 
                 custom_activity = ?,
                 updated_at = NOW()
             WHERE id = ?`,
            [
                activity_id = null,
                data.activity_name,
                "/sales/" + data.activity_icon || null,
                data.allow_checkin_checkout || 0,
                data.show_calendar || 0,
                data.mark_completed || 0,
                data.allow_edit || 1,
                data.active || 1,
                data.custom_activity || 0,
                id
            ]
        );

        // If outcomes are provided, update them
        if (data.outcomes && Array.isArray(data.outcomes)) {
            // First, delete existing outcomes
            await connection.execute(
                'DELETE FROM salesactivityoutcomes WHERE salesactivities_id = ?',
                [id]
            );

            // Then insert new outcomes
            for (const outcome of data.outcomes) {
                await connection.execute(
                    `INSERT INTO salesactivityoutcomes (
                        salesactivities_id,
                        outcome,
                        created_at,
                        updated_at
                    ) VALUES (?, ?, NOW(), NOW())`,
                    [
                        id,
                        outcome
                    ]
                );
            }
        }

        await connection.commit();

        return {
            id: id,
            ...result,
            outcomes_count: data.outcomes ? data.outcomes.length : 0
        };

    } catch (error) {
        await connection.rollback();
        throw error;
    } finally {
        connection.release();
    }
}

async function softDeleteSalesActivity(id) {
    await db.execute('UPDATE salesactivities SET active = 0 WHERE id = ?', [id]);
}
async function calculateDistance(origin, destination) {
    try {
        const originStr = `${origin.lat},${origin.lng}`;
        const destinationStr = `${destination.lat},${destination.lng}`;

        // console.log("origin:", originStr, "destination:", destinationStr);

        const response = await googleMapsClient.distancematrix({
            params: {
                origins: [originStr],
                destinations: [destinationStr],
                key: process.env.GOOGLE_MAPS_API_KEY
            }
        });

        const element = response.data.rows[0].elements[0];

        if (element.status === 'OK') {
            // console.log("Distance:", element.distance.text);
            return {
                distance: element.distance.text,
                distanceValue: element.distance.value, // in meters
                duration: element.duration.text
            };
        }

        return null;
    } catch (error) {
        console.error('Error calculating distance:', error.response?.data || error.message);
        return null;
    }
}


async function getPreviousActivity(userId, currentDate) {
    const [result] = await db.execute(
        `SELECT checkin_latitude, checkin_longitude 
         FROM salesactivitydatas 
         WHERE owner_id = ? 
         AND DATE(checkin_time) = DATE(?)
         AND status != 'Pending'
         ORDER BY checkin_time DESC 
         LIMIT 1`,
        [userId, currentDate]
    );
    return result[0];
}

async function getDefaultTransport(userId) {
    const [userRole] = await db.execute(
        `SELECT r.default_transport 
         FROM users u 
         JOIN roles r ON u.roles_id = r.id 
         WHERE u.id = ?`,
        [userId]
    );
    return userRole[0]?.default_transport;
}

async function getConveyanceCharge(transportMode) {
    // console.log("transportModeFetching  "+transportMode);
    const [charge] = await db.execute(
        'SELECT conveyance_charge FROM conveyanceconfigs WHERE transport_mode = ?',
        [transportMode]
    );
    return charge[0]?.conveyance_charge || 0;
}

async function updateActivityStatus(id, data, userId) {
    // console.log(id +" transport_mode "+data.transport_mode +" amount "+data.amount +" latitude "+data.latitude +" longitude "+data.longitude +" location "+data.location +" remarks "+data.remarks +"\n bill_attachment "+data.bill_attachment +" attachment "+data.attachment);
    const [currentStatus] = await db.execute(
        'SELECT salesactivities_id, status, distance FROM salesactivitydatas WHERE id = ?',
        [id]
    );
    console.log("currentStatus ",currentStatus[0].salesactivities_id);
    if (!currentStatus[0]) {
        throw new Error('Activity not found');
    }
    // console.log("currentStatus "+currentStatus[0].status);
    const status = currentStatus[0].status;
    const updateData = {};
    // console.log("status "+status);
    // console.log("data "+data);
    const dateInIndia = new Date().toLocaleString("en-US", { timeZone: "Asia/Kolkata" });
    const checkinDate = new Date(dateInIndia);
    if (status === 'Pending') {
        // Get previous activity location
        const previousActivity = await getPreviousActivity(userId,  new Date());
        // console.log("previousActivity "+previousActivity);
        // Update for check-in
        updateData.status = 'Inprogress';
        updateData.checkin_latitude = data.latitude;
        updateData.checkin_location = data.location;
        updateData.checkin_longitude = data.longitude;
        updateData.check_in_attachment = data.attachment;
        updateData.checkin_time = new Date();
        // console.log("updateData "+updateData.checkin_time);
        updateData.claim_status = 'Pending';
        // Calculate distance if previous activity exists
        if (previousActivity) {
            const origin = {
                lat: parseFloat(previousActivity.checkin_latitude),
                lng: parseFloat(previousActivity.checkin_longitude)
            };
            const destination = {
                lat: parseFloat(data.latitude),
                lng: parseFloat(data.longitude)
            };

            const distanceData = await calculateDistance(origin, destination);
            // console.log("distanceData "+distanceData);
            if (distanceData) {
                const distanceInKm = distanceData.distanceValue/1000;
                updateData.distance = distanceInKm;

                // Get default transport mode and conveyance charge
                const defaultTransport = await getDefaultTransport(userId);
                const conveyanceCharge = await getConveyanceCharge(defaultTransport);

                // Calculate and store claim amount
                updateData.transport_mode = defaultTransport;
                updateData.conveyance_charge = conveyanceCharge;
                updateData.claim_amount = distanceInKm * conveyanceCharge;
            }
        }
    } else if (status === 'Inprogress') {
        // Get check-in time and activity details
        const [activityData] = await db.execute(
            'SELECT checkin_time, salesactivities_id, checkin_latitude, checkin_longitude, notes, transport_mode FROM salesactivitydatas WHERE id = ?',
            [id]
        );
        // console.log("activityData "+activityData[0]);
        const [existingMeeting] =  await db.execute(
            'SELECT * FROM salesactivitydatas WHERE salesactivities_id = 3 AND status = "Inprogress" AND owner_id = ?',
            [userId]);

            if (currentStatus[0].salesactivities_id === 10 && existingMeeting && existingMeeting.length > 0) {
                throw new Error('You have an existing meeting.  Please Close the existing meeting.');
            }
        
        const checkInTime = new Date(activityData[0].checkin_time);
        const checkOutTime = new Date();
        
        // Calculate time spent
        const timeSpentSeconds = Math.floor((checkOutTime - checkInTime) / 1000);
        const hours = Math.floor(timeSpentSeconds / 3600);
        const minutes = Math.floor((timeSpentSeconds % 3600) / 60);
        const seconds = timeSpentSeconds % 60;
        const timeSpentDisplay = `${hours.toString().padStart(2, '0')}:${minutes.toString().padStart(2, '0')}:${seconds.toString().padStart(2, '0')}`;
            // console.log("timeSpentDisplay "+timeSpentDisplay);
        // Set basic checkout data
        updateData.status = 'Closed';
        updateData.checkout_latitude = data.latitude;
        updateData.checkout_location = data.location;
        updateData.checkout_longitude = data.longitude;
        updateData.checkout_time = checkOutTime;
        updateData.timespent = timeSpentSeconds;
        updateData.check_out_attachment = data.attachment;
        updateData.timespent_display = timeSpentDisplay;
        updateData.updater_id = userId;
        updateData.notes = data.remarks || null;
        updateData.claim_status = 'Pending';
        //updateData.transport_mode = data.transport_mode;
        // For salesactivities_id = 10, calculate distance and claim amount
        if (activityData[0].salesactivities_id === 10) {
            const origin = {
                lat: parseFloat(activityData[0].checkin_latitude),
                lng: parseFloat(activityData[0].checkin_longitude)
            };
            const destination = {
                lat: parseFloat(data.latitude),
                lng: parseFloat(data.longitude)
            };

            const distanceData = await calculateDistance(origin, destination);
                    if (distanceData) {
                const distanceInKm = distanceData.distanceValue/1000;
                updateData.distance = distanceInKm;
                // Get default transport mode and conveyance charge
                const defaultTransport = await getDefaultTransport(userId);
                const conveyanceCharge = await getConveyanceCharge(defaultTransport);

                // Calculate and store claim amount
                updateData.transport_mode = defaultTransport;
                updateData.conveyance_charge = conveyanceCharge;
                updateData.claim_amount = distanceInKm * conveyanceCharge;
                console.log("updateData.claim_amount ",updateData);
            }
        } else {
            if(data.transport_mode) {
                if(data.transport_mode === 'Public Transport') {
                    // Validate amount and bill attachment for public transport
                    if (!data.amount || !data.bill_attachment) {
                        throw new Error('Amount and bill attachment are required for Public Transport');
                    }

                    // Validate amount is a valid number
                    const amount = parseFloat(data.amount);
                    if (isNaN(amount) || amount <= 0) {
                        throw new Error('Invalid amount for Public Transport');
                    }

                    updateData.transport_mode = data.transport_mode;
                    updateData.conveyance_charge = 0;
                    updateData.claim_amount = amount;
                    // if(data.bill_attachment)
                    updateData.claim_image = data.bill_attachment ;
                } else {
                    // Handle other transport modes
                    const conveyanceCharge = await getConveyanceCharge(data.transport_mode);
                    updateData.transport_mode = data.transport_mode;
                    updateData.conveyance_charge = conveyanceCharge;
                    updateData.claim_amount = currentStatus[0].distance * conveyanceCharge;
                }
            }
        }
    } else {
        throw new Error('Invalid status for update');
    }

    const keys = Object.keys(updateData);
    const values = Object.values(updateData);
    
    const setClause = keys.map(key => `${key} = ?`).join(', ');
    // console.log(`UPDATE salesactivitydatas SET ${setClause} WHERE id = ?`,
    //     [...values, id])
    const [result] = await db.execute(
        `UPDATE salesactivitydatas SET ${setClause} WHERE id = ?`,
        [...values, id]
    );

    return {
        success: result.affectedRows > 0,
        newStatus: updateData.status
    };
}
async function getAttendanceHistory(userId, filter = 'today', month = null, year = null) {
    let query = `
        SELECT sad.*, 
               u.name as owner_name,
               cu.name as updater_name
        FROM salesactivitydatas sad
        LEFT JOIN users u ON sad.owner_id = u.id
        LEFT JOIN users cu ON sad.updater_id = cu.id
        WHERE sad.salesactivities_id = 10 
        AND sad.owner_id = ?`;
    
    const params = [userId];

    // Add date filters
    switch (filter) {
        case 'today':
            query += ' AND DATE(sad.start_date) = CURDATE()';
            break;
        
        case 'weekly':
            query += ' AND YEARWEEK(sad.start_date, 1) = YEARWEEK(CURDATE(), 1)';
            break;
        
        case 'monthly':
            if (!month || !year) {
                throw new Error('Month and year are required for monthly filter');
            }
            query += ' AND MONTH(sad.start_date) = ? AND YEAR(sad.start_date) = ?';
            params.push(month, year);
            break;

        default:
            throw new Error('Invalid filter type');
    }

    query += ' ORDER BY sad.start_date DESC';

    const [rows] = await db.execute(query, params);



    // Format the response
    return {
        filter_type: filter,
        period: getPeriodLabel(filter, month, year),
        attendance_data: rows.map(row => ({
            id: row.id,
            date: formatDate(row.start_date),
            check_in_time: row.checkin_time,
            check_in_location: row.checkin_location,
            check_out_time: row.checkout_time,
            check_out_location: row.checkout_location,
            status: row.status,
            time_spent: row.timespent_display,
            owner_name: row.owner_name,
            updater_name: row.updater_name,
            distance: row.distance || 0,
            claim_amount: row.claim_amount || 0,
            transport_mode: row.transport_mode
        }))
    };
}

// Helper functions for date formatting
function formatDate(date) {
    if (!date) return null;
    return new Date(date).toLocaleDateString('en-GB', {
        day: '2-digit',
        month: '2-digit',
        year: 'numeric'
    });
}

// function formatTime(time) {
//     if (!time) return null;
//     return new Date(time).toLocaleTimeString('en-US', {
//         hour: '2-digit',
//         minute: '2-digit',
//         hour12: true
//     });
// }

function formatTime(time) {
    if (!time) return null;
    const date = new Date(time);

    // IST offset in minutes
    const IST_OFFSET = 5.5 * 60;
    // Convert to IST
    const utc = date.getTime() + (date.getTimezoneOffset() * 60000);
    const istDate = new Date(utc + (IST_OFFSET * 60000));

    // Format in 12-hour time and force AM/PM uppercase
    let formatted = istDate.toLocaleTimeString('en-US', {
        hour: '2-digit',
        minute: '2-digit',
        hour12: true
    });

    // Ensure AM/PM is always uppercase and with a space
    formatted = formatted.replace(/([AP]M)$/, ' $1').toUpperCase();

    return formatted;
}

function formatisoTime(time) {
    if (!time) return null;

    // If it's a Date object, convert to ISO string
    if (time instanceof Date) {
        time = time.toISOString();
    } else {
        // Otherwise, ensure it's a string
        time = String(time);
    }

    // Force parse as UTC (add 'Z' if missing)
    let utcString = time.includes('T') ? time : time.replace(' ', 'T') + 'Z';
    const date = new Date(utcString);

    // IST offset in minutes
    const IST_OFFSET = 5.5 * 60;
    const istDate = new Date(date.getTime() + IST_OFFSET * 60000);

    // Format in 12-hour time with uppercase AM/PM
    let formatted = istDate.toLocaleTimeString('en-US', {
        hour: '2-digit',
        minute: '2-digit',
        hour12: true
    });

    formatted = formatted.replace(/([AP]M)$/, ' $1').toUpperCase();
    return formatted;
}





function formatDateTime(datetime) {
    if (!datetime) return null;
    const date = new Date(datetime);
    return date.toLocaleString('en-GB', {
        day: '2-digit',
        month: '2-digit',
        year: 'numeric',
        hour: '2-digit',
        minute: '2-digit',
        second: '2-digit',
        hour12: false
    });
}



function getPeriodLabel(filter, month, year) {
    switch (filter) {
        case 'today':
            return new Date().toLocaleDateString('en-GB', {
                day: '2-digit',
                month: '2-digit',
                year: 'numeric'
            });
        
        case 'weekly':
            const now = new Date();
            const startOfWeek = new Date(now.setDate(now.getDate() - now.getDay()));
            const endOfWeek = new Date(now.setDate(now.getDate() - now.getDay() + 6));
            return `${formatDate(startOfWeek)} to ${formatDate(endOfWeek)}`;
        
        case 'monthly':
            return new Date(year, month - 1).toLocaleDateString('en-GB', {
                month: 'long',
                year: 'numeric'
            });
            
        default:
            return '';
    }
}

// attendance history for all users
async function getAllAttendanceHistory(filter = 'today', month = null, year = null, start_date = null, end_date = null, user = null) {
    let query = `
        SELECT sad.*, 
               u.name as owner_name,
               cu.name as updater_name
        FROM salesactivitydatas sad
        LEFT JOIN users u ON sad.owner_id = u.id
        LEFT JOIN users cu ON sad.updater_id = cu.id
        WHERE sad.salesactivities_id = 10`;
    
    const params = [];

    // Add date filters
    switch (filter) {
        case 'date_range':
            query += ' AND DATE(sad.start_date) BETWEEN ? AND ?';
            params.push(start_date, end_date);
            break;

        case 'today':
            query += ' AND DATE(sad.start_date) = CURDATE()';
            break;
        
        case 'weekly':
            query += ' AND YEARWEEK(sad.start_date, 1) = YEARWEEK(CURDATE(), 1)';
            break;
        
        case 'monthly':
            if (!month || !year) {
                throw new Error('Month and year are required for monthly filter');
            }
            query += ' AND MONTH(sad.start_date) = ? AND YEAR(sad.start_date) = ?';
            params.push(month, year);
            break;

        default:
            throw new Error('Invalid filter type');
    }

    if (user) { 
        query += ' AND sad.owner_id = ?';
        params.push(user);
    }

    query += ' ORDER BY sad.start_date DESC, u.name ASC';

    const [rows] = await db.execute(query, params);
    console.log(formatisoTime('2025-07-02 18:29:01')); // "11:59 PM"
    console.log(formatisoTime(new Date('2025-07-02T18:29:01Z'))); // "11:59 PM"
    console.log(formatisoTime(1720200541000)); // (timestamp in ms)


    // Format the response
    return {
        filter_type: filter,
        period: getPeriodLabel(filter, month, year, start_date, end_date),
        attendance_data: rows.map(row => ({
            id: row.id,
            date: formatDate(row.start_date),
            check_in_time: row.checkin_time,
            check_in_location: row.checkin_location,
            check_out_time: row.checkout_time,
            check_out_location: row.checkout_location,
            status: row.status,
            time_spent: row.timespent_display,
            owner_name: row.owner_name,
            updater_name: row.updater_name,
            distance: row.distance || 0,
            claim_amount: row.claim_amount || 0,
            transport_mode: row.transport_mode
        }))
    };
}

// Update getPeriodLabel function to handle date range
function getPeriodLabel(filter, month, year, start_date, end_date) {
    switch (filter) {
        case 'date_range':
            return `${formatDate(start_date)} to ${formatDate(end_date)}`;
            
        case 'today':
            return new Date().toLocaleDateString('en-GB', {
                day: '2-digit',
                month: '2-digit',
                year: 'numeric'
            });
        
        case 'weekly':
            const now = new Date();
            const startOfWeek = new Date(now.setDate(now.getDate() - now.getDay()));
            const endOfWeek = new Date(now.setDate(now.getDate() - now.getDay() + 6));
            return `${formatDate(startOfWeek)} to ${formatDate(endOfWeek)}`;
        
        case 'monthly':
            return new Date(year, month - 1).toLocaleDateString('en-GB', {
                month: 'long',
                year: 'numeric'
            });
            
        default:
            return '';
    }
}


async function getConveyanceList(userId, filter = 'today', month = null, year = null) {
    let query = `SELECT * FROM salesactivitydatas 
                 WHERE owner_id = ? 
                 AND salesactivities_id IN(3,10)`;
    
    const params = [userId];

    // Add date filters
    switch (filter) {
        case 'today':
            query += ' AND DATE(start_date) = CURDATE()';
            break;
        
        case 'weekly':
            query += ' AND YEARWEEK(start_date, 1) = YEARWEEK(CURDATE(), 1)';
            break;
        
        case 'monthly':
            if (!month || !year) {
                throw new Error('Month and year are required for monthly filter');
            }
            query += ' AND MONTH(start_date) = ? AND YEAR(start_date) = ?';
            params.push(month, year);
            break;

        default:
            throw new Error('Invalid filter type');
    }

    query += ' ORDER BY start_date ASC';
    
    const [activities] = await db.execute(query, params);

    // Group activities by date
    const groupedActivities = activities.reduce((acc, activity) => {
        const date = new Date(activity.start_date);
        const formattedDate = date.toLocaleDateString('en-GB', {
            day: '2-digit',
            month: '2-digit',
            year: 'numeric'
        });

        if (!acc[formattedDate]) {
            acc[formattedDate] = {
                conveyance_date: formattedDate,
                distance: "0.00",
                charge: 0,
                activity: []
            };
        }

        acc[formattedDate].activity.push(activity);

        // Sum up distance and charge if available
        if (activity.distance) {
            acc[formattedDate].distance = (parseFloat(acc[formattedDate].distance) + parseFloat(activity.distance)).toFixed(2);
        }
        if (activity.claim_amount) {
            acc[formattedDate].charge += parseFloat(activity.claim_amount);
        }

        return acc;
    }, {});

    // Return formatted response
    return {
        filter_type: filter,
        period: getPeriodLabel(filter, month, year),
        conveyance_data: Object.values(groupedActivities).sort((a, b) => {
            const dateA = a.conveyance_date.split('-').reverse().join('-');
            const dateB = b.conveyance_date.split('-').reverse().join('-');
            return new Date(dateA) - new Date(dateB);
        })
    };
}

// Get conveyance list
async function getconveyance(start_date = null, end_date = null, user_id = null, role = null, user = null) {
    let query = '';
    let params = [];
    if (role === 1) {
        query += `SELECT sad.*,
            SUM(CAST(distance AS DECIMAL(10,2))) as total_distance,
            SUM(
                TIME_TO_SEC(TIMEDIFF(sad.end_time, sad.start_time))/3600
            ) as total_hours
        FROM salesactivitydatas sad
        WHERE sad.salesactivities_id IN (3, 10)
        AND sad.status IN ('Closed')`;
    }
    else if (role === 2) {
        query += `SELECT sad.*,
            SUM(CAST(sad.distance AS DECIMAL(10,2))) as total_distance,
            SUM(
                TIME_TO_SEC(TIMEDIFF(sad.end_time, sad.start_time))/3600
            ) as total_hours
        FROM salesactivitydatas sad
        INNER JOIN users u ON sad.owner_id = u.id
        INNER JOIN territories t ON u.territory_id = t.id
        WHERE sad.salesactivities_id IN (3, 10) 
        AND sad.status IN ('Closed')
        AND u.territory_id = (SELECT territory_id FROM users WHERE id = ?)`;
        params.push(user_id);
    }
    // Add date filtering if dates are provided
    if (start_date && end_date) {
        query += ` AND sad.start_date BETWEEN ? AND ?`;
        params.push(start_date, end_date);
    }

    if (user) {
        query += ` AND sad.owner_id = ?`;
        params.push(user);
    }

    // Group by to get totals while preserving individual records
    query += ` GROUP BY sad.id ORDER BY sad.id DESC`;

    const [rows] = await db.execute(query, params);

    // Calculate grand totals
    const totals = {
        total_distance: rows.reduce((sum, row) => sum + (parseFloat(row.distance) || 0), 0).toFixed(2),
        total_hours: rows.reduce((sum, row) => {
            const start = new Date(`2000-01-01 ${row.start_time}`);
            const end = new Date(`2000-01-01 ${row.end_time}`);
            return sum + ((end - start) / (1000 * 60 * 60));
        }, 0).toFixed(2)
    };

    return { records: rows, totals };
}


// claimalist
async function getallclaims(start_date = null, end_date = null, user_id = null, role = null, territory_id = null) {

    let query = '';
    const params = [];
    if (role === 1) {
        // console.log("Admin or Finance");
        query += `SELECT sad.* 
                FROM salesactivitydatas sad
                INNER JOIN users u ON sad.owner_id = u.id`;
        
        if (territory_id) {
            query += ` WHERE sad.salesactivities_id IN (3, 10)
                    AND sad.claim_status IN ('Applied', 'Reapplied')
                    AND u.territory_id = ?`;
            params.push(territory_id);
        } else {
            query += ` WHERE sad.salesactivities_id IN (3, 10)
                    AND sad.claim_status IN ('Applied', 'Reapplied')`;
        }
    }
    else if (role === 2) {
        // console.log("Sales Executive");
        query += `SELECT sad.* 
                FROM salesactivitydatas sad
                INNER JOIN users u ON sad.owner_id = u.id 
                INNER JOIN territories t ON u.territory_id = t.id
                WHERE sad.salesactivities_id IN (3, 10)
                AND sad.claim_status IN ('Applied', 'Reapplied')
                AND t.id = (SELECT territory_id FROM users WHERE id = ?)`;
        params.push(user_id);
    } else if (role === 8) {
        // console.log("Manager");
        query += `SELECT sad.* 
                FROM salesactivitydatas sad
                INNER JOIN users u ON sad.owner_id = u.id`;
        
        if (territory_id) {
            query += ` WHERE sad.salesactivities_id IN (3, 10)
                    AND sad.claim_status IN ('Manager Approved')
                    AND u.territory_id = ?`;
            params.push(territory_id);
        } else {
            query += ` WHERE sad.salesactivities_id IN (3, 10)
                    AND sad.claim_status IN ('Manager Approved')`;
        }
    }

    // Add date filtering if dates are provided
    if (start_date && end_date) {
        query += ` AND DATE(sad.start_date) BETWEEN ? AND ?`;
        params.push(start_date, end_date);
    }

    if (user_id) {
        query += ` AND sad.owner_id = ?`;
        params.push(user_id);
    }

    // Add ordering
    query += ` ORDER BY sad.id DESC `;

    const [rows] = await db.execute(query, params);
    return rows;
}

async function updateclaim(id, status, remarks) {
    try { 
        const [result] = await db.execute(
            `UPDATE salesactivitydatas SET 
                claim_status = ?,
                remarks = ?,
                updated_at = NOW()
            WHERE id = ?`,
            [status, remarks, id]
        );
        return result.affectedRows > 0;
    } catch (error) {
        console.error("Error updating claim:", error.message);
        throw error;
    }
}

async function updateBulkClaims(claimIds, status, remarks) {
    try {
        if (!Array.isArray(claimIds) || claimIds.length === 0) {
            throw new Error('Claim IDs array is required and cannot be empty');
        }

        // Convert undefined values to null to prevent MySQL binding errors
        const safeRemarks = remarks !== undefined ? remarks : null;
        const safeStatus = status !== undefined ? status : null;

        // Create placeholders for the IN clause
        const placeholders = claimIds.map(() => '?').join(',');
        
        const [result] = await db.execute(
            `UPDATE salesactivitydatas SET 
                claim_status = ?,
                remarks = ?,
                updated_at = NOW()
            WHERE id IN (${placeholders})`,
            [safeStatus, safeRemarks, ...claimIds]
        );
        
        return {
            success: result.affectedRows > 0,
            affectedRows: result.affectedRows,
            totalClaims: claimIds.length,
            updatedClaims: result.affectedRows
        };
    } catch (error) {
        console.error("Error updating bulk claims:", error.message);
        throw error;
    }
}


// ConvayanceChanges

async function getConvayanceChanges() {
    const [rows] = await db.execute(
        `SELECT * FROM conveyancecharges`
    );
    return rows;
}

// update conveyance changes
async function updateConvayanceChanges(id, conveyance_charge) {
    try {
        // console.log("Updating conveyance charge:", id, conveyance_charge);
        const [result] = await db.execute(
            `UPDATE conveyancecharges SET 
                conveyance_charge = ?,
                updated_on = NOW(),
                updated_at = NOW()
            WHERE id = ?`,
            [conveyance_charge, id]
        );
        // console.log(result);
        // console.log("Update successful:", result);
        return result;
    } catch (error) {
        console.error("Error updating conveyance charge:", error.message);
        throw error;
    }
}

async function softDeleteSalesActivityData (id) {
    const [result] = await db.execute('UPDATE salesactivitydatas SET active = 0 WHERE id = ?', [id]);
    return result.affectedRows > 0;
}

// Sales Activity All data 
async function getAllSalesActivityWithData(start_date = null, end_date = null) {
    try {
        // First get all active sales activities
        const [activities] = await db.execute(
            'SELECT id, activity_name, activity_icon FROM salesactivities WHERE active = 1'
        );

        // Get all fields from salesactivitydatas for each activity
        const result = [];

        for (const activity of activities) {
            let query = `
                SELECT 
                    sad.*,
                    u.name as owner_name,
                    COALESCE(sao.outcome, '') as outcome_name,
                    CASE 
                        WHEN sad.targetable_type = 'contacts' THEN CONCAT(c.first_name, ' ', c.last_name)
                        WHEN sad.targetable_type = 'accounts' THEN ac.name
                        WHEN sad.targetable_type = 'deals' THEN d.name
                        ELSE NULL
                    END as target_name
                FROM salesactivitydatas sad
                LEFT JOIN users u ON sad.owner_id = u.id
                LEFT JOIN salesactivityoutcomes sao ON sad.outcome = sao.id
                LEFT JOIN contacts c ON sad.targetable_type = 'contacts' AND sad.targetable_id = c.id
                LEFT JOIN accounts ac ON sad.targetable_type = 'accounts' AND sad.targetable_id = ac.id
                LEFT JOIN deals d ON sad.targetable_type = 'deals' AND sad.targetable_id = d.id
                WHERE sad.salesactivities_id = ? AND sad.is_deleted = 0`;

            const queryParams = [activity.id];

            // Add date filtering if dates are provided
            if (start_date && end_date) {
                query += ` AND DATE(sad.start_date) BETWEEN ? AND ?`;
                queryParams.push(start_date, end_date);
            }

            // Add ordering
            query += ` ORDER BY sad.start_date DESC, sad.start_time DESC`;

            const [activityData] = await db.execute(query, queryParams);

            const formattedData = {
                activity_name: activity.activity_name,
                activity_icon: activity.activity_icon,
                data: activityData.map(row => ({
                    id: row.id,
                    salesactivities_id: row.salesactivities_id,
                    title: row.title,
                    activity_type: row.activity_type,
                    activity_date: formatDate(row.activity_date),
                    description: row.description,
                    activity_title: row.activity_title,
                    schedule: {
                        start_date: formatDate(row.start_date),
                        end_date: formatDate(row.end_date),
                        start_time: formatTime(row.start_time),
                        end_time: formatTime(row.end_time)
                    },
                    duration: row.duration,
                    outcome: {
                        id: row.outcome,
                        name: row.outcome_name
                    },
                    status: row.status,
                    priority: row.priority,
                    location: row.location,
                    owner: {
                        id: row.owner_id,
                        name: row.owner_name
                    },
                    target: {
                        type: row.targetable_type,
                        id: row.targetable_id,
                        name: row.target_name
                    },
                    travel: {
                        transport_mode: row.transport_mode,
                        distance: row.distance,
                        claim_amount: row.claim_amount
                    },
                    check_points: {
                        checkin: {
                            time: formatDateTime(row.checkin_time),
                            location: row.checkin_location,
                            coordinates: {
                                latitude: row.checkin_latitude,
                                longitude: row.checkin_longitude
                            }
                        },
                        checkout: {
                            time: formatDateTime(row.checkout_time),
                            location: row.checkout_location,
                            coordinates: {
                                latitude: row.checkout_latitude,
                                longitude: row.checkout_longitude
                            }
                        }
                    },
                    time_tracking: {
                        timespent: row.timespent,
                        timespent_display: row.timespent_display
                    },
                    custom_field: row.custom_field ? JSON.parse(row.custom_field) : null,
                    permissions: {
                        allow_reschedule: row.allow_reschedule,
                        allow_complete: row.allow_complete
                    },
                    active: row.active,
                    timestamps: {
                        created_at: formatDateTime(row.created_at),
                        updated_at: formatDateTime(row.updated_at)
                    },
                    start_date: formatDate(row.start_date),
                    end_date: formatDate(row.end_date)
                }))
            };

            result.push(formattedData);
        }

        return {
            total_activities: activities.length,
            activities: result
        };

    } catch (error) {
        console.error('Error in getAllSalesActivityWithData:', error);
        throw new Error(`Failed to fetch sales activities with data: ${error.message}`);
    }
}

// get all meetings by id
async function getAllMeetingsById(id) {
    try {
        // console.log('Fetching meeting with ID:', id);
        
        // First get the meeting details
        const [meetingRows] = await db.execute(`
            SELECT 
                sad.*,
                u.name as owner_name,
                CASE 
                    WHEN sad.targetable_type = 'contacts' THEN CONCAT(c.first_name, ' ', c.last_name)
                    WHEN sad.targetable_type = 'accounts' THEN ac.name
                    WHEN sad.targetable_type = 'deals' THEN d.name
                    ELSE NULL
                END as target_name
            FROM salesactivitydatas sad
            LEFT JOIN users u ON sad.owner_id = u.id
            LEFT JOIN contacts c ON sad.targetable_type = 'contacts' AND sad.targetable_id = c.id
            LEFT JOIN accounts ac ON sad.targetable_type = 'accounts' AND sad.targetable_id = ac.id
            LEFT JOIN deals d ON sad.targetable_type = 'deals' AND sad.targetable_id = d.id
            WHERE sad.id = ?`,
            [id]
        );
        
        // console.log('Query result:', meetingRows);

        if (meetingRows.length === 0) {
            // console.log('No meeting found with ID:', id);
            return null;
        }

        const meeting = meetingRows[0];
        // console.log('Raw meeting data:', meeting);

        // Get attendee details separately
        if (meeting.attendees) {
            const attendeeIds = meeting.attendees.split(',').map(id => id.trim());
            if (attendeeIds.length > 0) {
                const placeholders = attendeeIds.map(() => '?').join(',');
                const [attendeeRows] = await db.execute(`
                    SELECT id, name, email, mobile
                    FROM users
                    WHERE id IN (${placeholders})`,
                    attendeeIds
                );
                meeting.attendee_details = attendeeRows;
            } else {
                meeting.attendee_details = [];
            }
        } else {
            meeting.attendee_details = [];
        }

        // Format dates and times
        // try {
        //     meeting.start_date = formatDate(meeting.start_date);
        //     meeting.end_date = formatDate(meeting.end_date);
        //     meeting.start_time = formatTime(meeting.start_time);
        //     meeting.end_time = formatTime(meeting.end_time);
        //     meeting.checkin_time = formatDateTime(meeting.checkin_time);
        //     meeting.checkout_time = formatDateTime(meeting.checkout_time);
        // } catch (formatError) {
        //     console.error('Error formatting dates:', formatError);
        // }

        // Structure target information
        meeting.target = {
            type: meeting.targetable_type,
            id: meeting.targetable_id,
            name: meeting.target_name
        };

        // Remove the raw fields
        delete meeting.targetable_type;
        delete meeting.targetable_id;
        delete meeting.target_name;
        
        // console.log('Processed meeting data:', meeting);
        return meeting;
    } catch (error) {
        console.error('Detailed error in getAllMeetingsById:', error);
        throw new Error(`Error fetching meeting details: ${error.message}`);
    }
}

async function getOwnerAccessToken(ownerId) {
    // This function should implement the logic to retrieve the owner's access token
    // For example, you might query a database or use a token management service
    // Here's a placeholder implementation
    try {
        // Assuming you have a function or service to get the token
        const token = await fetchTokenFromDatabase(ownerId);
        return token;
    } catch (error) {
        console.error('Error fetching owner access token:', error);
        return null;
    }
}

// Placeholder function to simulate fetching a token from a database
async function fetchTokenFromDatabase(ownerId) {
    // Implement the logic to fetch the token from your database or token management service
    // This is a placeholder and should be replaced with actual implementation
    return 'your_access_token_here';
}

// get all related to module
async function getActivitiesByModule(id, module) {
    try {
    const modules = module.toUpperCase();
    const [meetings] = await db.execute(`
        SELECT * FROM salesactivitydatas
        WHERE targetable_type = ? AND targetable_id = ? AND salesactivities_id = 3
    `, [modules, id]);

    const [tasks] = await db.execute(`
        SELECT * FROM tasks
        WHERE related_type = ? AND related_to = ? 
    `, [modules, id]);

    const [notes] = await db.execute(`
        SELECT * FROM notes
        WHERE belongsto = ? AND belongsto_id = ? 
    `, [modules, id]);


    return {
        meetings,
        tasks,
        notes
    };

    } catch (error) {
        console.error('Error fetching activities by module:', error);
        throw new Error(`Error fetching activities by module: ${error.message}`);
    }
}

async function getinprogress(userId) {
    try {
        const [activities] = await db.execute(`
            SELECT * FROM salesactivitydatas
            WHERE owner_id = ? AND status = 'Inprogress' AND salesactivities_id = 3 
        `, [userId]);
        return activities;
    } catch (error) {
        console.error('Error fetching in progress activities:', error);
        throw new Error(`Error fetching in progress activities: ${error.message}`);
    }    
}

async function softCheckout(id, updateData = {}, socket = false) {
    console.log('softCheckout function called with id:', id, 'updateData:', updateData, 'socket:', socket);

    try {
        const [activityData] = await db.execute(
            'SELECT checkin_time, salesactivities_id, checkin_latitude, checkin_longitude, notes, transport_mode FROM salesactivitydatas WHERE id = ?',
            [id]
        );

        const activity = activityData[0];
        if (!activity) {
            return { success: false, message: "Activity not found" };
        }

        const shouldCheckDistance = activity.salesactivities_id === 10 || (activity.salesactivities_id === 3 && socket);
        if (shouldCheckDistance) {
            const origin = {
                lat: parseFloat(activity.checkin_latitude),
                lng: parseFloat(activity.checkin_longitude)
            };
            // console.log({origin})

            const destination = {
                lat: updateData.softcheckout_lat || activity.checkin_latitude,
                lng: updateData.softcheckout_long || activity.checkin_longitude
            };
            // console.log({destination})

            // console.log("origin:", origin);
            // console.log("destination:", destination);

            const distanceData = await calculateDistance(origin, destination);
            // console.log("distanceData:", distanceData);

            if (distanceData) {
                const distanceInKm = distanceData.distanceValue / 1000;
                console.log("distanceInKm:", distanceInKm);

                if (distanceInKm > 2 ) {
                    console.log("Distance is greater than 2 km, updating soft checkout");
                    updateData.distance = distanceInKm;
                    updateData.softcheckout_lat = updateData.softcheckout_lat;
                    updateData.softcheckout_long = updateData.softcheckout_long;
                    updateData.soft_checkout = 1;
                    updateData.soft_datetime = new Date();

                    const keys = Object.keys(updateData);
                    const values = Object.values(updateData);
                    const setClause = keys.map(key => `${key} = ?`).join(', ');

                    // console.log(`UPDATE salesactivitydatas SET ${setClause} WHERE id = ?`, [...values, id]);

                    const [result] = await db.execute(
                        `UPDATE salesactivitydatas SET ${setClause} WHERE id = ?`,
                        [...values, id]
                    );

                    return {
                        success: result.affectedRows > 0,
                        message: 'Soft checkout updated with distance',
                        distance: distanceInKm
                    };
                } else if ( activity.salesactivities_id === 10 && distanceInKm == 0 ) {
                    console.log("Distance is greater than 2 km, updating soft checkout");
                    updateData.distance = distanceInKm;
                    // updateData.checkout_latitude = destination.lat;
                    // updateData.softcheckout_long = destination.lng;

                    const keys = Object.keys(updateData);
                    const values = Object.values(updateData);
                    const setClause = keys.map(key => `${key} = ?`).join(', ');

                    // console.log(`UPDATE salesactivitydatas SET ${setClause} WHERE id = ?`, [...values, id]);

                    const [result] = await db.execute(
                        `UPDATE salesactivitydatas SET ${setClause} WHERE id = ?`,
                        [...values, id]
                    );

                    return {
                        success: result.affectedRows > 0,
                        message: 'Attendance Closed Successfully',
                        distance: distanceInKm
                    };
                } else {
                    

                    return {
                        success: false,
                        message: 'Distance is too short to update soft checkout',
                        distance: distanceInKm
                    };
                }
            } else {
                return {
                    success: false,
                    message: 'Unable to calculate distance'
                };
            }
        }

        return {
            success: false,
            message: 'No distance check required based on activity type or socket'
        };

    } catch (error) {
        console.error("Error in softCheckout:", error);
        return {
            success: false,
            message: 'Internal server error',
            error: error.message
        };
    }
}



module.exports = {
    createSalesActivity,
    getSalesActivities,
    updateSalesActivity,
    createSalesActivityData,
    getAllSalesActivityData,
    getSalesActivityData,
    updateSalesActivityData,
    softDeleteSalesActivity,
    updateActivityStatus,
    getAttendanceHistory,
    getAllAttendanceHistory,
    getConveyanceList,
    getconveyance,
    softDeleteSalesActivityData,
    getallclaims,
    updateclaim,
    getConvayanceChanges,
    updateConvayanceChanges,
    getAllSalesActivityWithData,
    getAllMeetings,
    getAllMeetingsById,
    updateMeetingStatus,
    getinprogress,  
    softCheckout,
    getActivitiesByModule,
    updateBulkClaims
};