const db = require('../config/db');
const { Client } = require('@googlemaps/google-maps-services-js');
const googleMapsClient = new Client({});

async function createSalesActivity(activityData) {
    const connection = await db.getConnection();
    
    try {
        await connection.beginTransaction();

        // Validate required fields
        if (!activityData.activity_name) {
            throw new Error('Activity name is required');
        }

        // Insert into salesactivities table
        const [result] = await connection.execute(
            `INSERT INTO salesactivities (
                activity_id, 
                activity_name, 
                activity_icon, 
                allow_checkin_checkout, 
                show_calendar, 
                mark_completed, 
                allow_edit, 
                active, 
                custom_activity, 
                created_on,
                created_at
            ) 
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, NOW(), NOW())`,
            [
                activity_id = null,
                activityData.activity_name,
                "/sales/" + activityData.activity_icon || null,
                activityData.allow_checkin_checkout || 0,
                activityData.show_calendar || 0,
                activityData.mark_completed || 0,
                activityData.allow_edit || 1,
                activityData.active || 1,
                activityData.custom_activity || 0
            ]
        );

        const salesActivityId = result.insertId;

        // Insert outcomes if provided
        if (activityData.outcomes && Array.isArray(activityData.outcomes)) {
            for (const outcome of activityData.outcomes) {
                await connection.execute(
                    `INSERT INTO salesactivityoutcomes (
                        salesactivities_id,
                        outcome,
                        created_at,
                        updated_at
                    ) VALUES (?, ?, NOW(), NOW())`,
                    [
                        salesActivityId,
                        outcome
                    ]
                );
            }
        }

        await connection.commit();

        return {
            id: salesActivityId,
            ...result,
            outcomes_count: activityData.outcomes ? activityData.outcomes.length : 0
        };

    } catch (error) {
        await connection.rollback();
        throw error;
    } finally {
        connection.release();
    }
}

async function getAllSalesActivityData(id) {
    const [result] = await db.execute(
        'SELECT * FROM salesactivitydatas WHERE owner_id = ? AND salesactivities_id = 3',
        [id]
    );
    return result;
 }

async function getSalesActivities() {
    const [rows] = await db.execute('SELECT * FROM salesactivities');
    return rows;
}

async function createSalesActivityData(Data) {
    console.log(Data);
    Data.status = 'Pending';
    Object.keys(Data).forEach(key => {
        if (Data[key] === undefined || Data[key] === '' || Data[key] === null) {
            delete Data[key];
        }
    });
    const keys =  Object.keys(Data);
    const values = Object.values(Data);
    const placeholders = '?' + ', ?'.repeat(values.length - 1);
   const [result] = await db.execute(
       `INSERT INTO salesactivitydatas (${keys}) VALUES (${placeholders})`,
       values
   );
   return result.insertId;
}

async function getAllSalesActivityData(id) {
   const [result] = await db.execute(
       'SELECT * FROM salesactivitydatas WHERE owner_id = ? AND salesactivities_id = 3 ORDER BY id DESC',
       [id]
   );
   return result;
}
async function getSalesActivityData(id) {
    // First get the sales activity data
    const [activityData] = await db.execute(
        'SELECT salesactivitydatas.*, ' +
        'owner.name as owner_name, ' +
        'creator.name as creator_name, ' +
        'updater.name as updater_name ' +
        'FROM salesactivitydatas ' +
        'LEFT JOIN users as owner ON salesactivitydatas.owner_id = owner.id ' +
        'LEFT JOIN users as creator ON salesactivitydatas.creater_id = creator.id ' +
        'LEFT JOIN users as updater ON salesactivitydatas.updater_id = updater.id ' +
        'WHERE salesactivitydatas.id = ?',
        [id]
    );

    if (!activityData[0]) {
        return null;
    }

    const activity = activityData[0];

    // Get attendees details
    if (activity.attendees) {
        const attendeeIds = activity.attendees.split(',').map(id => parseInt(id.trim()));
        const [attendees] = await db.execute(
            'SELECT id, name FROM users WHERE id IN (?)',
            [attendeeIds]
        );
        activity.attendees_details = attendees;
    }

    // Get target details based on targetable_type
    if (activity.targetable_id && activity.targetable_type) {
        let targetQuery;
        switch (activity.targetable_type.toLowerCase()) {
            case 'contacts':
                targetQuery = 'SELECT id, first_name, last_name, emails FROM contacts WHERE id = ?';
                break;
            case 'accounts':
                targetQuery = 'SELECT id, name, website FROM accounts WHERE id = ?';
                break;
            case 'deals':
                targetQuery = 'SELECT id, name, amount FROM deals WHERE id = ?';
                break;
            default:
                targetQuery = null;
        }

        if (targetQuery) {
            const [targetData] = await db.execute(targetQuery, [activity.targetable_id]);
            if(activity.targetable_type.toLowerCase() === 'contacts'){
                targetData.forEach(contact => {
                    contact.name = contact.first_name + ' ' + contact.last_name;
                });
            }
            activity.target_details = targetData[0] || null;
        }
    }

    return activity;
}
async function updateSalesActivityData(Data, id) {
    // Remove undefined or empty values
    Object.keys(Data).forEach(key => {
        if (Data[key] === undefined || Data[key] === '' || Data[key] === null) {
            delete Data[key];
        }
    });

    // Prepare update fields and values
    const updateFields = Object.keys(Data)
        .map(key => `${key} = ?`)
        .join(', ');
    
    const updateValues = Object.values(Data);
    
    const [result] = await db.execute(
        `UPDATE salesactivitydatas SET ${updateFields} WHERE id = ?`,
        [...updateValues, id]
    );

     return result;
}
async function updateSalesActivity(id, data) {
    const connection = await db.getConnection();
    
    try {
        await connection.beginTransaction();

        // Validate required fields
        if (!id) {
            throw new Error('Activity ID is required for update');
        }

        // Update salesactivities table
        const [result] = await connection.execute(
            `UPDATE salesactivities 
             SET activity_id = ?, 
                 activity_name = ?, 
                 activity_icon = ?, 
                 allow_checkin_checkout = ?, 
                 show_calendar = ?, 
                 mark_completed = ?, 
                 allow_edit = ?, 
                 active = ?, 
                 custom_activity = ?,
                 updated_at = NOW()
             WHERE id = ?`,
            [
                activity_id = null,
                data.activity_name,
                "/sales/" + data.activity_icon || null,
                data.allow_checkin_checkout || 0,
                data.show_calendar || 0,
                data.mark_completed || 0,
                data.allow_edit || 1,
                data.active || 1,
                data.custom_activity || 0,
                id
            ]
        );

        // If outcomes are provided, update them
        if (data.outcomes && Array.isArray(data.outcomes)) {
            // First, delete existing outcomes
            await connection.execute(
                'DELETE FROM salesactivityoutcomes WHERE salesactivities_id = ?',
                [id]
            );

            // Then insert new outcomes
            for (const outcome of data.outcomes) {
                await connection.execute(
                    `INSERT INTO salesactivityoutcomes (
                        salesactivities_id,
                        outcome,
                        created_at,
                        updated_at
                    ) VALUES (?, ?, NOW(), NOW())`,
                    [
                        id,
                        outcome
                    ]
                );
            }
        }

        await connection.commit();

        return {
            id: id,
            ...result,
            outcomes_count: data.outcomes ? data.outcomes.length : 0
        };

    } catch (error) {
        await connection.rollback();
        throw error;
    } finally {
        connection.release();
    }
}

async function softDeleteSalesActivity(id) {
    await db.execute('UPDATE salesactivities SET active = 0 WHERE id = ?', [id]);
}
async function calculateDistance(origin, destination) {
    try {
        const response = await googleMapsClient.distancematrix({
            params: {
                origins: [origin],
                destinations: [destination],
                key: process.env.GOOGLE_MAPS_API_KEY
            }
        });

        if (response.data.rows[0].elements[0].status === 'OK') {
            return {
                distance: response.data.rows[0].elements[0].distance.text,
                distanceValue: response.data.rows[0].elements[0].distance.value, // in meters
                duration: response.data.rows[0].elements[0].duration.text
            };
        }
        return null;
    } catch (error) {
        console.error('Error calculating distance:', error);
        return null;
    }
}

async function getPreviousActivity(userId, currentDate) {
    const [result] = await db.execute(
        `SELECT checkin_latitude, checkin_longitude 
         FROM salesactivitydatas 
         WHERE owner_id = ? 
         AND DATE(checkin_time) = DATE(?)
         AND status != 'Pending'
         ORDER BY checkin_time DESC 
         LIMIT 1`,
        [userId, currentDate]
    );
    return result[0];
}

async function getDefaultTransport(userId) {
    const [userRole] = await db.execute(
        `SELECT r.default_transport 
         FROM users u 
         JOIN roles r ON u.roles_id = r.id 
         WHERE u.id = ?`,
        [userId]
    );
    return userRole[0]?.default_transport;
}

async function getConveyanceCharge(transportMode) {
    const [charge] = await db.execute(
        'SELECT conveyance_charge FROM conveyanceconfigs WHERE transport_mode = ?',
        [transportMode]
    );
    return charge[0]?.conveyance_charge || 0;
}

async function updateActivityStatus(id, data, userId) {
    console.log(id);
    const [currentStatus] = await db.execute(
        'SELECT status FROM salesactivitydatas WHERE id = ?',
        [id]
    );

    if (!currentStatus[0]) {
        throw new Error('Activity not found');
    }

    const status = currentStatus[0].status;
    const updateData = {};

    if (status === 'Pending') {
        // Get previous activity location
        const previousActivity = await getPreviousActivity(userId, new Date());
        
        // Update for check-in
        updateData.status = 'Inprogress';
        updateData.checkin_latitude = data.latitude;
        updateData.checkin_location = data.location;
        updateData.checkin_longitude = data.longitude;
        updateData.check_in_attachment = data.attachment;
        updateData.checkin_time = new Date();

        // Calculate distance if previous activity exists
        if (previousActivity) {
            const origin = {
                lat: parseFloat(previousActivity.checkin_latitude),
                lng: parseFloat(previousActivity.checkin_longitude)
            };
            const destination = {
                lat: parseFloat(data.latitude),
                lng: parseFloat(data.longitude)
            };

            const distanceData = await calculateDistance(origin, destination);
            if (distanceData) {
                const distanceInKm = distanceData.distanceValue/1000;
                updateData.distance = distanceInKm;

                // Get default transport mode and conveyance charge
                const defaultTransport = await getDefaultTransport(userId);
                const conveyanceCharge = await getConveyanceCharge(defaultTransport);

                // Calculate and store claim amount
                updateData.transport_mode = defaultTransport;
                updateData.conveyance_charge = conveyanceCharge;
                updateData.claim_amount = distanceInKm * conveyanceCharge;
            }
        }
    } else if (status === 'Inprogress') {
        // Get check-in time and activity details
        const [activityData] = await db.execute(
            'SELECT checkin_time, salesactivities_id, checkin_latitude, checkin_longitude, notes, transport_mode FROM salesactivitydatas WHERE id = ?',
            [id]
        );
        
        const checkInTime = new Date(activityData[0].checkin_time);
        const checkOutTime = new Date();
        
        // Calculate time spent
        const timeSpentSeconds = Math.floor((checkOutTime - checkInTime) / 1000);
        const hours = Math.floor(timeSpentSeconds / 3600);
        const minutes = Math.floor((timeSpentSeconds % 3600) / 60);
        const seconds = timeSpentSeconds % 60;
        const timeSpentDisplay = `${hours.toString().padStart(2, '0')}:${minutes.toString().padStart(2, '0')}:${seconds.toString().padStart(2, '0')}`;
        
        // Set basic checkout data
        updateData.status = 'Closed';
        updateData.checkout_latitude = data.latitude;
        updateData.checkout_location = data.location;
        updateData.checkout_longitude = data.longitude;
        updateData.checkout_time = checkOutTime;
        updateData.timespent = timeSpentSeconds;
        updateData.check_out_attachment = data.attachment;
        updateData.timespent_display = timeSpentDisplay;
        updateData.updater_id = userId;
        updateData.notes = data.remarks;
        // For salesactivities_id = 10, calculate distance and claim amount
        if (activityData[0].salesactivities_id === 10) {
            const origin = {
                lat: parseFloat(activityData[0].checkin_latitude),
                lng: parseFloat(activityData[0].checkin_longitude)
            };
            const destination = {
                lat: parseFloat(data.latitude),
                lng: parseFloat(data.longitude)
            };

            const distanceData = await calculateDistance(origin, destination);
            if (distanceData) {
                const distanceInKm = distanceData.distanceValue/1000;
                updateData.distance = distanceInKm;

                // Get default transport mode and conveyance charge
                const defaultTransport = await getDefaultTransport(userId);
                const conveyanceCharge = await getConveyanceCharge(data.transport_mode?data.transport_mode:defaultTransport);

                // Calculate and store claim amount
                updateData.transport_mode = data.transport_mode?data.transport_mode:defaultTransport;
                updateData.conveyance_charge = conveyanceCharge;
                updateData.claim_amount = distanceInKm * conveyanceCharge;
            }
        }
    } else {
        throw new Error('Invalid status for update');
    }

    const keys = Object.keys(updateData);
    const values = Object.values(updateData);
    
    const setClause = keys.map(key => `${key} = ?`).join(', ');
    
    const [result] = await db.execute(
        `UPDATE salesactivitydatas SET ${setClause} WHERE id = ?`,
        [...values, id]
    );

    return {
        success: result.affectedRows > 0,
        newStatus: updateData.status
    };
}
async function getAttendanceHistory(userId, filter = 'today', month = null, year = null) {
    let query = `
        SELECT sad.*, 
               u.name as owner_name,
               cu.name as updater_name
        FROM salesactivitydatas sad
        LEFT JOIN users u ON sad.owner_id = u.id
        LEFT JOIN users cu ON sad.updater_id = cu.id
        WHERE sad.salesactivities_id = 10 
        AND sad.owner_id = ?`;
    
    const params = [userId];

    // Add date filters
    switch (filter) {
        case 'today':
            query += ' AND DATE(sad.start_date) = CURDATE()';
            break;
        
        case 'weekly':
            query += ' AND YEARWEEK(sad.start_date, 1) = YEARWEEK(CURDATE(), 1)';
            break;
        
        case 'monthly':
            if (!month || !year) {
                throw new Error('Month and year are required for monthly filter');
            }
            query += ' AND MONTH(sad.start_date) = ? AND YEAR(sad.start_date) = ?';
            params.push(month, year);
            break;

        default:
            throw new Error('Invalid filter type');
    }

    query += ' ORDER BY sad.start_date DESC';

    const [rows] = await db.execute(query, params);



    // Format the response
    return {
        filter_type: filter,
        period: getPeriodLabel(filter, month, year),
        attendance_data: rows.map(row => ({
            id: row.id,
            date: formatDate(row.start_date),
            check_in_time: formatTime(row.checkin_time),
            check_in_location: row.checkin_location,
            check_out_time: formatTime(row.checkout_time),
            check_out_location: row.checkout_location,
            status: row.status,
            time_spent: row.timespent_display,
            owner_name: row.owner_name,
            updater_name: row.updater_name,
            distance: row.distance || 0,
            claim_amount: row.claim_amount || 0,
            transport_mode: row.transport_mode
        }))
    };
}

// Helper functions for date formatting
function formatDate(date) {
    if (!date) return null;
    return new Date(date).toLocaleDateString('en-GB', {
        day: '2-digit',
        month: '2-digit',
        year: 'numeric'
    });
}

function formatTime(time) {
    if (!time) return null;
    return new Date(time).toLocaleTimeString('en-US', {
        hour: '2-digit',
        minute: '2-digit',
        hour12: true
    });
}


function formatDateTime(datetime) {
    if (!datetime) return null;
    const date = new Date(datetime);
    return date.toLocaleString('en-GB', {
        day: '2-digit',
        month: '2-digit',
        year: 'numeric',
        hour: '2-digit',
        minute: '2-digit',
        second: '2-digit',
        hour12: false
    });
}

function getPeriodLabel(filter, month, year) {
    switch (filter) {
        case 'today':
            return new Date().toLocaleDateString('en-GB', {
                day: '2-digit',
                month: '2-digit',
                year: 'numeric'
            });
        
        case 'weekly':
            const now = new Date();
            const startOfWeek = new Date(now.setDate(now.getDate() - now.getDay()));
            const endOfWeek = new Date(now.setDate(now.getDate() - now.getDay() + 6));
            return `${formatDate(startOfWeek)} to ${formatDate(endOfWeek)}`;
        
        case 'monthly':
            return new Date(year, month - 1).toLocaleDateString('en-GB', {
                month: 'long',
                year: 'numeric'
            });
            
        default:
            return '';
    }
}

// attendance history for all users
async function getAllAttendanceHistory(filter = 'today', month = null, year = null, start_date = null, end_date = null) {
    let query = `
        SELECT sad.*, 
               u.name as owner_name,
               cu.name as updater_name
        FROM salesactivitydatas sad
        LEFT JOIN users u ON sad.owner_id = u.id
        LEFT JOIN users cu ON sad.updater_id = cu.id
        WHERE sad.salesactivities_id = 10`;
    
    const params = [];

    // Add date filters
    switch (filter) {
        case 'date_range':
            query += ' AND DATE(sad.start_date) BETWEEN ? AND ?';
            params.push(start_date, end_date);
            break;

        case 'today':
            query += ' AND DATE(sad.start_date) = CURDATE()';
            break;
        
        case 'weekly':
            query += ' AND YEARWEEK(sad.start_date, 1) = YEARWEEK(CURDATE(), 1)';
            break;
        
        case 'monthly':
            if (!month || !year) {
                throw new Error('Month and year are required for monthly filter');
            }
            query += ' AND MONTH(sad.start_date) = ? AND YEAR(sad.start_date) = ?';
            params.push(month, year);
            break;

        default:
            throw new Error('Invalid filter type');
    }

    query += ' ORDER BY sad.start_date DESC, u.name ASC';

    const [rows] = await db.execute(query, params);

    // Format the response
    return {
        filter_type: filter,
        period: getPeriodLabel(filter, month, year, start_date, end_date),
        attendance_data: rows.map(row => ({
            id: row.id,
            date: formatDate(row.start_date),
            check_in_time: formatTime(row.checkin_time),
            check_in_location: row.checkin_location,
            check_out_time: formatTime(row.checkout_time),
            check_out_location: row.checkout_location,
            status: row.status,
            time_spent: row.timespent_display,
            owner_name: row.owner_name,
            updater_name: row.updater_name,
            distance: row.distance || 0,
            claim_amount: row.claim_amount || 0,
            transport_mode: row.transport_mode
        }))
    };
}

// Update getPeriodLabel function to handle date range
function getPeriodLabel(filter, month, year, start_date, end_date) {
    switch (filter) {
        case 'date_range':
            return `${formatDate(start_date)} to ${formatDate(end_date)}`;
            
        case 'today':
            return new Date().toLocaleDateString('en-GB', {
                day: '2-digit',
                month: '2-digit',
                year: 'numeric'
            });
        
        case 'weekly':
            const now = new Date();
            const startOfWeek = new Date(now.setDate(now.getDate() - now.getDay()));
            const endOfWeek = new Date(now.setDate(now.getDate() - now.getDay() + 6));
            return `${formatDate(startOfWeek)} to ${formatDate(endOfWeek)}`;
        
        case 'monthly':
            return new Date(year, month - 1).toLocaleDateString('en-GB', {
                month: 'long',
                year: 'numeric'
            });
            
        default:
            return '';
    }
}


async function getConveyanceList(userId, filter = 'today', month = null, year = null) {
    let query = `SELECT * FROM salesactivitydatas 
                 WHERE owner_id = ? 
                 AND salesactivities_id IN(3,10)`;
    
    const params = [userId];

    // Add date filters
    switch (filter) {
        case 'today':
            query += ' AND DATE(start_date) = CURDATE()';
            break;
        
        case 'weekly':
            query += ' AND YEARWEEK(start_date, 1) = YEARWEEK(CURDATE(), 1)';
            break;
        
        case 'monthly':
            if (!month || !year) {
                throw new Error('Month and year are required for monthly filter');
            }
            query += ' AND MONTH(start_date) = ? AND YEAR(start_date) = ?';
            params.push(month, year);
            break;

        default:
            throw new Error('Invalid filter type');
    }

    query += ' ORDER BY start_date ASC';
    
    const [activities] = await db.execute(query, params);

    // Group activities by date
    const groupedActivities = activities.reduce((acc, activity) => {
        const date = new Date(activity.start_date);
        const formattedDate = date.toLocaleDateString('en-GB', {
            day: '2-digit',
            month: '2-digit',
            year: 'numeric'
        });

        if (!acc[formattedDate]) {
            acc[formattedDate] = {
                conveyance_date: formattedDate,
                distance: "0.00",
                charge: 0,
                activity: []
            };
        }

        acc[formattedDate].activity.push(activity);

        // Sum up distance and charge if available
        if (activity.distance) {
            acc[formattedDate].distance = (parseFloat(acc[formattedDate].distance) + parseFloat(activity.distance)).toFixed(2);
        }
        if (activity.claim_amount) {
            acc[formattedDate].charge += parseFloat(activity.claim_amount);
        }

        return acc;
    }, {});

    // Return formatted response
    return {
        filter_type: filter,
        period: getPeriodLabel(filter, month, year),
        conveyance_data: Object.values(groupedActivities).sort((a, b) => {
            const dateA = a.conveyance_date.split('-').reverse().join('-');
            const dateB = b.conveyance_date.split('-').reverse().join('-');
            return new Date(dateA) - new Date(dateB);
        })
    };
}

// Get conveyance list
async function getconveyance(start_date = null, end_date = null) {
    let query = `
        SELECT 
            *,
            SUM(CAST(distance AS DECIMAL(10,2))) as total_distance,
            SUM(
                TIME_TO_SEC(TIMEDIFF(end_time, start_time))/3600
            ) as total_hours
        FROM salesactivitydatas 
        WHERE salesactivities_id IN (3, 10)`;
    const params = [];

    // Add date filtering if dates are provided
    if (start_date && end_date) {
        query += ` AND DATE(start_date) BETWEEN ? AND ?`;
        params.push(start_date, end_date);
    }

    // Group by to get totals while preserving individual records
    query += ` GROUP BY id ORDER BY id DESC`;

    const [rows] = await db.execute(query, params);

    // Calculate grand totals
    const totals = {
        total_distance: rows.reduce((sum, row) => sum + (parseFloat(row.distance) || 0), 0).toFixed(2),
        total_hours: rows.reduce((sum, row) => {
            const start = new Date(`2000-01-01 ${row.start_time}`);
            const end = new Date(`2000-01-01 ${row.end_time}`);
            return sum + ((end - start) / (1000 * 60 * 60));
        }, 0).toFixed(2)
    };

    return { records: rows, totals };
}


// claimalist
async function getallclaims(start_date = null, end_date = null) {
    let query = `SELECT * FROM salesactivitydatas WHERE salesactivities_id IN (3, 10)`;
    const params = [];

    // Add date filtering if dates are provided
    if (start_date && end_date) {
        query += ` AND DATE(start_date) BETWEEN ? AND ?`;
        params.push(start_date, end_date);
    }

    // Add ordering
    query += ` AND status = 'Applied' ORDER BY id DESC `;

    const [rows] = await db.execute(query, params);
    return rows;
}


// update claim status
async function updateclaim(id, status) {
    console.log(id, status);
    const [result] = await db.execute('UPDATE salesactivitydatas SET status = ? WHERE id = ?', [status, id]);
    console.log(result);
    return result;
}


// ConvayanceChanges

async function getConvayanceChanges() {
    const [rows] = await db.execute(
        `SELECT * FROM conveyancecharges`
    );
    return rows;
}

// update conveyance changes
async function updateConvayanceChanges(id, conveyance_charge) {
    try {
        console.log("Updating conveyance charge:", id, conveyance_charge);
        const [result] = await db.execute(
            `UPDATE conveyancecharges SET 
                conveyance_charge = ?,
                updated_on = NOW(),
                updated_at = NOW()
            WHERE id = ?`,
            [conveyance_charge, id]
        );
        console.log(result);
        console.log("Update successful:", result);
        return result;
    } catch (error) {
        console.error("Error updating conveyance charge:", error.message);
        throw error;
    }
}

async function softDeleteSalesActivityData (id) {
    const [result] = await db.execute('UPDATE salesactivitydatas SET active = 1 WHERE id = ?', [id]);
    return result.affectedRows > 0;
}

// Sales Activity All data 
async function getAllSalesActivityWithData() {
    try {
        // First get all active sales activities
        const [activities] = await db.execute(
            'SELECT id, activity_name, activity_icon FROM salesactivities WHERE active = 1'
        );

        // Get all fields from salesactivitydatas for each activity
        const result = [];

        for (const activity of activities) {
            const query = `
                SELECT 
                    sad.*,
                    u.name as owner_name,
                    COALESCE(sao.outcome, '') as outcome_name,
                    CASE 
                        WHEN sad.targetable_type = 'contacts' THEN CONCAT(c.first_name, ' ', c.last_name)
                        WHEN sad.targetable_type = 'accounts' THEN ac.name
                        WHEN sad.targetable_type = 'deals' THEN d.name
                        ELSE NULL
                    END as target_name
                FROM salesactivitydatas sad
                LEFT JOIN users u ON sad.owner_id = u.id
                LEFT JOIN salesactivityoutcomes sao ON sad.outcome = sao.id
                LEFT JOIN contacts c ON sad.targetable_type = 'contacts' AND sad.targetable_id = c.id
                LEFT JOIN accounts ac ON sad.targetable_type = 'accounts' AND sad.targetable_id = ac.id
                LEFT JOIN deals d ON sad.targetable_type = 'deals' AND sad.targetable_id = d.id
                WHERE sad.salesactivities_id = ? AND sad.is_deleted = 0
                ORDER BY sad.start_date DESC, sad.start_time DESC`;

            const [activityData] = await db.execute(query, [activity.id]);

            const formattedData = {
                activity_name: activity.activity_name,
                activity_icon: activity.activity_icon,
                data: activityData.map(row => ({
                    id: row.id,
                    salesactivities_id: row.salesactivities_id,
                    title: row.title,
                    activity_type: row.activity_type,
                    activity_date: formatDate(row.activity_date),
                    description: row.description,
                    activity_title: row.activity_title,
                    schedule: {
                        start_date: formatDate(row.start_date),
                        end_date: formatDate(row.end_date),
                        start_time: formatTime(row.start_time),
                        end_time: formatTime(row.end_time)
                    },
                    duration: row.duration,
                    outcome: {
                        id: row.outcome,
                        name: row.outcome_name
                    },
                    status: row.status,
                    priority: row.priority,
                    location: row.location,
                    owner: {
                        id: row.owner_id,
                        name: row.owner_name
                    },
                    target: {
                        type: row.targetable_type,
                        id: row.targetable_id,
                        name: row.target_name
                    },
                    travel: {
                        transport_mode: row.transport_mode,
                        distance: row.distance,
                        claim_amount: row.claim_amount
                    },
                    check_points: {
                        checkin: {
                            time: formatDateTime(row.checkin_time),
                            location: row.checkin_location,
                            coordinates: {
                                latitude: row.checkin_latitude,
                                longitude: row.checkin_longitude
                            }
                        },
                        checkout: {
                            time: formatDateTime(row.checkout_time),
                            location: row.checkout_location,
                            coordinates: {
                                latitude: row.checkout_latitude,
                                longitude: row.checkout_longitude
                            }
                        }
                    },
                    time_tracking: {
                        timespent: row.timespent,
                        timespent_display: row.timespent_display
                    },
                    custom_field: row.custom_field ? JSON.parse(row.custom_field) : null,
                    permissions: {
                        allow_reschedule: row.allow_reschedule,
                        allow_complete: row.allow_complete
                    },
                    active: row.active,
                    timestamps: {
                        created_at: formatDateTime(row.created_at),
                        updated_at: formatDateTime(row.updated_at)
                    }
                }))
            };

            result.push(formattedData);
        }

        return {
            total_activities: activities.length,
            activities: result
        };

    } catch (error) {
        console.error('Error in getAllSalesActivityWithData:', error);
        throw new Error(`Failed to fetch sales activities with data: ${error.message}`);
    }
}



module.exports = {
    createSalesActivity,
    getSalesActivities,
    updateSalesActivity,
    createSalesActivityData,
    getAllSalesActivityData,
    getSalesActivityData,
    updateSalesActivityData,
    softDeleteSalesActivity,
    updateActivityStatus,
    getAttendanceHistory,
    getAllAttendanceHistory,
    getConveyanceList,
    getconveyance,
    softDeleteSalesActivityData,
    getallclaims,
    updateclaim,
    getConvayanceChanges,
    updateConvayanceChanges,
    getAllSalesActivityWithData
};