const db = require('../config/db');

class SalesActivityCustomFieldModel {
    static async createField(fieldData, choices = []) {
        const connection = await db.getConnection();
        try {
            await connection.beginTransaction();

            // Validate field data
            if (!fieldData.field_label || !fieldData.field_type) {
                throw new Error('Field label and type are required');
            }

            // Generate field_name if not provided
            const field_name = fieldData.field_name || 
                             fieldData.field_label.toLowerCase().replace(/[^a-z0-9]/g, '_');

            // Insert custom field with proper value escaping
            const [result] = await connection.execute(
                `INSERT INTO salesactivitycustomfields (
                    field_label,
                    field_name,
                    field_type,
                    required,
                    tool_tip,
                    placeholder,
                    active,
                    \`order\`,
                    created_at,
                    updated_at
                ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, NOW(), NOW())`,
                [
                    fieldData.field_label,
                    field_name,
                    fieldData.field_type,
                    fieldData.required ? 1 : 0,
                    fieldData.tool_tip || null,
                    fieldData.placeholder || null,
                    fieldData.active === undefined ? 1 : fieldData.active,
                    fieldData.order || 0
                ]
            );

            const fieldId = result.insertId;

            // Handle choices for dropdown/multiselect
            if (['dropdown', 'multiselect'].includes(fieldData.field_type) && Array.isArray(choices)) {
                for (const choice of choices) {
                    if (!choice.label || !choice.value) continue;
                    
                    await connection.execute(
                        `INSERT INTO salesactivitycustomfieldchoices (
                            field_id,
                            choice_label,
                            choice_value,
                            created_at,
                            updated_at
                        ) VALUES (?, ?, ?, NOW(), NOW())`,
                        [fieldId, choice.label, choice.value]
                    );
                }
            }

            await connection.commit();
            return fieldId;

        } catch (error) {
            await connection.rollback();
            throw error;
        } finally {
            connection.release();
        }
    }

    static async updateField(id, fieldData, choices = []) {
        const connection = await db.getConnection();
        try {
            await connection.beginTransaction();

            // Validate field data
            if (!fieldData.field_label || !fieldData.field_type) {
                throw new Error('Field label and type are required');
            }

            // Generate field_name if not provided
            const field_name = fieldData.field_name || 
                             fieldData.field_label.toLowerCase().replace(/[^a-z0-9]/g, '_');

            // Update custom field
            const [result] = await connection.execute(
                `UPDATE salesactivitycustomfields 
                 SET field_label = ?,
                     field_name = ?,
                     field_type = ?,
                     required = ?,
                     tool_tip = ?,
                     placeholder = ?,
                     active = ?,
                     \`order\` = ?,
                     updated_at = NOW()
                 WHERE id = ?`,
                [
                    fieldData.field_label,
                    field_name,
                    fieldData.field_type,
                    fieldData.required ? 1 : 0,
                    fieldData.tool_tip || null,
                    fieldData.placeholder || null,
                    fieldData.active === undefined ? 1 : fieldData.active,
                    fieldData.order || 0,
                    id
                ]
            );

            // Update choices if field type is dropdown/multiselect
            if (['dropdown', 'multiselect'].includes(fieldData.field_type)) {
                // Delete existing choices
                await connection.execute(
                    'DELETE FROM salesactivitycustomfieldchoices WHERE field_id = ?',
                    [id]
                );

                // Insert new choices
                if (Array.isArray(choices) && choices.length > 0) {
                    for (const choice of choices) {
                        if (!choice.label || !choice.value) continue;

                        await connection.execute(
                            `INSERT INTO salesactivitycustomfieldchoices (
                                field_id,
                                choice_label,
                                choice_value,
                                created_at,
                                updated_at
                            ) VALUES (?, ?, ?, NOW(), NOW())`,
                            [id, choice.label, choice.value]
                        );
                    }
                }
            }

            await connection.commit();
            return result.affectedRows > 0;

        } catch (error) {
            await connection.rollback();
            throw error;
        } finally {
            connection.release();
        }
    }

    static async listFields() {
        const [fields] = await db.execute(
            'SELECT * FROM salesactivitycustomfields WHERE active = 1 ORDER BY `order` ASC'
        );

        // Get choices for dropdown/multiselect fields
        for (const field of fields) {
            if (['dropdown', 'multiselect'].includes(field.field_type)) {
                const [choices] = await db.execute(
                    'SELECT * FROM salesactivitycustomfieldchoices WHERE field_id = ?',
                    [field.id]
                );
                field.choices = choices;
            }
        }

        return fields;
    }

    static async deleteField(id) {
        const connection = await db.getConnection();
        try {
            await connection.beginTransaction();

            // Soft delete the field
            const [result] = await connection.execute(
                'UPDATE salesactivitycustomfields SET active = 0, updated_at = NOW() WHERE id = ?',
                [id]
            );

            await connection.commit();
            return result.affectedRows > 0;

        } catch (error) {
            await connection.rollback();
            throw error;
        } finally {
            connection.release();
        }
    }
}

module.exports = SalesActivityCustomFieldModel;