const db = require('../config/db');

async function createSalesActivityData(data) {
    const { salesactivities_id, activity_type, activity_date, activity_title, start_date, end_date, start_time, end_time, owner_id, creater_id, status } = data;
    const [result] = await db.execute(
        'INSERT INTO salesactivitydatas (salesactivities_id, activity_type, activity_date, activity_title, start_date, end_date, start_time, end_time, owner_id, creater_id, status, created_at) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW())',
        [salesactivities_id, activity_type, activity_date, activity_title, start_date, end_date, start_time, end_time, owner_id, creater_id, status]
    );
    return result.insertId;
}

async function getSalesActivitiesByDate(date, userId) {
    try {
        // Get user's role and access scope
        const [userRole] = await db.execute(
            `SELECT r.access_scope, r.id as role_id 
             FROM users u 
             JOIN roles r ON u.roles_id = r.id 
             WHERE u.id = ?`,
            [userId]
        );

        if (!userRole || !userRole.length) {
            throw new Error('User role not found');
        }

        const accessScope = JSON.parse(userRole[0].access_scope);

        const currentDate = new Date(date);
        const lastDate = new Date(currentDate);
        lastDate.setDate(lastDate.getDate() - 1);
        const last7Date = new Date(currentDate);
        last7Date.setDate(last7Date.getDate() - 7);

        const formatDate = (date) => date.toISOString().split('T')[0];

        const [records] = await db.execute('SELECT * FROM salesrecords WHERE menu_title = "Records" AND active = 1');
        const [salesactivities] = await db.execute('SELECT * FROM salesrecords WHERE menu_title = "Sales Activities" AND active = 1');
        const [customactivities] = await db.execute('SELECT * FROM salesactivities WHERE custom_activity = 1 AND active = 1');
        
        // Get overdue tasks
        const [overduetasks] = await db.execute(
            'SELECT t.*, COALESCE(a.name, c.first_name) as related_name FROM tasks t ' +
            'LEFT JOIN accounts a ON t.related_type = "accounts" AND t.related_to = a.id ' +
            'LEFT JOIN contacts c ON t.related_type = "contacts" AND t.related_to = c.id ' +
            'WHERE t.user_id = ? AND t.mark_as_complete = "false" AND t.due_date < ?',
            [userId, formatDate(currentDate)]
        );

        // Format overdue tasks
        const overdue = overduetasks.map(task => ({
            id: task.id,
            collaboratoes: task.collaboratoes,
            completed_date: task.completed_date,
            completed_time: task.completed_time,
            description: task.description,
            due_date: task.due_date,
            due_time: task.due_time,
            mark_as_complete: task.mark_as_complete,
            owner: task.owner,
            related_to: task.related_name,
            related_type: task.related_type,
            task_type: task.task_type,
            title: task.title,
            mobile_number: task.mobile_number
        }));

        // Get contacts
        const [contacts] = await db.execute(
            'SELECT * FROM contacts WHERE (owner_id = ? OR creater_id = ?) AND create_at BETWEEN ? AND ?',
            [userId, userId, formatDate(lastDate), formatDate(currentDate)]
        );

        // Get recent contacts
        const [recentcontacts] = await db.execute(
            'SELECT * FROM contacts WHERE (owner_id = ? OR creater_id = ?) AND create_at BETWEEN ? AND ?',
            [userId, userId, formatDate(last7Date), formatDate(currentDate)]
        );

        // Get links
        const [links] = await db.execute('SELECT * FROM links WHERE active = 1 AND user_id = ?', [userId]);

        // Get activities
        const [activities] = await db.execute(
            'SELECT * FROM salesactivitydatas WHERE start_date = ? AND owner_id = ? AND targetable_type != "users" AND activity_title != "attendance punch"',
            [formatDate(currentDate), userId]
        );

        // Get attendance
        const [attendance] = await db.execute(
            'SELECT * FROM salesactivitydatas WHERE start_date = ? AND owner_id = ? AND salesactivities_id = 10 LIMIT 1',
            [formatDate(currentDate), userId]
        );

        // Get currency
        const [currency] = await db.execute('SELECT * FROM currencies WHERE is_default = 1 LIMIT 1');

        // Get location tracking status
        const [locationRole] = await db.execute(
            'SELECT r.location FROM roles r INNER JOIN users u ON r.id = u.roles_id WHERE u.id = ?',
            [userId]
        );

        const locationTrackingStatus = locationRole[0]?.location === 'Yes' ? 1 : 0;

        return {
            records,
            salesactivities,
            customactivities,
            overduetasks: overdue,
            contacts,
            recentcontacts,
            links,
            activities,
            currency: currency[0],
            attendance: attendance[0] || null,
            locationtracking: [],
            Locationtrackingstatus: locationTrackingStatus,
            access_scope: accessScope // Added access_scope to response
        };
    } catch (error) {
        console.error('Error in getSalesActivitiesByDate:', error);
        throw error;
    }
}

async function getconveyance(start_date = null, end_date = null) {
    let query = `SELECT * FROM salesactivitydatas WHERE salesactivities_id IN (3, 10)`;
    const params = [];

    // Add date filtering if dates are provided
    if (start_date && end_date) {
        query += ` AND DATE(start_date) BETWEEN ? AND ?`;
        params.push(start_date, end_date);
    }

    // Add ordering
    query += ` ORDER BY start_date DESC`;

    const [rows] = await db.execute(query, params);
    return rows;
}

module.exports = {
    createSalesActivityData,
    getSalesActivitiesByDate,
    getconveyance,
};