const { json } = require('express');
const db = require('../config/db');
const { object } = require('@google/maps/lib/internal/validate');
const NotificationModel = require('../models/NotificationModel');

async function create(Data) {
    try {
        // Convert time format if needed
        if (Data.due_time) {
            // Convert 12-hour format to 24-hour format
            const timeParts = Data.due_time.match(/(\d+):(\d+)\s*(AM|PM)/i);
            if (timeParts) {
                let hours = parseInt(timeParts[1]);
                const minutes = timeParts[2];
                const meridiem = timeParts[3].toUpperCase();

                // Convert to 24-hour format
                if (meridiem === 'PM' && hours < 12) hours += 12;
                if (meridiem === 'AM' && hours === 12) hours = 0;

                Data.due_time = `${hours.toString().padStart(2, '0')}:${minutes}:00`;
            }
        }

        // Insert the task
        const keys = Object.keys(Data);
        const values = Object.values(Data);
        const placeholders = '?' + ', ?'.repeat(values.length - 1);
        
        const [result] = await db.execute(
            `INSERT INTO tasks (${keys}) VALUES (${placeholders})`,
            values
        );
        
        // Fetch the created task
        const [rows] = await db.execute(
            'SELECT t.*, u.name as assigned_to FROM tasks t LEFT JOIN users u ON t.user_id = u.id WHERE t.id = ?',
            [result.insertId]
        );

        const task = rows[0];

        // Create notification for assigned user
        if (task.user_id) {
            try {
                await NotificationModel.create({
                    user_id: task.user_id,
                    message: `New task assigned: ${task.title}`,
                    trigger_date: task.due_date,
                    trigger_time: task.due_time,
                    read_status: 0,
                    targetable_type: 'task',
                    targetable_id: task.id
                });
            } catch (notificationError) {
                console.error('Error creating notification:', notificationError);
                // Continue execution even if notification fails
            }
        }

        // Create notifications for collaborators if any
        if (task.collaborators) {
            try {
                const collaborators = typeof task.collaborators === 'string' 
                    ? JSON.parse(task.collaborators) 
                    : task.collaborators;

                for (const collaboratorId of collaborators) {
                    await NotificationModel.create({
                        user_id: collaboratorId,
                        message: `You've been added as collaborator to task: ${task.title}`,
                        trigger_date: task.due_date,
                        trigger_time: task.due_time,
                        read_status: 0,
                        targetable_type: 'task',
                        targetable_id: task.id
                    });
                }
            } catch (collaboratorError) {
                console.error('Error creating collaborator notifications:', collaboratorError);
                // Continue execution even if collaborator notifications fail
            }
        }

        // Return the created task
        return task;
    } catch (error) {
        console.error('Error in task creation:', error);
        throw new Error('Failed to create task: ' + error.message || error.response || error);
    }
}

// function formatDate(dateStr) {
//     if (!dateStr) return null;

//     // If it's a full datetime, extract just the date
//     const dateOnly = dateStr.split('T')[0] || dateStr;

//     // Expected format from DB: 'YYYY-MM-DD'
//     const [year, month, day] = dateOnly.split('-');

//     // Return in 'dd-mm-yyyy' or whatever format you need
//     return `${day}-${month}-${year}`;
// }

// function formatTime(time) {
//     if (!time) return null;
//     return new Date(`1970-01-01T${time}`).toLocaleTimeString('en-US', {
//         hour: '2-digit',
//         minute: '2-digit',
//         hour12: true
//     });
// }

// get the list
async function findAll(id, filter = {}) {
    let query = 'SELECT * FROM tasks WHERE user_id = ? AND active=1 ORDER BY id DESC';
    let params = [id];

    if(filter.contact_id) {
        query = 'SELECT * FROM tasks WHERE related_to = ? AND user_id = ? AND related_type = "contacts" AND active=1 ORDER BY id DESC';
        params = [filter.contact_id, id];
    }

    if(filter.sales_account_id) {
        query = 'SELECT * FROM tasks WHERE related_to = ? AND user_id = ? AND related_type = "accounts" AND active=1 ORDER BY id DESC';
        params = [filter.sales_account_id, id];
    }

    if(filter.deal_id) {
        query = 'SELECT * FROM tasks WHERE related_to = ? AND user_id = ? AND related_type = "deals" AND active=1 ORDER BY id DESC';
        params = [filter.deal_id, id];
    }
    const [rows] = await db.execute(query, params);
    // console.log(rows);
    const row = rows.map(row => ({
        ...row,
        due_date: row.due_date ? formatDate(row.due_date).replaceAll('/','-').split('-').reverse().join('-') : null,
        completed_date: row.completed_date ? formatDate(row.completed_date).replaceAll('/','-').split('-').reverse().join('-') : null
    }));
    console.log(rows.due_date);
    // const row = rows[0]
    console.log("row",row[0].due_date);
    // const formattedRows = rows.map(task => {
    //     if (task.due_date) {
    //         task.due_date = formatDate(task.due_date);
    //     }
    //     if (task.completed_date) {
    //         task.completed_date = formatDate(task.completed_date);
    //     }
    //     return task;
    // });
    
    // return formattedRows;
    return row;
}




async function findFields(userId) {
    // console.log(userId)
    const connection = await db.getConnection();
    try {
        // Get user's role and access scope
        const [userRole] = await connection.execute(
            `SELECT r.access_scope, r.id as role_id, u.territory_id 
             FROM users u 
             JOIN roles r ON u.roles_id = r.id 
             WHERE u.id = ?`,
            [userId]
        );

        console.log(userRole);

        if (!userRole || !userRole.length) {
            throw new Error('User role not found');
        }

        const accessScope = JSON.parse(userRole[0].access_scope) || {};
        const territories = userRole[0].territory_id ? userRole[0].territory_id.split(',') : [];

        // Get basic data
        const [outcomes] = await db.execute('SELECT * FROM salesactivityoutcomes ORDER BY id');
        const [task_type] = await db.execute('SELECT * FROM tasktypes ORDER BY id');
        const [users] = await db.execute('SELECT * FROM users WHERE active = 1 ORDER BY id');
        
        // Prepare base queries with permission filters
        let contactQuery = 'SELECT * FROM contacts WHERE active = 1';
        let accountQuery = 'SELECT * FROM accounts WHERE active = 1';
        let dealQuery = 'SELECT * FROM deals WHERE active = 1';
        
        const contactParams = [];
        const accountParams = [];
        const dealParams = [];

        // Apply permissions for contacts
        if (accessScope?.contact?.view === 'owned') {
            contactQuery += ' AND owner_id = ?';
            contactParams.push(userId);
        } else if (accessScope?.contact?.view === 'territory' && territories.length) {
            const placeholders = territories.map(() => '?').join(',');
            contactQuery += ` AND territory_id IN (${placeholders})`;
            contactParams.push(...territories);
        } else if (accessScope?.contact?.view === 'restricted') {
            contactQuery += ' AND 1=0'; // No results
        }

        // Apply permissions for accounts
        if (accessScope?.sales_account?.view === 'owned') {
            accountQuery += ' AND owner_id = ?';
            accountParams.push(userId);
        } else if (accessScope?.sales_account?.view === 'territory' && territories.length) {
            const placeholders = territories.map(() => '?').join(',');
            accountQuery += ` AND territory_id IN (${placeholders})`;
            accountParams.push(...territories);
        } else if (accessScope?.sales_account?.view === 'restricted') {
            accountQuery += ' AND 1=0'; // No results
        }

        // Apply permissions for deals
        if (accessScope?.deal?.view === 'owned') {
            dealQuery += ' AND owner_id = ?';
            dealParams.push(userId);
        } else if (accessScope?.deal?.view === 'territory' && territories.length) {
            const placeholders = territories.map(() => '?').join(',');
            dealQuery += ` AND territory_id IN (${placeholders})`;
            dealParams.push(...territories);
        } else if (accessScope?.deal?.view === 'restricted') {
            dealQuery += ' AND 1=0'; // No results
        }

        contactQuery += ' ORDER BY id';
        accountQuery += ' ORDER BY id';
        dealQuery += ' ORDER BY id';

        // Execute filtered queries
        const [contacts] = await db.execute(contactQuery, contactParams);
        const [accounts] = await db.execute(accountQuery, accountParams);
        const [deals] = await db.execute(dealQuery, dealParams);

        // Format the response
        const owners = users.map(user => ({
            id: user.id,
            name: user.name,
            email: user.email,
            mobile: user.mobile
        }));

        const related_to = {
            contacts: contacts.map(contact => ({
                id: contact.id,
                name: contact.first_name + ' ' + contact.last_name,
                email: contact.emails,
                mobile: contact.mobile_number
            })),
            accounts: accounts.map(account => ({
                id: account.id,
                name: account.name,
                mobile: account.phone
            })),
            deals: deals.map(deal => ({
                id: deal.id,
                name: deal.name
            }))
        };

        return { task_type, owners, related_to };

    } catch (error) {
        console.error('Error in findFields:', error);
        throw error;
    } finally {
        connection.release();
    }
}

async function findById(id) {  
    const [rows] = await db.execute('SELECT * FROM tasks WHERE id = ? ORDER BY id' , [id]);
    const row = rows[0];
    if (rows[0].due_date){
        rows[0].due_date = formatDate(rows[0].due_date).replaceAll('/','-').split('-').reverse().join('-');
    }
    if (rows[0].completed_date){
        rows[0].completed_date = formatDate(rows[0].completed_date).replaceAll('/','-').split('-').reverse().join('-');
    }
    console.log(rows[0].due_date);
    return rows[0];
}

async function update(id, Data) {
    // Remove undefined or empty values
    Object.keys(Data).forEach(key => {
        if (Data[key] === undefined || Data[key] === '') {
            delete Data[key];
        }
    });
    
    // If task is being marked as completed, add completion date and time
    if (Data.mark_as_complete === 'true') {
        const now = new Date();
        Data.completed_date = now.toISOString().split('T')[0]; // YYYY-MM-DD
        Data.completed_time = now.toTimeString().split(' ')[0]; // HH:MM:SS
    }
    
    // Prepare update fields and values
    const updateFields = Object.keys(Data)
        .map(key => `${key} = ?`)
        .join(', ');
    
    const updateValues = Object.values(Data);
    
    const [result] = await db.execute(
        `UPDATE tasks SET ${updateFields} WHERE id = ?`,
        [...updateValues, id]
    );
    
    return result.affectedRows > 0;
}
async function deleteRecord (id) {
    const [result] = await db.execute('UPDATE tasks SET active = 0 WHERE id = ?', [id]);
    return result.affectedRows > 0;
}
async function findByAccountId(accountId) {
    try {
        const [rows] = await db.execute(
            `SELECT t.*, 
                u.name as owner_name,
                COALESCE(t.task_type, 'General') as task_type_name
             FROM tasks t
             LEFT JOIN users u ON t.user_id = u.id
             WHERE t.related_type = "accounts" 
             AND t.related_to = ?
             AND active=0
              ORDER BY t.due_date DESC, t.due_time DESC`,
            [accountId]
        );

        return {
            status: 200,
            message: "Tasks fetched successfully",
            tasks: rows.map(task => ({
                id: task.id,
                title: task.title,
                description: task.description,
                due_date: formatDate(task.due_date),
                due_time: formatTime(task.due_time),
                status: task.mark_as_complete ? 'Completed' : 'Pending',
                completed_date: formatDate(task.completed_date),
                completed_time: formatTime(task.completed_time),
                owner_name: task.owner_name || '',
                task_type: task.task_type_name,
                outcome: task.outcome_name,
                collaborators: task.collaborators || '',
                snooze_date: formatDate(task.snooze_date),
                snooze_time: formatTime(task.snooze_time)
            }))
        };
    } catch (error) {
        console.error('Detailed error:', error);
        throw new Error(`Failed to fetch account tasks: ${error.message}`);
    }
}
async function findByDealId(dealId) {
    try {
        const [rows] = await db.execute(
            `SELECT t.*, 
                u.name as owner_name,
                COALESCE(t.task_type, 'General') as task_type_name
             FROM tasks t
             LEFT JOIN users u ON t.user_id = u.id
             WHERE t.related_type = "deals" 
             AND t.related_to = ?
             AND active=0
             ORDER BY t.due_date DESC, t.due_time DESC`,
            [dealId]
        );

        return {
            status: 200,
            message: "Tasks fetched successfully",
            tasks: rows.map(task => ({
                id: task.id,
                title: task.title,
                description: task.description,
                due_date: formatDate(task.due_date),
                due_time: formatTime(task.due_time),
                status: task.mark_as_complete ? 'Completed' : 'Pending',
                completed_date: formatDate(task.completed_date),
                completed_time: formatTime(task.completed_time),
                owner_name: task.owner_name || '',
                task_type: task.task_type_name,
                outcome: task.outcome_name,
                collaborators: task.collaborators || '',
                snooze_date: formatDate(task.snooze_date),
                snooze_time: formatTime(task.snooze_time)
            }))
        };
    } catch (error) {
        console.error('Detailed error:', error);
        throw new Error(`Failed to fetch deal tasks: ${error.message}`);
    }
}
// Helper functions for date/time formatting
function formatDate(date) {
    if (!date) return null;
    return new Date(date).toLocaleDateString('en-GB', {
        day: '2-digit',
        month: '2-digit',
        year: 'numeric'
    });
}

function formatTime(time) {
    if (!time) return null;
    return new Date(`1970-01-01T${time}`).toLocaleTimeString('en-US', {
        hour: '2-digit',
        minute: '2-digit',
        hour12: true
    });
}

// get all tasks
async function getalltasks(start_date = null, end_date = null, status = null, role_id = null, user_id = null, user = null) {
    console.log("getalltasks");
    
    let query = `
        SELECT t.*, u.name AS user_name, u2.name AS collaborator_name
        FROM tasks t
        LEFT JOIN users u ON t.user_id = u.id
        LEFT JOIN users u2 ON t.collaboratoes = u2.id
        WHERE t.active = 1`;

    const params = [];

    // Add role-based filtering
    if (role_id === 2 && user_id) {
        // For role_id = 2 (Sales Executive), filter by territory
        query += ` AND u.territory_id = (SELECT territory_id FROM users WHERE id = ?)`;
        params.push(user_id);
    }
    // For role_id = 1 (Admin), no additional filtering needed - shows all tasks

    // Add user filtering if user is provided
    if (user) {
        query += ` AND t.user_id = ?`;
        params.push(user);
    }

    // Add date filtering if dates are provided
    if (start_date && end_date) {
        query += ` AND DATE(t.due_date) BETWEEN ? AND ?`;
        params.push(start_date, end_date);
    }

    // Add status filtering if status is provided
    if (status) {
        if (status === 'Completed') {
            query += ` AND t.mark_as_complete = 'true'`;
        } else if (status === 'Pending') {
            query += ` AND (t.mark_as_complete = 'false' OR t.mark_as_complete IS NULL)`;
        }
    }

    query += ` ORDER BY t.id DESC`;

    const [rows] = await db.execute(query, params);
    console.log({rows});
    if (!rows || rows.length === 0) {
        return [];
    }
    if (rows[0].due_date){
        rows[0].due_date = formatDate(rows[0].due_date).replaceAll('/','-').split('-').reverse().join('-');
    }
    if (rows[0].completed_date){
        rows[0].completed_date = formatDate(rows[0].completed_date).replaceAll('/','-').split('-').reverse().join('-');
        // console.log("Completed date: ",rows[0].completed_date);
    }
    return rows;
}

// get task by id
async function gettasksbyid(id) {
    try {
    const [rows] = await db.execute(`
        SELECT t.*, u.name AS user_name, u2.name AS collaborator_name, tt.task_type AS task_type_name
        FROM tasks t
        LEFT JOIN users u ON t.user_id = u.id
        LEFT JOIN users u2 ON t.collaboratoes = u2.id
        LEFT JOIN tasktypes tt ON t.task_type = tt.id
        WHERE t.id = ?`, [id]);
        return rows[0];
    } catch (error) {
        console.error('Error in gettaskbyid:', error);
        throw new Error('Error fetching task by id');
    }
}

// bulk import
async function bulkImport(csvData, columnMappings, userId) {
    const connection = await db.getConnection();
    try {
        await connection.beginTransaction();
        console.log("bulkImport tasks init: " + JSON.stringify(csvData));
        console.log("Column mappings:", JSON.stringify(columnMappings, null, 2));

        const results = {
            total: csvData.length,
            successful: 0,
            failed: 0,
            errors: [],
            errorCsvData: [],
            errorCsvFile: null
        };

        // Pre-fetch validation data
        const [users] = await connection.execute('SELECT id, record_id FROM users WHERE active = 1 AND record_id IS NOT NULL');
        const [contacts] = await connection.execute('SELECT id, record_id FROM contacts WHERE active = 1 AND record_id IS NOT NULL');
        const [accounts] = await connection.execute('SELECT id, record_id FROM accounts WHERE active = 1 AND record_id IS NOT NULL');
        const [deals] = await connection.execute('SELECT id, record_id FROM deals WHERE active = 1 AND record_id IS NOT NULL');
        
        console.log('Available users:', users);
        console.log('Available contacts:', contacts);
        console.log('Available accounts:', accounts);
        console.log('Available deals:', deals);
        
        // Create lookup maps
        const userMap = new Map();
        users.forEach(user => {
            if (user.record_id) {
                userMap.set(user.record_id.toString(), user.id);
            }
        });
        
        const contactMap = new Map();
        contacts.forEach(contact => {
            if (contact.record_id) {
                contactMap.set(contact.record_id.toString(), contact.id);
            }
        });
        
        const accountMap = new Map();
        accounts.forEach(account => {
            if (account.record_id) {
                accountMap.set(account.record_id.toString(), account.id);
            }
        });
        
        const dealMap = new Map();
        deals.forEach(deal => {
            if (deal.record_id) {
                dealMap.set(deal.record_id.toString(), deal.id);
            }
        });

        for (let i = 0; i < csvData.length; i++) {
            const row = csvData[i];
            const rowNumber = i + 2; // +2 because CSV has header and arrays are 0-indexed
            
            console.log(`Processing row ${rowNumber}:`, row);
            console.log(`Available columns in CSV:`, Object.keys(row));
            console.log(`Task Owner.id value:`, row['Task Owner.id'] || 'NOT FOUND');
            console.log(`All CSV values:`, row);

            try {
                // Prepare task data
                const taskData = {
                    active: 1,
                    user_id: userId // Set default user_id to current user
                };

                // Process each mapped column
                for (const mapping of columnMappings) {
                    if (mapping.isMapped && mapping.apiField) {
                        const csvValue = row[mapping.csvColumn];
                        const fieldName = mapping.apiField;

                        console.log(`Processing field: ${fieldName}, CSV column: ${mapping.csvColumn}, CSV value: ${csvValue}`);

                        // Skip if no value provided
                        if (csvValue === undefined || csvValue === null || csvValue === '') {
                            console.log(`Skipping ${fieldName} - no value provided`);
                            continue;
                        }

                        let processedValue = csvValue.toString().trim();

                        // Handle different field types based on field_name
                        switch (fieldName) {
                            case 'user_id':
                                // Task Owner - get user ID from users table based on record_id
                                const userRecordId = csvValue.toString().trim();
                                console.log(`Looking for user with record_id: ${userRecordId}`);
                                console.log(`Available user record_ids:`, Array.from(userMap.keys()));
                                const mappedUserId = userMap.get(userRecordId);
                                if (!mappedUserId) {
                                    console.log(`User not found for record_id: ${userRecordId}, using current user_id: ${userId}`);
                                    processedValue = userId; // Use current user as fallback
                                } else {
                                    console.log(`Found user_id: ${mappedUserId} for record_id: ${userRecordId}`);
                                    processedValue = mappedUserId;
                                }
                                break;

                            case 'title':
                                // Subject - add directly to title field
                                processedValue = csvValue.toString().trim();
                                break;

                            case 'due_date':
                                // Due Date - handle various date formats
                                if (csvValue) {
                                    const date = new Date(csvValue);
                                    if (!isNaN(date.getTime())) {
                                        processedValue = date.toISOString().split('T')[0];
                                    } else {
                                        throw new Error(`Invalid due date format: "${csvValue}"`);
                                    }
                                }
                                break;

                            case 'contact_id':
                                // Contact Name.id - match with record_id in contacts table
                                const contactRecordId = csvValue.toString().trim();
                                const mappedContactId = contactMap.get(contactRecordId);
                                if (!mappedContactId) {
                                    throw new Error(`Invalid contact: "${contactRecordId}" - Contact not found`);
                                }
                                processedValue = mappedContactId;
                                break;

                            case 'related_to':
                                // Related To.id - match with respective table (accounts, contacts, deals)
                                const relatedRecordId = csvValue.toString().trim();
                                let mappedRelatedId = null;
                                let relatedType = null;

                                // Try to find in accounts first
                                mappedRelatedId = accountMap.get(relatedRecordId);
                                if (mappedRelatedId) {
                                    relatedType = 'accounts';
                                } else {
                                    // Try contacts
                                    mappedRelatedId = contactMap.get(relatedRecordId);
                                    if (mappedRelatedId) {
                                        relatedType = 'contacts';
                                    } else {
                                        // Try deals
                                        mappedRelatedId = dealMap.get(relatedRecordId);
                                        if (mappedRelatedId) {
                                            relatedType = 'deals';
                                        } else {
                                            throw new Error(`Invalid related record: "${relatedRecordId}" - Record not found in accounts, contacts, or deals`);
                                        }
                                    }
                                }
                                processedValue = mappedRelatedId;
                                // Store related_type separately
                                taskData.related_type = relatedType;
                                break;

                            case 'mark_as_complete':
                                // Status - pending = false, completed = true
                                const status = csvValue.toString().toLowerCase().trim();
                                if (status === 'pending') {
                                    processedValue = 'false';
                                } else if (status === 'completed') {
                                    processedValue = 'true';
                                } else {
                                    throw new Error(`Invalid status: "${csvValue}" - Must be 'pending' or 'completed'`);
                                }
                                break;

                            case 'created_by':
                                // Created By.id - match with users using record_id
                                const createdByRecordId = csvValue.toString().trim();
                                const mappedCreatedById = userMap.get(createdByRecordId);
                                if (!mappedCreatedById) {
                                    throw new Error(`Invalid created by: "${createdByRecordId}" - User not found`);
                                }
                                processedValue = mappedCreatedById;
                                break;

                            case 'updated_by':
                                // Modified By.id - match with users using record_id
                                const modifiedByRecordId = csvValue.toString().trim();
                                const mappedModifiedById = userMap.get(modifiedByRecordId);
                                if (!mappedModifiedById) {
                                    throw new Error(`Invalid modified by: "${modifiedByRecordId}" - User not found`);
                                }
                                processedValue = mappedModifiedById;
                                break;

                            case 'created_at':
                                // Created Time - handle datetime format
                                if (csvValue) {
                                    const date = new Date(csvValue);
                                    if (!isNaN(date.getTime())) {
                                        processedValue = date.toISOString().slice(0, 19).replace('T', ' ');
                                    } else {
                                        throw new Error(`Invalid created time format: "${csvValue}"`);
                                    }
                                }
                                break;

                            case 'updated_at':
                                // Modified Time - handle datetime format
                                if (csvValue) {
                                    const date = new Date(csvValue);
                                    if (!isNaN(date.getTime())) {
                                        processedValue = date.toISOString().slice(0, 19).replace('T', ' ');
                                    } else {
                                        throw new Error(`Invalid modified time format: "${csvValue}"`);
                                    }
                                }
                                break;

                            case 'description':
                                // Description - add directly
                                processedValue = csvValue.toString().trim();
                                break;

                            case 'completed_date':
                                // Closed Time - extract date part
                                if (csvValue) {
                                    const date = new Date(csvValue);
                                    if (!isNaN(date.getTime())) {
                                        processedValue = date.toISOString().split('T')[0];
                                    } else {
                                        throw new Error(`Invalid closed time format: "${csvValue}"`);
                                    }
                                }
                                break;

                            case 'completed_time':
                                // Closed Time - extract time part
                                if (csvValue) {
                                    const date = new Date(csvValue);
                                    if (!isNaN(date.getTime())) {
                                        processedValue = date.toTimeString().split(' ')[0];
                                    } else {
                                        throw new Error(`Invalid closed time format: "${csvValue}"`);
                                    }
                                }
                                break;

                            case 'priority':
                                // Priority - store as is
                                processedValue = csvValue.toString().trim();
                                break;

                            case 'record_id':
                                // Record Id - store as is
                                processedValue = csvValue.toString().trim();
                                break;

                            default:
                                processedValue = csvValue.toString().trim();
                        }

                        // Add to task data
                        taskData[fieldName] = processedValue;
                    }
                }

                // Insert the task
                const keys = Object.keys(taskData);
                const values = Object.values(taskData);
                const placeholders = keys.map(() => '?').join(', ');

                console.log('Task data to insert:', taskData);
                console.log('SQL Query:', `INSERT INTO tasks (${keys.join(', ')}) VALUES (${placeholders})`);
                console.log('Values:', values);

                const [result] = await connection.execute(
                    `INSERT INTO tasks (${keys.join(', ')}) VALUES (${placeholders})`,
                    values
                );

                results.successful++;

            } catch (error) {
                results.failed++;
                const errorRecord = {
                    row: rowNumber,
                    error: error.message,
                    data: row
                };
                results.errors.push(errorRecord);
                
                // Add error record to CSV data
                const errorCsvRow = {
                    ...row,
                    'Error Row': rowNumber,
                    'Error Message': error.message
                };
                results.errorCsvData.push(errorCsvRow);
            }
        }

        await connection.commit();

        // Generate error CSV file if there are errors
        if (results.errorCsvData.length > 0) {
            const fs = require('fs');
            const path = require('path');
            
            // Create filename with timestamp
            const timestamp = new Date().toISOString().replace(/[:.]/g, '-').slice(0, 19);
            const filename = `tasks_bulk_import_errors_${timestamp}.csv`;
            const filepath = path.join(__dirname, '../public/uploads', filename);
            
            // Ensure uploads directory exists
            const uploadsDir = path.dirname(filepath);
            if (!fs.existsSync(uploadsDir)) {
                fs.mkdirSync(uploadsDir, { recursive: true });
            }
            
            // Get CSV headers from the original CSV data (first row) plus error columns
            const originalHeaders = Object.keys(csvData[0] || {});
            const errorHeaders = ['Error Row', 'Error Message'];
            const headers = [...originalHeaders, ...errorHeaders];
            
            // Create CSV content
            let csvContent = headers.join(',') + '\n';
            
            results.errorCsvData.forEach(row => {
                const values = headers.map(header => {
                    let value = '';
                    if (header === 'Error Row') {
                        value = row['Error Row'] || '';
                    } else if (header === 'Error Message') {
                        value = row['Error Message'] || '';
                    } else {
                        value = row[header] || '';
                    }
                    // Escape quotes and wrap in quotes if contains comma or quote
                    const escapedValue = value.toString().replace(/"/g, '""');
                    return `"${escapedValue}"`;
                });
                csvContent += values.join(',') + '\n';
            });
            
            // Write CSV file
            fs.writeFileSync(filepath, csvContent);
            
            // Set the file path for response
            results.errorCsvFile = `/uploads/${filename}`;
        }

        return results;

    } catch (error) {
        await connection.rollback();
        throw error;
    } finally {
        connection.release();
    }
}
module.exports = {
    create,
    findAll,
    findFields,
    findById,
    update,
    deleteRecord,
    findByAccountId,
    findByDealId,
    getalltasks,
    gettasksbyid,
    bulkImport
};