const db = require('../config/db');

class TerritoryModel {
    static async createTerritory(data) {
        const connection = await db.getConnection();
        try {
            await connection.beginTransaction();

            // Clean and prepare data
            const territoryData = {
                territory: data.territory,
                active: data.active || 1,
                create_date: new Date(),
                create_by: data.create_by,
                created_at: new Date(),
                updated_at: new Date(),
                territory_id: data.territory_id || null
            };

            const [result] = await connection.execute(
                `INSERT INTO territories 
                (territory, active, create_date, create_by, created_at, updated_at, territory_id)
                VALUES (?, ?, ?, ?, ?, ?, ?)`,
                [
                    territoryData.territory,
                    territoryData.active,
                    territoryData.create_date,
                    territoryData.create_by,
                    territoryData.created_at,
                    territoryData.updated_at,
                    territoryData.territory_id
                ]
            );

            await connection.commit();
            return result.insertId;

        } catch (error) {
            await connection.rollback();
            throw error;
        } finally {
            connection.release();
        }
    }

    static async updateTerritory(id, data) {
        const connection = await db.getConnection();
        try {
            await connection.beginTransaction();

            const updateData = {
                ...data,
                updated_at: new Date()
            };

            delete updateData.id; // Remove id from update data

            const keys = Object.keys(updateData);
            const values = Object.values(updateData);

            const [result] = await connection.execute(
                `UPDATE territories SET ${keys.map(key => `${key} = ?`).join(', ')} WHERE id = ?`,
                [...values, id]
            );

            await connection.commit();
            return result.affectedRows > 0;

        } catch (error) {
            await connection.rollback();
            throw error;
        } finally {
            connection.release();
        }
    }

    static async listTerritories() {
        const [rows] = await db.execute(
            'SELECT * FROM territories WHERE active = 1 ORDER BY created_at DESC'
        );
        return rows;
    }

    static async softDeleteTerritory(id) {
        const connection = await db.getConnection();
        try {
            await connection.beginTransaction();

            const [result] = await connection.execute(
                'UPDATE territories SET active = 0, updated_at = ? WHERE id = ?',
                [new Date(), id]
            );

            await connection.commit();
            return result.affectedRows > 0;

        } catch (error) {
            await connection.rollback();
            throw error;
        } finally {
            connection.release();
        }
    }
}

module.exports = TerritoryModel;