const db = require('../config/db');
const bcrypt = require('bcryptjs');

async function createUser(userData) {
    const { username, password, email, full_name, role } = userData;
    const hashedPassword = await bcrypt.hash(password, 10);
    
    const [result] = await db.execute(
        'INSERT INTO users (username, password, email, full_name, role) VALUES (?, ?, ?, ?, ?)',
        [username, hashedPassword, email, full_name, role]
    );
    
    return result.insertId;
}

async function findUserByUsername(email) {
    try {
        // First get user and territory IDs
        const [rows] = await db.execute(
            `SELECT 
                u.id,
                u.name,
                u.email,
                u.mobile,
                u.roles_id,
                u.password,
                u.token,
                u.territory_id,
                r.access_scope 
            FROM users u 
            JOIN roles r ON (u.roles_id=r.id) 
            WHERE u.email = ? 
            AND u.active = 1`,
            [email]
        );

        if (rows.length > 0) {
            // Parse access_scope
            if (rows[0].access_scope && typeof rows[0].access_scope === 'string') {
                try {
                    rows[0].access_scope = JSON.parse(rows[0].access_scope);
                } catch (error) {
                    console.error('Error parsing access_scope:', error);
                    rows[0].access_scope = {};
                }
            }

            // Get territories if territory_id exists
            if (rows[0].territory_id) {
                try {
                    // Parse territory IDs
                    const territoryIds = rows[0].territory_id.split(',').map(id => id.trim());
                    
                    // Create placeholders for IN clause
                    const placeholders = territoryIds.map(() => '?').join(',');
                    
                    // Get territory details
                    const [territories] = await db.execute(
                        `SELECT id, territory 
                         FROM territories 
                         WHERE id IN (${placeholders})`,
                        territoryIds // Pass array directly
                    );

                    rows[0].territories = territories;
                } catch (error) {
                    console.error('Error fetching territories:', error);
                    rows[0].territories = [];
                }
            } else {
                rows[0].territories = [];
            }
        }

        return rows[0];
    } catch (error) {
        console.error('Error in findUserByUsername:', error);
        throw error;
    }
}
async function findUserById(id) {
    const [rows] = await db.execute('SELECT * FROM users WHERE id = ?', [id]);
    return rows[0];
}

async function updatetoken(userId, fcm_token) {
    const [rows] = await db.execute('UPDATE users SET token = ? WHERE id = ?', [fcm_token, userId]);
    return rows[0];
}

async function validateUserPassword(password, hashedPassword) {
    return await bcrypt.compare(password, hashedPassword);
}

async function getActiveUsers() {
    const [rows] = await db.execute('SELECT * FROM users WHERE active = 1');
    return rows;
}

module.exports = {
    createUser,
    findUserByUsername,
    updatetoken,
    findUserById,
    validateUserPassword,
    getActiveUsers
};