import { z } from "zod";
import { CallbackManagerForToolRun, type Callbacks } from "../callbacks/manager.js";
import type { BaseLangChainParams, ToolDefinition } from "../language_models/base.js";
import type { RunnableConfig } from "../runnables/config.js";
import { RunnableToolLike, type RunnableInterface } from "../runnables/base.js";
import type { ToolCall } from "../messages/tool.js";
import type { MessageContent } from "../messages/base.js";
import { JSONSchema } from "../utils/json_schema.js";
export type ResponseFormat = "content" | "content_and_artifact" | string;
export type ToolReturnType = any;
export type ContentAndArtifact = [MessageContent, any];
export type ZodObjectAny = z.ZodObject<any, any, any, any>;
/**
 * Base type that establishes the types of input schemas that can be used for LangChain tool
 * definitions.
 */
export type ToolInputSchemaBase = ZodObjectAny | z.ZodEffects<ZodObjectAny> | JSONSchema;
/**
 * Parameters for the Tool classes.
 */
export interface ToolParams extends BaseLangChainParams {
    /**
     * The tool response format.
     *
     * If "content" then the output of the tool is interpreted as the contents of a
     * ToolMessage. If "content_and_artifact" then the output is expected to be a
     * two-tuple corresponding to the (content, artifact) of a ToolMessage.
     *
     * @default "content"
     */
    responseFormat?: ResponseFormat;
    /**
     * Whether to show full details in the thrown parsing errors.
     *
     * @default false
     */
    verboseParsingErrors?: boolean;
}
export type ToolRunnableConfig<ConfigurableFieldType extends Record<string, any> = Record<string, any>> = RunnableConfig<ConfigurableFieldType> & {
    toolCall?: ToolCall;
};
/**
 * Schema for defining tools.
 *
 * @version 0.2.19
 */
export interface StructuredToolParams extends Pick<StructuredToolInterface, "name" | "schema"> {
    /**
     * An optional description of the tool to pass to the model.
     */
    description?: string;
}
/**
 * Utility type that resolves the output type of a tool input schema.
 *
 * Input & Output types are a concept used with Zod schema, as Zod allows for transforms to occur
 * during parsing. When using JSONSchema, input and output types are the same.
 *
 * The input type for a given schema should match the structure of the arguments that the LLM
 * generates as part of its {@link ToolCall}. The output type will be the type that results from
 * applying any transforms defined in your schema. If there are no transforms, the input and output
 * types will be the same.
 */
export type ToolInputSchemaOutputType<T extends ToolInputSchemaBase> = T extends z.ZodSchema ? z.output<T> : T extends JSONSchema ? unknown : never;
/**
 * Utility type that resolves the input type of a tool input schema.
 *
 * Input & Output types are a concept used with Zod schema, as Zod allows for transforms to occur
 * during parsing. When using JSONSchema, input and output types are the same.
 *
 * The input type for a given schema should match the structure of the arguments that the LLM
 * generates as part of its {@link ToolCall}. The output type will be the type that results from
 * applying any transforms defined in your schema. If there are no transforms, the input and output
 * types will be the same.
 */
export type ToolInputSchemaInputType<T extends ToolInputSchemaBase> = T extends z.ZodSchema ? z.input<T> : T extends JSONSchema ? unknown : never;
/**
 * Defines the type that will be passed into a tool handler function as a result of a tool call.
 *
 * @param SchemaT - The type of the tool input schema. Usually you don't need to specify this.
 * @param SchemaInputT - The TypeScript type representing the structure of the tool arguments generated by the LLM. Useful for type checking tool handler functions when using JSONSchema.
 */
export type StructuredToolCallInput<SchemaT extends ToolInputSchemaBase = ZodObjectAny, SchemaInputT = ToolInputSchemaInputType<SchemaT>> = (ToolInputSchemaOutputType<SchemaT> extends string ? string : never) | SchemaInputT | ToolCall;
/**
 * Interface that defines the shape of a LangChain structured tool.
 *
 * A structured tool is a tool that uses a schema to define the structure of the arguments that the
 * LLM generates as part of its {@link ToolCall}.
 *
 * @param SchemaT - The type of the tool input schema. Usually you don't need to specify this.
 * @param SchemaInputT - The TypeScript type representing the structure of the tool arguments generated by the LLM. Useful for type checking tool handler functions when using JSONSchema.
 */
export interface StructuredToolInterface<SchemaT extends ToolInputSchemaBase = ZodObjectAny | z.ZodEffects<ZodObjectAny> | JSONSchema, SchemaInputT = ToolInputSchemaInputType<SchemaT>> extends RunnableInterface<StructuredToolCallInput<SchemaT, SchemaInputT>, ToolReturnType> {
    lc_namespace: string[];
    /**
     * A Zod schema representing the parameters of the tool.
     */
    schema: SchemaT;
    /**
     * @deprecated Use .invoke() instead. Will be removed in 0.3.0.
     *
     * Calls the tool with the provided argument, configuration, and tags. It
     * parses the input according to the schema, handles any errors, and
     * manages callbacks.
     * @param arg The input argument for the tool.
     * @param configArg Optional configuration or callbacks for the tool.
     * @param tags Optional tags for the tool.
     * @returns A Promise that resolves with a string.
     */
    call(arg: StructuredToolCallInput<SchemaT, SchemaInputT>, configArg?: Callbacks | RunnableConfig, 
    /** @deprecated */
    tags?: string[]): Promise<ToolReturnType>;
    /**
     * The name of the tool.
     */
    name: string;
    /**
     * A description of the tool.
     */
    description: string;
    /**
     * Whether to return the tool's output directly.
     *
     * Setting this to true means that after the tool is called,
     * an agent should stop looping.
     */
    returnDirect: boolean;
}
/**
 * A special interface for tools that accept a string input, usually defined with the {@link Tool} class.
 *
 * @param SchemaT - The type of the tool input schema. Usually you don't need to specify this.
 * @param SchemaInputT - The TypeScript type representing the structure of the tool arguments generated by the LLM. Useful for type checking tool handler functions when using JSONSchema.
 */
export interface ToolInterface<SchemaT extends ToolInputSchemaBase = ZodObjectAny | z.ZodEffects<ZodObjectAny> | JSONSchema, SchemaInputT = ToolInputSchemaInputType<SchemaT>> extends StructuredToolInterface<SchemaT, SchemaInputT> {
    /**
     * @deprecated Use .invoke() instead. Will be removed in 0.3.0.
     *
     * Calls the tool with the provided argument and callbacks. It handles
     * string inputs specifically.
     * @param arg The input argument for the tool, which can be a string, undefined, or an input of the tool's schema.
     * @param callbacks Optional callbacks for the tool.
     * @returns A Promise that resolves with a string.
     */
    call(arg: string | undefined | SchemaInputT | ToolCall, callbacks?: Callbacks | RunnableConfig): Promise<ToolReturnType>;
}
/**
 * Base interface for the input parameters of the {@link DynamicTool} and
 * {@link DynamicStructuredTool} classes.
 */
export interface BaseDynamicToolInput extends ToolParams {
    name: string;
    description: string;
    /**
     * Whether to return the tool's output directly.
     *
     * Setting this to true means that after the tool is called,
     * an agent should stop looping.
     */
    returnDirect?: boolean;
}
/**
 * Interface for the input parameters of the DynamicTool class.
 */
export interface DynamicToolInput extends BaseDynamicToolInput {
    func: (input: string, runManager?: CallbackManagerForToolRun, config?: ToolRunnableConfig) => Promise<ToolReturnType>;
}
/**
 * Interface for the input parameters of the DynamicStructuredTool class.
 *
 * @param SchemaT - The type of the tool input schema. Usually you don't need to specify this.
 * @param SchemaOutputT - The TypeScript type representing the result of applying the schema to the tool arguments. Useful for type checking tool handler functions when using JSONSchema.
 */
export interface DynamicStructuredToolInput<SchemaT extends ToolInputSchemaBase = ZodObjectAny, SchemaOutputT = ToolInputSchemaOutputType<SchemaT>> extends BaseDynamicToolInput {
    /**
     * Tool handler function - the function that will be called when the tool is invoked.
     *
     * @param input - The input to the tool.
     * @param runManager - The run manager for the tool.
     * @param config - The configuration for the tool.
     * @returns The result of the tool.
     */
    func: (input: BaseDynamicToolInput["responseFormat"] extends "content_and_artifact" ? ToolCall : SchemaOutputT, runManager?: CallbackManagerForToolRun, config?: RunnableConfig) => Promise<ToolReturnType>;
    schema: SchemaT extends ZodObjectAny ? SchemaT : SchemaT;
}
/**
 * Confirm whether the inputted tool is an instance of `StructuredToolInterface`.
 *
 * @param {StructuredToolInterface | JSONSchema | undefined} tool The tool to check if it is an instance of `StructuredToolInterface`.
 * @returns {tool is StructuredToolInterface} Whether the inputted tool is an instance of `StructuredToolInterface`.
 */
export declare function isStructuredTool(tool?: StructuredToolInterface | ToolDefinition | JSONSchema): tool is StructuredToolInterface;
/**
 * Confirm whether the inputted tool is an instance of `RunnableToolLike`.
 *
 * @param {unknown | undefined} tool The tool to check if it is an instance of `RunnableToolLike`.
 * @returns {tool is RunnableToolLike} Whether the inputted tool is an instance of `RunnableToolLike`.
 */
export declare function isRunnableToolLike(tool?: unknown): tool is RunnableToolLike;
/**
 * Confirm whether or not the tool contains the necessary properties to be considered a `StructuredToolParams`.
 *
 * @param {unknown | undefined} tool The object to check if it is a `StructuredToolParams`.
 * @returns {tool is StructuredToolParams} Whether the inputted object is a `StructuredToolParams`.
 */
export declare function isStructuredToolParams(tool?: unknown): tool is StructuredToolParams;
/**
 * Whether or not the tool is one of StructuredTool, RunnableTool or StructuredToolParams.
 * It returns `is StructuredToolParams` since that is the most minimal interface of the three,
 * while still containing the necessary properties to be passed to a LLM for tool calling.
 *
 * @param {unknown | undefined} tool The tool to check if it is a LangChain tool.
 * @returns {tool is StructuredToolParams} Whether the inputted tool is a LangChain tool.
 */
export declare function isLangChainTool(tool?: unknown): tool is StructuredToolParams;
