"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.pull = exports.push = void 0;
const base_js_1 = require("./base.cjs");
Object.defineProperty(exports, "push", { enumerable: true, get: function () { return base_js_1.basePush; } });
const index_js_1 = require("../load/index.cjs");
/**
 * Pull a prompt from the hub.
 * @param ownerRepoCommit The name of the repo containing the prompt, as well as an optional commit hash separated by a slash.
 * @param options.apiKey LangSmith API key to use when pulling the prompt
 * @param options.apiUrl LangSmith API URL to use when pulling the prompt
 * @param options.includeModel Whether to also instantiate and attach a model instance to the prompt,
 *   if the prompt has associated model metadata. If set to true, invoking the resulting pulled prompt will
 *   also invoke the instantiated model. You must have the appropriate LangChain integration package installed.
 * @returns
 */
async function pull(ownerRepoCommit, options) {
    const promptObject = await (0, base_js_1.basePull)(ownerRepoCommit, options);
    let modelClass;
    if (options?.includeModel) {
        if (Array.isArray(promptObject.manifest.kwargs?.last?.kwargs?.bound?.id)) {
            const modelName = promptObject.manifest.kwargs?.last?.kwargs?.bound?.id.at(-1);
            if (modelName === "ChatOpenAI") {
                modelClass = (await import("@langchain/openai")).ChatOpenAI;
            }
            else if (modelName === "ChatAnthropic") {
                modelClass = (await import("@langchain/anthropic")).ChatAnthropic;
            }
            else if (modelName === "ChatAzureOpenAI") {
                modelClass = (await import("@langchain/openai")).AzureChatOpenAI;
            }
            else if (modelName === "ChatGoogleVertexAI") {
                modelClass = (await import("@langchain/google-vertexai")).ChatVertexAI;
            }
            else if (modelName === "ChatGoogleGenerativeAI") {
                modelClass = (await import("@langchain/google-genai"))
                    .ChatGoogleGenerativeAI;
            }
            else if (modelName === "ChatBedrockConverse") {
                modelClass = (await import("@langchain/aws")).ChatBedrockConverse;
            }
            else if (modelName === "ChatMistral") {
                modelClass = (await import("@langchain/mistralai")).ChatMistralAI;
            }
            else if (modelName === "ChatGroq") {
                modelClass = (await import("@langchain/groq")).ChatGroq;
            }
            else if (modelName !== undefined) {
                console.warn(`Received unknown model name from prompt hub: "${modelName}"`);
            }
        }
    }
    const loadedPrompt = await (0, index_js_1.load)(JSON.stringify(promptObject.manifest), undefined, (0, base_js_1.generateOptionalImportMap)(modelClass), (0, base_js_1.generateModelImportMap)(modelClass));
    return loadedPrompt;
}
exports.pull = pull;
