const express = require('express');
const router = express.Router();
const conveyanceController = require('../controllers/conveyanceController');
const auth = require('../middleware/auth');
const multer = require('multer');
const path = require('path');
const fs = require('fs');

// Configure multer for file upload
const storage = multer.diskStorage({
    destination: (req, file, cb) => {
        const dir = path.join(process.cwd(), 'public', 'uploads', 'claims');
        if (!fs.existsSync(dir)) {
            fs.mkdirSync(dir, { recursive: true });
        }
        cb(null, dir);
    },
    filename: (req, file, cb) => {
        const uniqueSuffix = Date.now() + '-' + Math.round(Math.random() * 1E9);
        cb(null, 'claim-' + uniqueSuffix + path.extname(file.originalname));
    }
});

// File filter function
const fileFilter = (req, file, cb) => {
    // Accept images and PDFs only
    if (!file.originalname.match(/\.(jpg|jpeg|png|pdf)$/)) {
        return cb(new Error('Only image and PDF files are allowed!'), false);
    }
    cb(null, true);
};

// Create multer instances for different upload scenarios
const uploadSingle = multer({ 
    storage: storage,
    fileFilter: fileFilter,
    limits: {
        fileSize: 10 * 1024 * 1024 // 10MB limit
    }
});

const uploadMultiple = multer({ 
    storage: storage,
    fileFilter: fileFilter,
    limits: {
        fileSize: 10 * 1024 * 1024, // 10MB limit per file
        files: 5 // Maximum 5 files
    }
});

// Flexible upload middleware that can handle both single and multiple files
const flexibleUpload = multer({
    storage: storage,
    fileFilter: fileFilter,
    limits: {
        fileSize: 10 * 1024 * 1024, // 10MB limit per file
        files: 5 // Maximum 5 files
    }
}).any(); // Accept any field name

// Custom middleware to handle flexible uploads
const handleFlexibleUpload = (req, res, next) => {
    flexibleUpload(req, res, (err) => {
        if (err) {
            return handleMulterError(err, req, res, next);
        }
        
        // Process uploaded files and organize them
        if (req.files && req.files.length > 0) {
            // Group files by field name
            const filesByField = {};
            req.files.forEach(file => {
                if (!filesByField[file.fieldname]) {
                    filesByField[file.fieldname] = [];
                }
                filesByField[file.fieldname].push(file);
            });
            
            // Set appropriate properties based on field names
            if (filesByField['claim_image']) {
                req.files = filesByField['claim_image']; // Multiple files for claim_image
            }
            if (filesByField['claim_images']) {
                req.files = filesByField['claim_images']; // Multiple files for claim_images
            }
            
            // If no specific field found, use all files
            if (!req.files && req.files && req.files.length > 0) {
                req.files = req.files;
            }
        }
        
        next();
    });
};

// Enhanced error handling middleware for multer
const handleMulterError = (err, req, res, next) => {
    if (err instanceof multer.MulterError) {
        switch (err.code) {
            case 'LIMIT_FILE_SIZE':
                return res.status(400).json({
                    status: 400,
                    success: false,
                    message: 'File size should be less than 10MB'
                });
            case 'LIMIT_FILE_COUNT':
                return res.status(400).json({
                    status: 400,
                    success: false,
                    message: 'Maximum 5 files can be uploaded'
                });
            case 'LIMIT_UNEXPECTED_FILE':
                return res.status(400).json({
                    status: 400,
                    success: false,
                    message: `Unexpected field: ${err.field}. Please check the field name for file upload.`
                });
            default:
                return res.status(400).json({
                    status: 400,
                    success: false,
                    message: `File upload error: ${err.message}`
                });
        }
    }
    
    // Handle other file-related errors
    if (err.message && err.message.includes('Only image and PDF files are allowed')) {
        return res.status(400).json({
            status: 400,
            success: false,
            message: err.message
        });
    }
    
    next(err);
};

// All routes require authentication
router.use(auth);

// Reapply claim route with flexible file upload support
router.put('/:id/reapply', auth, handleFlexibleUpload, conveyanceController.reapplyClaim);

// Get conveyance data
router.post('/get', conveyanceController.getConveyance);

// View detailed conveyance for a specific date
router.post('/view', conveyanceController.viewConveyance);

// Apply the Claim 
router.post('/claimapply', conveyanceController.claimapply);

// Claim history
router.get('/claimhistory', conveyanceController.claimhistory);

// Claim history by status
router.get('/history/status', conveyanceController.claimhistoryByStatus);

// Edit claim route with flexible file upload support
router.put('/:id/edit', auth, handleFlexibleUpload, conveyanceController.editClaim);

module.exports = router;