const express = require('express');
const router = express.Router();
const productController = require('../controllers/productController');
const auth = require('../middleware/auth');
const multer = require('multer');
const path = require('path');

// Configure multer storage
const storage = multer.diskStorage({
    destination: function (req, file, cb) {
        cb(null, 'tmp/');
    },
    filename: function (req, file, cb) {
        cb(null, Date.now() + path.extname(file.originalname));
    }
});

// Create multer upload instance
const upload = multer({ 
    storage: storage,
    limits: {
        fileSize: 10 * 1024 * 1024 // 10MB limit
    }
});

// All routes require authentication
router.use(auth);
router.get('/filters', productController.getAllFiltersList);
router.post('/filtersViewUpdate', productController.updateFilterView);
router.delete('/filters/:id', productController.deleteFilterView);
// Get product fields
router.post('/fields', auth, productController.getProductFields);

// Create new customer
router.post('/', productController.create);

// Get all customers
router.get('/',productController.getAll);

// Get customer by ID
router.get('/:id', productController.getById);

// Update customer
router.put('/:id', productController.update);

// Delete customer
router.delete('/:id', productController.remove);

// Filter products //productListWithFilters
router.post('/productListWithFilters', auth, productController.filterProducts);

// Update product price
router.post('/update-price', auth, productController.updateProductPrice);

// Delete product price
router.post('/delete-price', auth, productController.deletePrice);


// Add product view
router.post('/views', auth, productController.addProductView);

// Update product view
router.put('/views/update', auth, productController.editProductView);

// Delete product view
router.delete('/views/delete', auth, productController.deleteProductView);

// Get categories with products
router.get('/categories/view', auth, productController.getCategories);

// Add deal products
router.post('/deal/products', auth, productController.addDealProducts);

// Get deal products
router.post('/deal/products/list', auth, productController.getDealProducts);

// Sync deal with quote
router.post('/deal/sync-quote', auth, productController.syncQuote);

// Remove deal sync
router.post('/deal/remove-sync', auth, productController.removeSync);

// Remove product from deal
router.post('/deal/product/remove', auth, productController.removeDealProduct);

// Upload file - add multer middleware
router.post('/file/upload', auth, upload.single('file'), productController.uploadFile);

// Delete file
router.post('/file/delete', auth, productController.deleteFile);

// Get lookup data (users and territories)
router.get('/filter/lookup-data', auth, productController.getLookupData);

module.exports = router;