const express = require('express');
const cors = require('cors');
const dotenv = require('dotenv');
const path = require('path');
const mysql = require('mysql2/promise');

// Load environment variables
dotenv.config();

const app = express();


// Middleware
app.use(cors());
app.use(express.json());
app.use(express.urlencoded({ extended: true }));
app.use('/public', express.static(path.join(__dirname, '/public')));


// Error handling middleware for JSON parsing errors
app.use((err, req, res, next) => {
    if (err instanceof SyntaxError && err.status === 400 && 'body' in err) {
        // console.error('Bad JSON');
        return res.status(400).json({ message: 'Invalid JSON payload' });
    }
    next();
});
const contactController = require('./controllers/contactController');
// Import routes
const authRoutes = require('./routes/auth');
const contactRoutes = require('./routes/contacts');
const contactFieldRoutes = require('./routes/contactFields');
const accountFieldRoutes = require('./routes/accountFields');
const salesAccountRoutes = require('./routes/salesAccounts');
const salesActivitiesRoutes = require('./routes/salesActivitiesRoutes');
const roleRoutes = require('./routes/role');
const taskRoutes = require('./routes/tasks');
const dealRoutes = require('./routes/deals');
const dealFieldRoutes = require('./routes/dealFields');
const productFieldRoutes = require('./routes/productFields');
const productRoutes = require('./routes/products');
const fileUploadRoutes = require('./routes/fileUpload');
const adminRoutes = require('./routes/admin');
const addNotesRoutes = require('./routes/addNotesRoute');
const quotationFieldRoutes = require('./routes/quotationFields');
const quotationRoutes = require('./routes/quotations');
const dashboardRoutes = require('./routes/dashboard');
const filesRouter = require('./routes/files');
const cpqRoutes = require('./routes/cpq');
const conveyanceRoutes = require('./routes/conveyance');
const notificationRoutes = require('./routes/NotificationRoutes');
const territoryRoutes = require('./routes/territory');
const salesActivityCustomFieldRoutes = require('./routes/salesActivityCustomField');
const reportRoutes = require('./routes/reports');


// Database connection pool
const pool = mysql.createPool({
    host: process.env.DB_HOST,
    user: process.env.DB_USER,
    password: process.env.DB_PASSWORD,
    database: process.env.DB_NAME,
    port: process.env.DB_PORT,
    waitForConnections: true,
    connectionLimit: 10,
    queueLimit: 0
});

// Test database connection
pool.getConnection()
    .then(connection => {
        console.log('Database connected successfully');
        connection.release();
    })
    .catch(err => {
        console.error('Error connecting to the database:', err);
    });

// Initialize attendance scheduler
const attendanceScheduler = require('./services/attendanceScheduler');
attendanceScheduler.initialize();

// Use routes
app.use('/api/auth', authRoutes);
app.use('/api/sales_accounts/fields', accountFieldRoutes);
app.use('/api/contacts/fields', contactFieldRoutes);
app.use('/api/contacts', contactRoutes);
app.use('/api/sales_accounts', salesAccountRoutes);
app.use('/api/role', roleRoutes);
app.use('/api/salesactivities', salesActivitiesRoutes);
app.use('/api/tasks', taskRoutes);
app.use('/api/deals', dealRoutes);
app.use('/api/deal/fields', dealFieldRoutes);
app.use('/api/product/fields', productFieldRoutes);
app.use('/api/products', productRoutes);
app.use('/api/files', fileUploadRoutes);
app.use('/api/admin', adminRoutes);
app.use('/api/notes', addNotesRoutes);
app.use('/api/quotation-fields', quotationFieldRoutes);
app.use('/api/quotations', quotationRoutes);
app.use('/api/dashboard', dashboardRoutes);
app.use('/api/files', filesRouter);
app.use('/api/cpq', cpqRoutes);
app.use('/api/conveyance', conveyanceRoutes);
app.use('/api/notifications', notificationRoutes);
app.use('/api/territories', territoryRoutes);
app.use('/api/salesactivitycustomfields', salesActivityCustomFieldRoutes);
app.use('/api/reports', reportRoutes);

// Basic route for testing
app.get('/', (req, res) => {
    res.json({ message: 'Welcome to Field Sales CRM API' });
});


// Error handling middleware
app.use((err, req, res, next) => {
    console.error(err.stack);
    res.status(500).json({ message: 'Something went wrong!' });
});

app.use('*',(req, res, next) => {
    res.status(404).json({ message: 'Route Not Found' });
});



// Start server
const PORT = process.env.PORT || 5000;
app.listen(PORT, () => {
    console.log(`Server is running on port ${PORT}`);
});