const db = require('../config/db');
const NotificationModel = require('../models/NotificationModel');
const { addMinutes, format } = require('date-fns');

const checkUpcomingEvents = async () => {
    try {
        const targetTime = addMinutes(new Date(), 10);
        const targetDate = format(targetTime, 'yyyy-MM-dd');
        const targetTimeStr = format(targetTime, 'HH:mm:ss');

        // Get upcoming tasks
        const [tasks] = await db.execute(
            `SELECT 
                t.*,
                u.id as user_id,
                u.name as assigned_to,
                u.token as fcm_token
            FROM tasks t
            JOIN users u ON t.user_id = u.id
            WHERE DATE(t.due_date) = ?
            AND TIME(t.due_time) = ?
            AND t.reminded = 0
            AND t.active = 1
            AND u.active = 1`,
            [targetDate, targetTimeStr]
        );

        // Get upcoming sales activities
        const [activities] = await db.execute(
            `SELECT 
                sad.*,
                u.id as user_id,
                u.name as owner_name,
                sa.activity_name,
                u.token as fcm_token
            FROM salesactivitydatas sad
            JOIN users u ON sad.owner_id = u.id
            JOIN salesactivities sa ON sad.salesactivities_id = sa.id
            WHERE DATE(sad.start_date) = ?
            AND TIME(sad.start_time) = ?
            AND sad.reminded = 0
            AND u.active = 1`,
            [targetDate, targetTimeStr]
        );

        // Process tasks notifications
        const taskNotifications = tasks.map(async (task) => {
            const notificationData = {
                user_id: task.user_id,
                message: `Reminder: Task "${task.title}" is due in 10 minutes`,
                trigger_date: task.due_date,
                trigger_time: task.due_time,
                targetable_type: 'task',
                targetable_id: task.id
            };

            await NotificationModel.create(notificationData);

            // Mark task as reminded
            await db.execute(
                'UPDATE tasks SET reminded = 1 WHERE id = ?',
                [task.id]
            );
        });

        // Process activities notifications
        const activityNotifications = activities.map(async (activity) => {
            const notificationData = {
                user_id: activity.user_id,
                message: `Reminder: ${activity.activity_name} starts in 10 minutes`,
                trigger_date: activity.start_date,
                trigger_time: activity.start_time,
                targetable_type: 'activity',
                targetable_id: activity.id
            };

            await NotificationModel.create(notificationData);

            // Mark activity as reminded
            await db.execute(
                'UPDATE salesactivitydatas SET reminded = 1 WHERE id = ?',
                [activity.id]
            );
        });

        // Wait for all notifications to be processed
        await Promise.all([...taskNotifications, ...activityNotifications]);

        return {
            tasks: tasks.length,
            activities: activities.length,
            total: tasks.length + activities.length
        };

    } catch (error) {
        console.error('Scheduler error:', error);
        throw error;
    }
};

module.exports = { checkUpcomingEvents };