const admin = require('../utils/firebaseAdmin');
const db = require('../config/db');

class NotificationService {
    async sendNotification(userId, title, body, data = {}) {
        try {
            // Get user's FCM token
            const [rows] = await db.execute(
                'SELECT token FROM users WHERE id = ? AND active = 1',
                [userId]
            );

            if (!rows.length || !rows[0].token) {
                throw new Error('User token not found');
            }

            const message = {
                notification: {
                    title,
                    body
                },
                data: {
                    ...data,
                    click_action: 'FLUTTER_NOTIFICATION_CLICK'
                },
                token: rows[0].token
            };

            const response = await admin.messaging().send(message);
            
            // Log notification
            await this.logNotification({
                user_id: userId,
                title,
                body,
                data: JSON.stringify(data),
                status: 'sent'
            });

            return {
                success: true,
                message: 'Notification sent successfully',
                messageId: response
            };

        } catch (error) {
            console.error('Error sending notification:', error);
            
            // Log failed notification
            await this.logNotification({
                user_id: userId,
                title,
                body,
                data: JSON.stringify(data),
                status: 'failed',
                error: error.message
            });

            throw new Error(`Failed to send notification: ${error.message}`);
        }
    }

    async sendBulkNotifications(userIds, title, body, data = {}) {
        try {
            // Get users' FCM tokens
            const [rows] = await db.execute(
                'SELECT id, token FROM users WHERE id IN (?) AND active = 1',
                [userIds]
            );

            const tokens = rows.filter(row => row.token).map(row => row.token);

            if (!tokens.length) {
                throw new Error('No valid tokens found');
            }

            const message = {
                notification: {
                    title,
                    body
                },
                data: {
                    ...data,
                    click_action: 'FLUTTER_NOTIFICATION_CLICK'
                },
                tokens: tokens
            };

            const response = await admin.messaging().sendMulticast(message);

            // Log notifications
            await Promise.all(rows.map(user => 
                this.logNotification({
                    user_id: user.id,
                    title,
                    body,
                    data: JSON.stringify(data),
                    status: 'sent'
                })
            ));

            return {
                success: true,
                message: 'Notifications sent successfully',
                successCount: response.successCount,
                failureCount: response.failureCount
            };

        } catch (error) {
            console.error('Error sending bulk notifications:', error);
            throw new Error(`Failed to send notifications: ${error.message}`);
        }
    }

    async logNotification(notificationData) {
        try {
            const [result] = await db.execute(
                `INSERT INTO firebaseinboxes 
                (user_id, title, body, data, status, created_at) 
                VALUES (?, ?, ?, ?, ?, CURRENT_TIMESTAMP)`,
                [
                    notificationData.user_id,
                    notificationData.title,
                    notificationData.body,
                    notificationData.data,
                    notificationData.status
                ]
            );

            return result.insertId;

        } catch (error) {
            console.error('Error logging notification:', error);
        }
    }
}

module.exports = new NotificationService();