import { findAllAttendance, findAttendanceById, viewlistAttendancePresentAbsent } from '../models/attendanceModel.js'
import { db } from "../utils/database.js";
async function getAttendance(req, res) {
    try {
        const { start_date, end_date } = req.query;
        
        // Validate date format if provided
        if (start_date && !isValidDate(start_date)) {
            return res.status(400).json({
                status: 400,
                success: false,
                message: "Invalid start_date format. Use YYYY-MM-DD"
            });
        }
        
        if (end_date && !isValidDate(end_date)) {
            return res.status(400).json({
                status: 400,
                success: false,
                message: "Invalid end_date format. Use YYYY-MM-DD"
            });
        }

        const attendance = await findAllAttendance(start_date, end_date);   
        res.status(200).json(
            {
                status: 200,
                success: true,
                message: "Attendance list fetched successfully",
                data: attendance
            }
        );
    } catch (error) {
        res.status(500).json(
            {
                status: 500,
                success: false,
                message: "Error fetching attendance",
                error: error.message
            }
        );
    }
}

// Helper function to validate date format
function isValidDate(dateString) {
    const regex = /^\d{4}-\d{2}-\d{2}$/;
    if (!regex.test(dateString)) return false;
    
    const date = new Date(dateString);
    return date instanceof Date && !isNaN(date) && date.toISOString().slice(0, 10) === dateString;
}

async function getAttendanceById(req, res) {
    const { id } = req.params;
    try {
        const attendance = await findAttendanceById(id);
        res.status(200).json(attendance);
    } catch (error) {
        res.status(500).json(
            {
                status: 500,
                success: false,
                message: "Error fetching attendance by id",
                error: error.message
            }
        );
    }
}
async function getAttendancePresentAbsent(req, res) {
    try {
        // ✅ Added user authentication check
        if (!req.user || !req.user.id) {
            return res.status(401).json({
                status: 401,
                success: false,
                message: "Unauthorized: No user info"
            });
        }

        const { start_date, end_date } = req.query;

        // Validation for date format and required parameters
        if (!start_date || !end_date) {
            return res.status(400).json({
                status: 400,
                success: false,
                message: "Both start_date and end_date are required"
            });
        }

        // Validate date format if provided
        if (start_date && !isValidDate(start_date)) {
            return res.status(400).json({
                status: 400,
                success: false,
                message: "Invalid start_date format. Use YYYY-MM-DD"
            });
        }

        if (end_date && !isValidDate(end_date)) {
            return res.status(400).json({
                status: 400,
                success: false,
                message: "Invalid end_date format. Use YYYY-MM-DD"
            });
        }

        // Fetch user details from DB to get roles_id and branch_id
        let userRows = await db.query(
            "SELECT id, roles_id, branch_id FROM users WHERE id = ?",
            [Number(req.user.id)]
        );
        let user = userRows[0];
        if (!user) {
            return res.status(404).json({
                status: 404,
                success: false,
                message: "User not found"
            });
        }
        if (!user.roles_id) {
            return res.status(400).json({
                status: 400,
                success: false,
                message: "User has no roles_id"
            });
        }

        // Get user's role and scope
        let scopeRows = await db.query(
            "SELECT scope FROM roles WHERE id = ?",
            [user.roles_id]
        );
        const scope = scopeRows[0]?.scope || '';

        // Get attendance data with role-based filtering
        const attendanceData = await viewlistAttendancePresentAbsent({
            startDate: start_date,
            endDate: end_date,
            scope: scope,
            userId: user.id,
            branchId: user.branch_id
        });

        res.status(200).json({
            status: 200,
            success: true,
            message: "Attendance present/absent list fetched successfully",
            data: {
                /*dateRange: {
                    startDate: start_date,
                    endDate: end_date
                },
                userScope: scope,
                summary: {
                    total: attendanceData.total,
                    presentCount: attendanceData.presentCount,
                    absentCount: attendanceData.absentCount
                },*/
                presentList: attendanceData.present,
                absentList: attendanceData.absent
            }
        });
    } catch (error) {
        console.error('Get attendance present/absent error:', error);
        res.status(500).json({
            status: 500,
            success: false,
            message: "Error fetching attendance present/absent list",
            error: error.message
        });
    }
}
export { 
    getAttendance,
    getAttendanceById,
    getAttendancePresentAbsent
};