import { Company } from '../models/company.js';
import { db } from '../utils/database.js';

export const createCompany = async (req, res) => {
    try {
        // Check if company_id already exists
        const exists = await Company.checkCompanyIdExists(req.body.company_id);
        if (exists) {
            return res.status(409).json({
                status: 409,
                message: 'Company ID already exists'
            });
        }
        const companyId = await Company.create(req.body, req.user.id);
        const company = await Company.findById(companyId);
        
        res.status(201).json({
            status: 201,
            message: 'Company created successfully',
            company
        });
    } catch (error) {
        console.error('Create company error:', error);
        res.status(500).json({
            status: 500,
            message: 'Failed to create company'
        });
    }
};
export const updateCompany = async (req, res) => {
    try {
        const { id } = req.params;

        // Check if company_id already exists for other companies
        const exists = await Company.checkCompanyIdExists(req.body.company_id, id);
        if (exists) {
            return res.status(409).json({
                status: 409,
                message: 'Company ID already exists'
            });
        }

        await Company.update(id, req.body);
        const company = await Company.findById(id);

        res.json({
            status: 200,
            message: 'Company updated successfully',
            company
        });
    } catch (error) {
        console.error('Update company error:', error);
        res.status(500).json({
            status: 500,
            message: 'Failed to update company'
        });
    }
};

export const deleteCompany = async (req, res) => {
    try {
        const { id } = req.params;
        await Company.softDelete(id);

        res.json({
            status: 200,
            message: 'Company deleted successfully'
        });
    } catch (error) {
        console.error('Delete company error:', error);
        res.status(500).json({
            status: 500,
            message: 'Failed to delete company'
        });
    }
};
export const listCompanies = async (req, res) => {
    try {
        const companies = await Company.list();
        
        res.json({
            status: 200,
            message: 'Companies retrieved successfully',
            companies
        });
    } catch (error) {
        console.error('List companies error:', error);
        res.status(500).json({
            status: 500,
            message: 'Failed to retrieve companies'
        });
    }
};
// export const getCompanyList = async (req, res) => {
//   try {
//     // 1. Get pagination and sorting params
//     const {
//       page = req.query.start,
//       limit = req.query.length,
//       order_by = req.query.order_by,
//       order_dir = req.query.order_dir,
//       search = req.query.search
//     } = req.query;
//     const pageNum = parseInt(page);
//     const limitNum = parseInt(limit);
//     const offset = (pageNum - 1) * limitNum;
//     const offset = parseInt(req.query.start, 10);
//     const limitNum = parseInt(req.query.length, 10);

//     // Validate offset and limit
//     const validOffset = isNaN(offset) ? 0 : offset;
//     const validLimit = isNaN(limitNum) ? 10 : limitNum;
//     // Calculate page number for metadata
// const pageNum = Math.floor(validOffset / validLimit) + 1;

//     // 2. Validate sortBy and sortOrder
//     const allowedSortFields = ['created_at', 'company_name', 'company_id', 'mobile_no', 'email', 'branch_name'];
//     const validSortBy = allowedSortFields.includes(order_by) ? order_by : 'created_at';
//     const validSortOrder = order_dir && order_dir.toUpperCase() === 'ASC' ? 'ASC' : 'DESC';

//     // 3. Build base query (with join for branch_name)
//     let baseQuery = `SELECT c.*, b.branch_name FROM companies c LEFT JOIN branches b ON c.branch_id = b.id WHERE c.status = 1`;
//     let params = [];

//     // 4. Add search if provided
//     if (search && search.trim()) {
//       const likeSearch = `%${search.trim()}%`;
//       baseQuery += ` AND (c.company_name LIKE ? OR c.company_id LIKE ? OR c.mobile_no LIKE ? OR c.email LIKE ? OR b.branch_name LIKE ?)`;
//       params.push(likeSearch, likeSearch, likeSearch, likeSearch, likeSearch);
//     }

//     // 5. Get total count
//     const countQuery = baseQuery.replace(/^SELECT c\.\*, b\.branch_name FROM companies c LEFT JOIN branches b ON c\.branch_id = b\.id/, 'SELECT COUNT(*) as total FROM companies c LEFT JOIN branches b ON c.branch_id = b.id');
//     const countResult = await db.query(countQuery, params);
//     const total = countResult[0]?.total || 0;

//     // 6. Add ORDER BY, LIMIT, OFFSET
//     baseQuery += ` ORDER BY ${validSortBy === 'branch_name' ? 'b.branch_name' : 'c.' + validSortBy} ${validSortOrder} LIMIT ? OFFSET ?`;
//     params.push(limitNum, offset);

//     // 7. Execute final query
//     const companies = await db.query(baseQuery, params);

//     // 8. Pagination meta
//     const totalPages = Math.ceil(total / limitNum);
//     const paginationMeta = {
//       total,
//       page: pageNum,
//       limit: limitNum,
//       totalPages
//     };

//     return res.status(200).json({
//       status: 200,
//       success: true,
//       message: 'Companies list fetched successfully',
//       data: companies,
//       total,
//       pagination: paginationMeta
//     });
//   } catch (error) {
//     return res.status(500).json({
//       status: 500,
//       success: false,
//       message: 'Error fetching company list',
//       data: [],
//       total: 0,
//       pagination: {
//         total: 0,
//         page: 1,
//         limit: 10,
//         totalPages: 0
//       },
//       error: error.message
//     });
//   }
// };
export const getCompanyList = async (req, res) => {
  try {
    // 1. Get pagination and sorting params
    const {
      start = 0,
      length = 10,
      order_by,
      order_dir,
      search
    } = req.query;

    const offset = parseInt(start, 10);
    const limitNum = parseInt(length, 10);

    // Validate offset and limit
    const validOffset = isNaN(offset) ? 0 : offset;
    const validLimit = isNaN(limitNum) ? 10 : limitNum;

    // Calculate page number for pagination metadata
    const pageNum = Math.floor(validOffset / validLimit) + 1;

    // 2. Validate sortBy and sortOrder
    const allowedSortFields = ['created_at', 'company_name', 'company_id', 'mobile_no', 'email', 'branch_name'];
    const validSortBy = allowedSortFields.includes(order_by) ? order_by : 'created_at';
    const validSortOrder = order_dir && order_dir.toUpperCase() === 'ASC' ? 'ASC' : 'DESC';

    // 3. Build base query (with join for branch_name)
    let baseQuery = `SELECT c.*, b.branch_name FROM companies c LEFT JOIN branches b ON c.branch_id = b.id WHERE c.status = 1`;
    let params = [];

    // 4. Add search if provided
    if (search && search.trim()) {
      const likeSearch = `%${search.trim()}%`;
      baseQuery += ` AND (c.company_name LIKE ? OR c.company_id LIKE ? OR c.mobile_no LIKE ? OR c.email LIKE ? OR b.branch_name LIKE ?)`;
      params.push(likeSearch, likeSearch, likeSearch, likeSearch, likeSearch);
    }

    // 5. Get total count
    const countQuery = baseQuery.replace(
      /^SELECT c\.\*, b\.branch_name FROM companies c LEFT JOIN branches b ON c\.branch_id = b\.id/,
      'SELECT COUNT(*) as total FROM companies c LEFT JOIN branches b ON c.branch_id = b.id'
    );
    const countResult = await db.query(countQuery, params);
    const total = countResult[0]?.total || 0;

    // 6. Add ORDER BY, LIMIT, OFFSET
    baseQuery += ` ORDER BY ${validSortBy === 'branch_name' ? 'b.branch_name' : 'c.' + validSortBy} ${validSortOrder} LIMIT ? OFFSET ?`;
    params.push(validLimit, validOffset);

    // 7. Execute final query
    const companies = await db.query(baseQuery, params);

    // 8. Pagination meta
    const totalPages = Math.ceil(total / validLimit);
    const paginationMeta = {
      total,
      page: pageNum,
      limit: validLimit,
      totalPages
    };

    return res.status(200).json({
      status: 200,
      success: true,
      message: 'Companies list fetched successfully',
      data: companies,
      total,
      pagination: paginationMeta
    });
  } catch (error) {
    return res.status(500).json({
      status: 500,
      success: false,
      message: 'Error fetching company list',
      data: [],
      total: 0,
      pagination: {
        total: 0,
        page: 1,
        limit: 10,
        totalPages: 0
      },
      error: error.message
    });
  }
};